-----------------------------------------------------------------------------
-- A few base functions for analysing dependencies in FlatCurry programs:
--
-- Michael Hanus, January 2002
-----------------------------------------------------------------------------

module AnaDependency(funsInExpr, indirectlyDependent) where

import Flat
import List


-- Computes the list of indirect dependencies for all functions.
-- Argument: a list of function declarations
-- Result: a list of pairs of functions names and the corresponding
--         called functions
indirectlyDependent :: [FuncDecl] -> [(String,[String])]
indirectlyDependent funs = depsClosure (directlyDependent funs)

-- list of direct dependencies for all functions
directlyDependent :: [FuncDecl] -> [(String,[String])]
directlyDependent [] = []
directlyDependent (Func f _ _ (Rule _ e) : funs) =
  (f,funsInExpr e) : directlyDependent funs
directlyDependent (Func f _ _ (External _) : funs) =
  (f,[]) : directlyDependent funs

-- compute all transitive dependencies between functions:
depsClosure :: [(String,[String])] -> [(String,[String])]
depsClosure directdeps = map (\ (f,ds)->(f,closure ds ds)) directdeps
 where
  closure olddeps [] = olddeps
  closure olddeps (f:fs) =
     let newdeps = filter (`notElem` olddeps) (getDeps directdeps f)
      in closure (newdeps++olddeps) (newdeps++fs)

  getDeps [] _ = []
  getDeps ((f,ds):fdeps) f1 = if f==f1 then ds
                                       else getDeps fdeps f1


-- Gets a list of all functions (and partially applied constructors)
-- called in an expression:
funsInExpr :: Expr -> [String]
funsInExpr (Var _) = []
funsInExpr (Lit _) = []
funsInExpr (Comb ct f es) =
  if ct==ConsCall then nub (concatMap funsInExpr es)
                  else nub (f : concatMap funsInExpr es)
funsInExpr (Apply e1 e2) = union (funsInExpr e1) (funsInExpr e2)
funsInExpr (Constr _ e) = funsInExpr e
funsInExpr (Or e1 e2) = union (funsInExpr e1) (funsInExpr e2)
funsInExpr (Case _ e bs) = union (funsInExpr e)
                                 (nub (concatMap funsInBranch bs))
                     where funsInBranch (Branch _ be) = funsInExpr be
funsInExpr (GuardedExpr _ e1 e2) = union (funsInExpr e1) (funsInExpr e2)
funsInExpr (Choice e) = funsInExpr e


-- end of AnaDependency

