------------------------------------------------------------------------------
-- Nondeterminism analysis:
-- check whether functions are set-valued/nondeterministic, i.e., might
-- have several results even for ground argument terms.
--
-- Michael Hanus, December 2001
------------------------------------------------------------------------------

module AnaNondeterminism(analyseSetValued) where

import Flat
import List
import AnaOverlapping
import AnaDependency

------------------------------------------------------------------------------
-- The non-determinism analysis must be applied to complete programs,
-- i.e., modules together with all their imported modules.
-- It assigns to a FlatCurry program the list of all function names
-- together with a flag which is True if this function might be defined
-- in a non-deterministic manner (i.e., might be a set-valued function).

analyseSetValued :: Prog -> [(String,Bool)]
analyseSetValued (Prog _ _ _ funs _ _) = map anaFun alldeps
  where
    anaFun (name,depfuns) = (name, any (isNondetDefined funs) (name:depfuns))

    alldeps = indirectlyDependent funs

-- (isNondetDefined fundecls f):
-- Is a function f defined by a nondeterministic rule?
isNondetDefined :: [FuncDecl] -> String -> Bool
isNondetDefined [] _ = False -- this case should occur only for constructors
isNondetDefined (Func f _ _ def : funs) f1 =
 if f==f1 then f/="failed" && isNondetRule def
          else isNondetDefined funs f1

isNondetRule (Rule _ e) = orInExpr e || extraVarInExpr e
isNondetRule (External _) = False


-- check an expression for occurrences of extra variables:
extraVarInExpr :: Expr -> Bool
extraVarInExpr (Var _) = False
extraVarInExpr (Lit _) = False
extraVarInExpr (Comb _ _ es) = foldr (||) False (map extraVarInExpr es)
extraVarInExpr (Apply e1 e2) = extraVarInExpr e1 || extraVarInExpr e2
extraVarInExpr (Constr vars e) = (not (null vars)) || extraVarInExpr e
extraVarInExpr (Or e1 e2) = extraVarInExpr e1 || extraVarInExpr e2
extraVarInExpr (Case _  e bs) = extraVarInExpr e || any extraVarInBranch bs
                where extraVarInBranch (Branch _ be) = extraVarInExpr be
extraVarInExpr (GuardedExpr vars e1 e2) =
            (not (null vars)) || extraVarInExpr e1 || extraVarInExpr e2
extraVarInExpr (Choice _) =  False -- committed choice encapsulates non-det.


