------------------------------------------------------------------------------
-- Analysis for operational completeness:
-- check whether functions are operational complete, i.e., calls only
-- non-rigid functions
--
-- Michael Hanus, January 2001
------------------------------------------------------------------------------

module AnaOpComplete(analyseOpCompleteness)   where

import Flat
import List
import AnaDependency

------------------------------------------------------------------------------
-- The operational completeness analysis must be applied to complete programs,
-- i.e., modules together with all their imported modules.
-- It assigns to a FlatCurry program the list of all function names
-- together with a flag which is True if this function is operationally
-- complete, i.e., does not call a rigid function.

analyseOpCompleteness :: Prog -> [(String,Bool)]
analyseOpCompleteness (Prog _ _ _ funs _ _) = map anaFun alldeps
  where
    anaFun (name,depfuns) = (name, all (isFlexDefined funs) (name:depfuns))

    alldeps = indirectlyDependent funs


-- (isFlexDefined fundecls f):
-- Is a function f defined by a flexible rule?
isFlexDefined :: [FuncDecl] -> String -> Bool
isFlexDefined [] f =  -- this case should occur only for constructors
                      -- and some special predefined functions
   f `notElem` ["findall","findfirst"]
isFlexDefined (Func f _ _ def : funs) f1 =
 if f==f1 then isFlexRule def
          else isFlexDefined funs f1

isFlexRule (Rule _ e) = isFlexExpr e
isFlexRule (External f) = f `elem` ["=:=","success","&","&>","return"]

-- Checks whether an expression is flexible, i.e., can only suspend
-- because of calls to other possibly rigid functions.

isFlexExpr :: Expr -> Bool
isFlexExpr (Var _)           = True
isFlexExpr (Lit _)           = True
isFlexExpr (Comb _ _ args)   = all isFlexExpr args
isFlexExpr (Apply _ _)       = False -- apply suspends if arg 1 is unbound
isFlexExpr (Constr _ e)      = isFlexExpr e
isFlexExpr (Or e1 e2)        = isFlexExpr e1 && isFlexExpr e2
isFlexExpr (Case ctype e bs) = ctype==Flex &&
                            all isFlexExpr (e : map (\(Branch _ be)->be) bs)
isFlexExpr (Choice _) = False
isFlexExpr (GuardedExpr _ e1 e2) = isFlexExpr e1 && isFlexExpr e2


-- end of AnaOpComplete
