-----------------------------------------------------------------------------
-- Configuration file containing the main parameters of the
-- current CIDER instance.
--
-- To modify or extend the functionality of CIDER (e.g., new analyses,
-- other source languages), this file must be adapted and then
-- CIDER must be recompiled.
--
-----------------------------------------------------------------------------

import Types
import AnaOverlapping
import AnaIndeterminism
import AnaNondeterminism
import AnaCompleteness
import AnaOpComplete
import AnaDependency
import DaVinci  -- for datatype DvGraph
import DependencyAnalysis
import AnaPatternMatch
import AnaTypes
import CalledByAnalysis
import Prolog2Curry
import Flat

infix 1 `intoCider`

------------------------------------------------------------------------------
-- First we define the list of all available program analyses:

anaList :: [(String, ProgAnalysis AnaRes)]
anaList =
  [("Get Type",          analyseTypes          `intoCider` showStringResult),
   ("Overlapping Rules", analyseOverlappings   `intoCider` showOverlapping),
   ("Pattern Complete",  analyseCompleteness   `intoCider` showComplete),
   ("Pattern Matching",  showPatternMatching   `intoCider` showStringResult),
   ("Solution Complete", analyseOpCompleteness `intoCider` showOpComplete),
   ("Set-Valued",        analyseSetValued      `intoCider` showSetValued),
   ("Indeterminism",     analyseIndeterminism  `intoCider` showIndeterminism),
   ("Depends on",        analyseIDependencies  `intoCider` showStringResult),
   ("Dependency Graph",  buildDGraphs          `intoCider` showGraphResult),
   ("Called By",         analyseCalledBys      `intoCider` showStringResult),
   ("Dead Code",         deadCodeAnalysis      `intoCider` showStringResult)]

-- This function is useful to integrate an existing program analysis
-- into CIDER by providing a transformation of the analysis results
-- into the analysis results support by CIDER:
intoCider :: ProgAnalysis a -> (a->AnaRes) -> ProgAnalysis AnaRes
intoCider analysis res2cider prog =
  map (\(name,res)->(name, res2cider res)) (analysis prog)

-- This is a standard transformation for analysis results of type String:
showStringResult :: String -> AnaRes
showStringResult s = Message s "blue"

-- This is a standard transformation for analysis results of type DvGraph:
showGraphResult :: DvGraph -> AnaRes
showGraphResult g = Graph g "blue"

-- Shows the result of the overlapping analysis.
showOverlapping :: Bool -> AnaRes
showOverlapping True  = Message "Overlapping"     "red"
showOverlapping False = Message "Not Overlapping" "blue"

-- Shows the result of the completeness analysis.
showComplete :: CompletenessType -> AnaRes
showComplete Complete     = Message "completely defined"   "blue"
showComplete InComplete   = Message "incompletely defined" "red"
showComplete InCompleteOr =
  Message "incompletely defined in each disjunction (but might be complete)"
          "red"

-- Shows the result of the operational completeness analysis.
showOpComplete :: Bool -> AnaRes
showOpComplete True  = Message "All solutions can be computed" "red"
showOpComplete False = Message "Evaluation might suspend"      "blue"

-- Shows the result of the indeterminism analysis.
showIndeterminism :: Bool -> AnaRes
showIndeterminism True  = Message "Indeterministic"     "red"
showIndeterminism False = Message "Not indeterministic" "blue"

-- Shows the result of the non-determinism analysis.
showSetValued :: Bool -> AnaRes
showSetValued True  =
                Message "Function might be nondeterministic (set-valued)" "red"
showSetValued False = Message "Deterministic function" "blue"


------------------------------------------------------------------------------
-- Parameters of CIDER for reading input files:

-- the filter used during opening of new files
-- (consisting of a description and the standard suffix):
filterOpenFiles = [("Curry Files",".curry"),
                   ("Literate Curry files",".lcurry"),
                   ("Prolog files",".pro")]
--filterOpenFiles = [("Curry Files",".curry"),
--                   ("Literate Curry files",".lcurry")]


-- the preprocessing action which is executed whenever the user loads
-- a module and the main module is compiled
-- (e.g., translate a Prolog source into (Flat)Curry):
preprocess_module :: String -> IO ()
preprocess_module mod = prolog2curry mod
-- preprocess_module _ = done  -- no preprocessing necessary


-- the action to actually read a FlatCurry module:
readFlatCurryModule :: String -> IO Prog
readFlatCurryModule mod = readPrologFlatCurry mod
--readFlatCurryModule mod = readFlatCurry mod


-- the symbolic port name of the I/O server in CIDER:
-- (modify only if somebody else on the same machine uses this port name)
ioServerPort = "cider_ios"
