-----------------------------------------------------------------------------
-- dependency analysis, building of dependency graphs
--
-- Johannes Koj, October 2000
-- Michael Hanus, July 2001
-----------------------------------------------------------------------------

module DependencyAnalysis(analyseDDependencies, analyseIDependencies,
                          buildDGraphs, deadCodeAnalysis) where

import Base
import Flat
import DaVinci
import AnaDependency
import List

-- compute all direct dependencies:
analyseDDependencies :: Prog -> [(String,String)]
analyseDDependencies (Prog modname _ _ funs _ trans) =
  map (analyseDDependency (stripModName (cutPath modname) trans)) funs
 where
   analyseDDependency :: (String->String) -> FuncDecl -> (String,String)
   analyseDDependency rename (Func name _ _ (Rule _ e)) =
          (name, ppFunList rename (nub (funsInExpr e)))
   analyseDDependency _ (Func name _ _ (External _)) =
          (name, "<external function>")


-- compute all direct dependencies:
analyseIDependencies :: Prog -> [(String,String)]
analyseIDependencies (Prog modname _ _ funs _ trans) =
  map (formatDeps (stripModName (cutPath modname) trans))
      (indirectlyDependent funs)
 where
   formatDeps rename (name,deps) = (name, ppFunList rename deps)


deadCodeAnalysis :: Prog -> [(String,String)]
deadCodeAnalysis (Prog modname _ _ funs _ trans) =
  map (deadCode rename  (indirectlyDependent funs) (definedFuns rename funs))
      funs
 where
   rename = stripModName (cutPath modname) trans

-- compute all functions not reachable from the current function:
deadCode :: (String->String) -> [(String,[String])] -> [String]
                             -> FuncDecl -> (String,String)
deadCode rename fundeps deffuns (Func name _ _ _) =
    (name, "Top-level functions defined in this module but not used by \""++
           rename name++"\":\n"++
           ppFunList rename (filter (`notElem` (name:allDepFuns)) deffuns))
  where
    allDepFuns = snd (head (filter (\(f,_)->f==name) fundeps))

definedFuns _ [] = []
definedFuns rename (Func _ _ _ (External _) : fs) = definedFuns rename fs
definedFuns rename (Func f _ _ (Rule _ _)   : fs) =
       if rename f /= f -- top-level function?
       then f : definedFuns rename fs
       else definedFuns rename fs


-- build the call graphs for functions:
buildDGraphs :: Prog -> [(String,DvGraph)]
buildDGraphs (Prog modname _ _ funs _ trans) =
  map (buildDGraph (stripModName (cutPath modname) trans) funs) funs


buildDGraph :: (String->String) -> [FuncDecl] -> FuncDecl -> (String,DvGraph)
buildDGraph rename funs (Func name _ _ _) =
   (name, dvNewGraph (deps2DvNodes fixres fixres))
 where
   fixres = fix [name] []

   fix []     res = res
   fix (f:fs) res = let newF = not (f `elem` (map (\(g,_,_)->g) res))
                        deps = funsInExpr (lookupRHS f funs)
                        nres = if newF then [(f,deps,nodeid)] else []
                        nfs  = if newF then deps else []
                    in fix (tail (nub ((f:fs)++nfs))) (res++nres)
      where nodeid free

   newSimpleEdge nid = dvSimpleEdge edgeid nid dvEmptyH  where edgeid free

   deps2DvNodes [] _ = []
   deps2DvNodes ((f,deps,id):ds) allnodes =
       let ids    = map (lookupId allnodes) (nub deps)
           nedges = map (\id->newSimpleEdge id) ids
       in (dvNodeWithEdges id (rename f) nedges dvEmptyH)
           : (deps2DvNodes ds allnodes)

   lookupId ((f2,_,id):fs) f | f==f2 = id
                             | otherwise = lookupId fs f

-- get right-hand side for a defined function:
lookupRHS _ []  = Var 1
lookupRHS f1 ((Func _ _ _ (External _)):fs) = lookupRHS f1 fs
lookupRHS f1 ((Func f2 _ _ (Rule _ e)):fs) | f1==f2    = e
                                           | otherwise = lookupRHS f1 fs


-- end of DependencyAnalysis
