----------------------------------------------------------------------------
-- Evaluator for FlatCurry
--
-- * use evalExpr for stepwise evaluation,
--   always keep the evaluator state est !
--
-- * start evaluation with initial state (initialEst funs e)
--
-- * read a module and its imports with readFlatCurryMod
--
-- * print an expression with (ppExpr est 0 e)
--   (the state is needed because of renaming!)
--
--
-- Johannes Koj, October 2000
----------------------------------------------------------------------------

import Flat
import List

----------------------------------------------------------------------------
-- the evaluator state contains: 
-- * the function decls
-- * a renaming for functions
-- * an index for variables and functions

data StackTree = SN [Expr] StackTree StackTree | SE

data EvalState = EState [FuncDecl] StackTree Int Int

getFuncDecls (EState fds _ _ _) = fds

getVarIndex (EState _ _ i _) = i
setVarIndex (EState fds st _ j) i = EState fds st i j

pushExpr (EState fds (SN es l r) i j) e = EState fds (SN (e:es) l r) i j
popExpr (EState fds (SN [] l r) i j) = EState fds (SN [] l r) i j
popExpr (EState fds (SN (e:es) l r) i j) = EState fds (SN es l r) i j
topExpr (EState _ (SN (e:_) _ _) _ _) = e
hasExprs (EState _ (SN [] _ _)_ _) = False
hasExprs (EState _ (SN (e:_) _ _) _ _) = True
setLeft (EState fds (SN s l r) _ _) (EState _ t i j) = EState fds (SN s t r) i j
setRight (EState fds (SN s l r) _ _) (EState _ t i j) = EState fds (SN s l t) i j
takeLeft (EState fds (SN _ SE _) i j) = EState fds (SN [] SE SE) i j
takeLeft (EState fds (SN _ (SN s l r) _) i j) = EState fds (SN s l r) i j
takeRight (EState fds (SN _ _ SE) i j) = EState fds (SN [] SE SE) i j
takeRight (EState fds (SN _ _ (SN s l r)) i j) = EState fds (SN s l r) i j

renFun (EState fds st i j) f = (EState fds st i (j+1),(show j)++"."++f)

getOrigName f | isPredefined f = f 
              | otherwise      = localname
  where
    (_,localname) = splitFlatModName f

    splitFlatModName name = let (modname,rname) = break (=='.') name in
       if rname=="" then ("prelude",name)
                    else (modname,tail rname)

initialEst funs e = EState funs (SN [] SE SE) ((maxVarIndex e)) 0

----------------------------------------------------------------------------

predefined = ["+","-","*","div","mod",">",">=","<","<=",
              "==","=:=","&","success"]
isPredefined c = c `elem` predefined

-- a simple "debugger"
--main_debug prog e =
--  do ((Prog _ _ _ funs _ _),_) <- readFlatCurryMod prog
--     evaluate (initialEst funs e) e

evaluate est e = let (res,red)=getRedex est e in putStrLn (ppExpr 0 e) >> getChar >> aux res red
  where aux _ False = putStrLn "there are no more solutions"
        aux ((redex,((est,e2):_)):_) True = evaluate est (set redex e e2)


----------------------------------------------------------------------------
-- this is the central function:
-- look for the redex and the corrsponding reduced expressions
-- change the state est to a new state

getRedex :: EvalState -> Expr -> ([(Expr,[(EvalState,Expr)])],Bool)
getRedex est (Var x) = ([(Var x,[(est,Var x)])],False)

getRedex est (Lit l) = ([(Lit l,[(est,Lit l)])],False)

getRedex est (GuardedExpr v (Comb FuncCall c es) e) | c=="success" = ([(GuardedExpr v (Comb FuncCall c es) e,[(est,e)])],True)
                                                    | otherwise = getRedex est (Comb FuncCall c es)

getRedex est (GuardedExpr vs (Constr vs2 cons) e) = getRedex est (Constr vs2 cons)

getRedex est (Constr vs e) = ([(Constr vs e,[(est,e)])],True)

getRedex est (Comb ConsCall c es) = get es
  where get [] = ([(Comb ConsCall c es,[(est,Comb ConsCall c es)])],False)
        get (e:es2) = let (res,red)=getRedex est e in if red then (res,True) else get es2

getRedex est (Apply (Apply e1 e2) e3) =
  getRedex est (Apply e1 e2)

getRedex est (Apply (Comb PartCall c es) e) | (getArity (getFuncDecls est) (getOrigName c))==(length es)+1 = ([(Apply (Comb PartCall c es) e,[(est,Comb FuncCall c (es++[e]))])],True)
                                            | otherwise = ([(Apply (Comb PartCall c es) e,[(est,Comb PartCall c (es++[e]))])],True)
   where getArity ((Func f1 ar _ _):fds) f2 | f1==f2 = ar
                                            | otherwise = getArity fds f2                        

getRedex est (Comb FuncCall c es) | not (isPredefined c) = if hasExprs est then get (popExpr est) (topExpr est) else let (est2,e2)=apply est c es in get est2 e2
  where get est3 (Var x) = ([(Comb FuncCall c es,[(est3,Var x)])],True)
        get est3 (Lit l) = ([(Comb FuncCall c es,[(est3,Lit l)])],True)
        get est3 (Comb ct c2 es2) = ([(Comb FuncCall c es,[(est3,Comb ct c2 es2)])],True)
        get est3 (Or expr1 expr2) = let (res1,red1)=get est3 expr1 
                                        (res2,red2)=get est3 expr2
                                    in ((if red1 then res1 else [])++(if red2 then res2 else []),red1 || red2)
        get est3 (Case ct e ces) = match est3 ct e (Case ct e ces) ces
        get est3 (Constr vs e) = ([(Comb FuncCall c es,[(est3,Constr vs e)])],True)
        get est3 (GuardedExpr v1 co e3) = ([(Comb FuncCall c es,[(est3,GuardedExpr v1 co e3)])],True)
        match est3 ct (Comb ConsCall _ _) _ [] = ([(Comb FuncCall c es,[])],False)
        match est3 ct (Comb ConsCall c1 es2) ce ((Branch (Pattern c2 vs) e):ps) | (c1==c2) = get est3 (bindFParams vs es2 e)
                                                                                | otherwise = match est3 ct (Comb ConsCall c1 es2) ce ps
        match est3 ct (Lit l) _ [] = ([(Comb FuncCall c es,[])],False)
        match est3 ct (Lit l1) ce ((Branch (LPattern l2) e):ps) | l1==l2 = get est3 e
                                                                | otherwise = match est3 ct (Lit l1) ce ps
        match est3 _ (Comb FuncCall c2 es2) ce _ = let (res,red)=getRedex est3 (Comb FuncCall c2 es2)
                                                   in if red then (map (\(e1,rexprs)->(e1,map (\(est,e2)->(pushExpr est (set e1 ce e2),e2)) rexprs)) res,red) 
                                                        else ([(Comb FuncCall c es,[])],False)
        match est3 _ (GuardedExpr vs e1 e2) ce _ = let (res,red)=getRedex est3 (GuardedExpr vs e1 e2)
                                                   in if red then (map (\(e1,rexprs)->(e1,map (\(est,e2)->(pushExpr est (set e1 ce e2),e2)) rexprs)) res,red)
                                                       else ([(Comb FuncCall c es,[])],False)
        match est3 _ (Apply e1 e2) ce _ = let (res,red)=getRedex est3 (Apply e1 e2)
                                          in if red then (map (\(e1,rexprs)->(e1,map (\(est,e2)->(pushExpr est (set e1 ce e2),e2)) rexprs)) res,red)
                                                       else ([(Comb FuncCall c es,[])],False)
        match est3 Rigid (Var _) ce _ = ([(Comb FuncCall c es,[])],False)
        match est3 Flex (Var x) _ ps = bind est3 ps []
          where bind est4 [] res = ([(Var x,map (\e->(est4,e)) res)],True)
                bind est4 ((Branch (Pattern i vars) _):ps) res = let (fv,est5)=getfreshvars (length vars) est4
                                                                 in bind est5 ps (res++[Comb ConsCall i fv])
                bind est4 ((Branch (LPattern (Intc i)) _):ps) res = bind est4 ps (res++[Lit (Intc i)]) 


getRedex est (Comb FuncCall c es) | (isPredefined c) = get c es
  where get "+" (e1:e2:[]) | (isGround e1) && (isGround e2) = ([(Comb FuncCall c es,[(est,add e1 e2)])],True)
                           | isGround e1 = getRedex est e2
                           | otherwise = getRedex est e1
          where add (Lit (Intc i1)) (Lit (Intc i2)) = Lit (Intc (i1+i2))
                add (Var _) (Var _) = Comb FuncCall c es
        get "-" (e1:e2:[]) | (isGround e1) && (isGround e2) = ([(Comb FuncCall c es,[(est,sub e1 e2)])],True)
                           | isGround e1 = getRedex est e2
                           | otherwise = getRedex est e1
          where sub (Lit (Intc i1)) (Lit (Intc i2)) = Lit (Intc (i1-i2))
                sub (Var _) (Var _) = Comb FuncCall c es
        get "*" (e1:e2:[]) | (isGround e1) && (isGround e2) = ([(Comb FuncCall c es,[(est,mul e1 e2)])],True)
                           | isGround e1 = getRedex est e2
                           | otherwise = getRedex est e1
          where mul (Lit (Intc i1)) (Lit (Intc i2)) = Lit (Intc (i1*i2))
                mul (Var _) (Var _) = Comb FuncCall c es
        get "div" (e1:e2:[]) | (isGround e1) && (isGround e2) = ([(Comb FuncCall c es,[(est,divide e1 e2)])],True)
                             | isGround e1 = getRedex est e2
                             | otherwise = getRedex est e1
          where divide (Lit (Intc i1)) (Lit (Intc i2)) = Lit (Intc (i1 `div` i2))
                divide (Var _) (Var _) = Comb FuncCall c es
        get ">" (e1:e2:[]) | (isGround e1) && (isGround e2) = ([(Comb FuncCall c es,[(est,gt e1 e2)])],True)
                           | isGround e1 = getRedex est e2
                           | otherwise = getRedex est e1
          where gt (Lit (Intc i1)) (Lit (Intc i2)) | i1>i2 = Comb ConsCall "True" []
                                                   | otherwise = Comb ConsCall "False" []
                gt (Var _) (Var _) = Comb FuncCall c es
        get ">=" (e1:e2:[]) | (isGround e1) && (isGround e2) = ([(Comb FuncCall c es,[(est,geq e1 e2)])],True)
                            | isGround e1 = getRedex est e2
                            | otherwise = getRedex est e1
          where geq (Lit (Intc i1)) (Lit (Intc i2)) | i1>=i2 = Comb ConsCall "True" []
                                                    | otherwise = Comb ConsCall "False" []
                gt (Var _) (Var _) = Comb FuncCall c es
        get "<" (e1:e2:[]) | (isGround e1) && (isGround e2) = ([(Comb FuncCall c es,[(est,lt e1 e2)])],True)
                           | isGround e1 = getRedex est e2
                           | otherwise = getRedex est e1
          where lt (Lit (Intc i1)) (Lit (Intc i2)) | i1<i2 = Comb ConsCall "True" []
                                                   | otherwise = Comb ConsCall "False" []
        get "<=" (e1:e2:[]) | (isGround e1) && (isGround e2) = ([(Comb FuncCall c es,[(est,leq e1 e2)])],True)
                            | isGround e1 = getRedex est e2
                            | otherwise = getRedex est e1
          where leq (Lit (Intc i1)) (Lit (Intc i2)) | i1<=i2 = Comb ConsCall "True" []
                                                    | otherwise = Comb ConsCall "False" []
        get "==" (e1:e2:[]) | (isGround e1) && (isGround e2) = ([(Comb FuncCall c es,[(est,eq e1 e2)])],True)
                            | isGround e1 = getRedex est e2
                            | otherwise = getRedex est e1
          where eq e3 e4 | e3==e4 = Comb ConsCall "True" []
                         | otherwise = Comb ConsCall "False" []
        get "=:=" ((Var x):(Lit l):[]) =
          ([(Var x,[(est,Lit l)])],True)
        get "=:=" ((Var x):(Comb ConsCall c2 es2):[]) 
          | x `elem` (cv (Comb ConsCall c es)) =
            ([(Comb FuncCall c es,[])],False)
          | otherwise = 
            let (fv,est2)=getfreshvars (length es2) est
            in ([(Var x,[(est2,Comb ConsCall c2 fv)])],True)
          where cv _ = []
        get "=:=" ((Var x):(Var y):[])
          | x==y =
            ([(Comb FuncCall c es,[(est,Comb FuncCall "success" [])])],True)
          | otherwise = 
            ([(Var x,[(est,Var y)])],True)
        get "=:=" ((Var x):(Comb FuncCall c es):[]) =
          getRedex est (Comb FuncCall c es)
        get "=:=" ((Var x):(Apply e1 e2):[]) =
          getRedex est (Apply e1 e2)
        get "=:=" ((Var x):(Constr vs e):[]) =
          getRedex est (Constr vs e)
        get "=:=" ((Var x):(GuardedExpr vs c e):[]) =
          getRedex est (GuardedExpr vs c e)
        get "=:=" ((Lit l1):(Lit l2):[]) 
          | l1==l2 = 
            ([(Comb FuncCall c es,[(est,Comb FuncCall "success" [])])],True)
          | otherwise = 
            ([(Comb FuncCall c es,[])],False)
        get "=:=" ((Lit l):(Var x):[]) =
          get "=:=" ((Var x):(Lit l):[])
        get "=:=" ((Lit _):(Comb FuncCall c2 es2):[]) =
          getRedex est (Comb FuncCall c2 es2)
        get "=:=" ((Lit _):(Comb ConsCall c2 es2):[]) =
          ([(Comb FuncCall c es,[])],False)
        get "=:=" ((Lit _):(Apply e1 e2):[]) =
          getRedex est (Apply e1 e2)
        get "=:=" ((Lit _):(Constr vs e):[]) =
          getRedex est (Constr vs e)
        get "=:=" ((Lit _):(GuardedExpr vs c e):[]) =
          getRedex est (GuardedExpr vs c e)
        get "=:=" ((Comb ConsCall c es):(Var x):[]) =
          get "=:=" ((Var x):(Comb ConsCall c es):[])
        get "=:=" ((Comb ConsCall c1 es1):(Comb ConsCall c2 es2):[]) 
          | c1==c2 = getArgs es1 es2
          | otherwise = ([(Comb FuncCall c es,[])],False)
          where getArgs [] _ = 
                  ([(Comb FuncCall c es,[(est,Comb FuncCall "success" [])])],True)
                getArgs (e1:e1s) (e2:e2s) 
                  | e1==e2 = getArgs e1s e2s
                  | otherwise = let (res,red)=get "=:=" [e1,e2]
                                in if red then (res,True) else ([(Comb FuncCall c es,[])],False)
        get "=:=" ((Comb ConsCall _ _):(Lit _):[]) =
          ([(Comb FuncCall c es,[])],False)
        get "=:=" ((Comb ConsCall _ _):(Comb FuncCall c es):[]) = 
          getRedex est (Comb FuncCall c es)
        get "=:=" ((Comb ConsCall _ _):(Apply e1 e2):[]) = 
          getRedex est (Apply e1 e2)
        get "=:=" ((Comb ConsCall _ _):(Constr vs e):[]) = 
          getRedex est (Constr vs e)
        get "=:=" ((Comb ConsCall _ _):(GuardedExpr vs c e):[]) = 
          getRedex est (GuardedExpr vs c e)
        get "=:=" ((Constr vs e):_:[]) =
          getRedex est (Constr vs e)
        get "=:=" ((GuardedExpr vs c e):_:[]) = 
          getRedex est (GuardedExpr vs c e)
        get "=:=" ((Comb FuncCall c2 es2):_:[]) =
          getRedex est (Comb FuncCall c2 es2)
        get "=:="((Apply e1 e2):_:[]) =
          getRedex est (Apply e1 e2)
        get "&" (e1:e2:[]) | e1==(Comb FuncCall "success" []) = ([(Comb FuncCall c es,[(est,e2)])],True)
                           | e2==(Comb FuncCall "success" [])= ([(Comb FuncCall c es,[(est,e1)])],True)
                           | otherwise = let (res,red)=getRedex (takeLeft est) e1 
                                         in if red then (map (\(e1,rexprs)->(e1,map (\(est2,e2)->(setLeft est est2,e2)) rexprs)) res,True) 
                                             else let (res,red)=getRedex (takeRight est) e2
                                                  in if red then (map (\(e1,rexprs)->(e1,map (\(est2,e2)->(setRight est est2,e2)) rexprs)) res,True)
                                                               else ([(Comb FuncCall c es,[(est,Comb FuncCall c es)])],False)

        get "success" [] = ([(Comb FuncCall c es,[])],False)

----------------------------------------------------------------------------
-- find rhs of function call and make the corresponding substitutions

apply est f es = getMatchedRHS est (getFuncDecls est) (getOrigName f) es

getMatchedRHS est (Func fname _ _ funrule : fds) name es =
   if fname==name then getMatchedRHS_aux est funrule es
                  else getMatchedRHS est fds name es

getMatchedRHS_aux est (Rule vs rhs) es = let (rhs2,est2)=replace est vs es True rhs
                                         in (est2,rhs2)

----------------------------------------------------------------------------
-- replace the variables with the expressions
-- boolean argument indicates that funs are renamed!

replace :: EvalState -> [Int] -> [Expr] -> Bool -> Expr -> (Expr,EvalState)
replace est vs es _ (Var v) = (replaceVar vs es v,est)
  where replaceVar [] [] var = Var var
        replaceVar (v:vs) (e:es) var = if v==var then e
                                                 else replaceVar vs es var
replace est _ _ _ (Lit l) = (Lit l,est)
replace est vs es1 ren (Comb ct c es2) | (ct/=ConsCall) && ren && (not (isPredefined c)) = let (est3,c2)=renFun est2 c 
                                                                                           in (Comb ct c2 es3,est3)
                                       | otherwise = (Comb ct c es3,est2)
  where (es3,est2)=replacelist est vs es1 ren es2
replace est vs es ren (Apply e1 e2) = let (e3,est3)=replace est vs es ren e1
                                          (e4,est4)=replace est3 vs es ren e2
                                      in (Apply e3 e4,est4)
replace est vs1 es ren (Constr vs2 e) = let vs3=if ren then (vs1++vs2) else vs1
                                            (fv,est2)=getfreshvars (length vs2) est
                                            es3=if ren then (es++fv) else es
                                            est3=if ren then est2 else est
                                            (e4,est4)=replace est3 vs3 es3 ren e
                                        in (Constr (map (\x->let (Var y,_)=replace est4 vs3 es3 ren (Var x) in y) vs2) e4,est4)
replace est vs es ren (Or e1 e2) = let (e3,est3)=replace est vs es ren e1
                                       (e4,est4)=replace est3 vs es ren e2
                                   in (Or e3 e4,est4)
replace est vs es ren (Case ct e ps) = let (e2,est2)=replace est vs es ren e
                                           (ps2,est3)=replacecases est2 vs es ren ps
                                       in (Case ct e2 ps2,est3)
replace est vs1 es ren (GuardedExpr vs2 e1 e2) = let vs3=if ren then (vs1++vs2) else vs1
                                                     (fv,est2)=getfreshvars (length vs2) est
                                                     es3=if ren then (es++fv) else es
                                                     est3=if ren then est2 else est
                                                     (e3,est4)=replace est3 vs3 es3 ren e1
                                                     (e4,est5)=replace est4 vs3 es3 ren e2
                                                 in (GuardedExpr (map (\x->let (Var y,_)=replace est5 vs3 es3 ren (Var x) in y) vs2)
                                                                         e3 e4,est4)
replace est vs es ren (Choice e) = replace est vs es ren e

replacecases est vs es ren ps = aux est ps []
  where aux est [] res = (res,est)
        aux est (p:ps) res = let (p2,est2)=replacecase est vs es ren p in aux est2 ps (res++[p2])

replacecase est vs1 es ren (Branch (Pattern l vs2) e) =
    let vs3 = if ren then (vs1++vs2) else vs1
        (fv,est2) = getfreshvars (length vs2) est
        es3 = if ren then (es++fv) else es
        est3 = if ren then est2 else est
        (e4,est4) = replace est3 vs3 es3 ren e
     in (Branch (Pattern l (map (\x->let (Var y,_)=replace est4 vs3 es3 ren (Var x) in y) vs2)) e4,est4)
replacecase est vs1 es ren (Branch (LPattern l) e) =
    let (e4,est4) = replace est vs1 es ren e
     in (Branch (LPattern l) e4,est4)

replacelist est vs es1 ren es2 = aux est es2 []
  where aux est [] res = (res,est)
        aux est (e:es) res = let (e2,est2)=replace est vs es1 ren e in aux est2 es (res++[e2])
                                         


----------------------------------------------------------------------------
-- handling variables in expressions:

-- get the maximum index of all variables in an expression:
-- (or -1 if there is no variable)
maxVarIndex :: Expr -> Int
maxVarIndex (Var v)         = v
maxVarIndex (Lit _)         = -1
maxVarIndex (Comb _ _ exps) = maxList (map maxVarIndex exps)
maxVarIndex (Apply e1 e2)   = max (maxVarIndex e1) (maxVarIndex e2)
maxVarIndex (Constr vars e) = max (maxList vars) (maxVarIndex e)
maxVarIndex (Or e1 e2)      = max (maxVarIndex e1) (maxVarIndex e2)
maxVarIndex (Case _ e cs)   = max (maxVarIndex e) (maxList (map maxCase cs))
  where maxCase (Branch (Pattern _ xs) e) = max (maxList xs) (maxVarIndex e)
        maxCase (Branch (LPattern _) e) = maxVarIndex e
maxVarIndex (GuardedExpr vars c e) =
                   max (max (maxList vars) (maxVarIndex c)) (maxVarIndex e)
maxVarIndex (Choice e)      = maxVarIndex e

-- return a list of fresh variables
getfreshvars count est = aux count ((getVarIndex est)+1) []
  where aux c i vs | c==0 = (vs,setVarIndex est (i+1))
                   | otherwise = aux (c-1) (i+1) (vs++[Var i])

-- maximum of two numbers:
max i j = if i<=j then j else i

-- maximum of a list of naturals (-1 if list is empty):
maxList xs = foldr max (-1) xs

----------------------------------------------------------------------------
-- some useful funs...
-- set redex e1 to reduced redex e2 in expression

set :: Expr -> Expr -> Expr -> Expr
set e1 (Var x) e2 | (Var x)==e1 = e2
                  | otherwise = Var x
set e1 (Lit l) e2 | (Lit l)==e1 = e2
                  | otherwise = Lit l
set e1 (Comb ct c es) e2 | (Comb ct c es)==e1 = e2
                         | otherwise = (Comb ct c (map (\e->set e1 e e2) es)) 
set e1 (Apply e3 e4) e2 | (Apply e3 e4)==e1 = e2
                        | otherwise = (Apply (set e1 e3 e2) (set e1 e4 e2))
set e1 (Constr vars e3) e2 | (Constr vars e3)==e1 = e2
                           | otherwise = (Constr vars (set e1 e3 e2))
set e1 (Case ct e3 ces) e2 | (Case ct e3 ces)==e1 = e2
                           | otherwise = (Case ct (set e1 e3 e2) (map (\(Branch pat e4)->(Branch pat (set e1 e4 e2))) ces))
set e1 (GuardedExpr vars e3 e4) e2 | (GuardedExpr vars e3 e4)==e1 = e2
                                   | otherwise = (GuardedExpr vars (set e1 e3 e2) (set e1 e4 e2))
set e1 (Choice e3) e2 | (Choice e3)==e1 = e2
                      | otherwise = (Choice (set e1 e3 e2))                  
 
isGround (Comb ConsCall _ es) = foldr (&&) True (map isGround es)
isGround (Comb FuncCall c _) | c=="success" = True
                             | otherwise = False
isGround (Comb PartCall _ _) = True
isGround (Var _) = True
isGround (Lit _) = True
isGround (Apply _ _) = False
isGround (Constr _ _) = False
isGround (Or _ _) = False
isGround (Case _ _ _) = False
isGround (GuardedExpr _ _ _) = False
isGround (Choice _) = False

makeVarsFresh est e vs = let (fv,est2)=getfreshvars (length vs) est
                         in (est2,fst (replace est2 vs fv False e))

bindFParams vs es e = fst (replace (EState [] SE 0 0) vs es False e)

----------------------------------------------------------------------------
-- pretty printer for case expressions:

ppExpr _ (Var n) = ppVar n
ppExpr _ (Lit l) = ppLit l
ppExpr b (Comb _ cf es) =
  "(\"" ++ getOrigName cf ++ "\" " ++ ppList (ppExpr b) es ++ ")"
ppExpr b (Apply e1 e2) =
  "(" ++ ppExpr b e1 ++ " " ++ ppExpr b e2 ++ ")"
ppExpr b (Constr xs e) =
  "(Constr " ++ ppVars xs ++ (ppExpr b) e ++ ")"
ppExpr b (Or e1 e2) =
  "(Or " ++ ppExpr b e1 ++ " " ++ ppExpr b e2 ++ ")"
ppExpr b (Case Rigid e cs) =
  "(Case " ++ ppExpr b e ++ " of\n " ++ ppList (ppBranch (b+2)) cs
  ++ blanks b ++ ")"
ppExpr b (Case Flex e cs) =
  "(FCase " ++ ppExpr b e ++ " of\n " ++ ppList (ppBranch (b+2)) cs
  ++ blanks b ++ ")"
ppExpr b (GuardedExpr xs e1 e2) =
  "(GuardedExpr " ++ ppVars xs
                  ++ ppExpr b e1 ++ " " ++ ppExpr b e2 ++ ")"
ppExpr b (Choice e) = "(Choice " ++ ppExpr b e ++ ")"

ppVar i = "v" ++ show i

ppVars vs = " [" ++ concat (intersperse "," (map ppVar vs)) ++ "] "

ppLit (Intc   i) = show i
ppLit (Floatc f) = show f
ppLit (Charc  c) = "'" ++ show (ord c) ++ "'"

ppBranch b (Branch (Pattern s xs) e) = blanks b ++ "(\"" ++ s ++ "\" "
                                     ++ ppList ppVar xs
                                     ++ " -> " ++ ppExpr b e ++ ")\n"
ppBranch b (Branch (LPattern l) e) = blanks b ++ "(" ++ ppLit l ++ " "
                                   ++ " -> " ++ ppExpr b e ++ ")\n"

ppList format elems = ppListElems format elems

ppListElems _ [] = ""
ppListElems format [x] = format x
ppListElems format (x1:x2:xs) = format x1 ++ " "
                                  ++ ppListElems format (x2:xs)

blanks b = if b==0 then "" else ' ':blanks (b-1)

-- end of program
