-----------------------------------------------------------------------------
-- State-Server for GUI
--
-- * the server is started with function serveGUI
--
-- * the server listens to messages of type GUIServerMsg
--
-- Johannes Koj, October 2000
-- Modified by Michael Hanus, December 2000
-----------------------------------------------------------------------------

import Base
import Conf
import Flat
import Types
import IOServer
import Tk
import Ports

-----------------------------------------------------------------------------
-- the server
 
data GUIServerMsg =
       SetEditWindow TkRefType | SetStatusMode String | SetAna String | 
       SetCopyString String | New | Open String String String [String] |
       Save String | SaveAs String String | StartDebugger String |
       GetEditWindow TkRefType | GetStatusMode String | GetExpr Expr |
       GetAnaRes String String String | GetAnaNames [String] |
       GetCopyString String | Terminate

serveGUI eval rigid
serveGUI (Terminate:_) _ _ _ _ _ _ _ = success

serveGUI ((SetEditWindow ref):msgs) _ statusmode flatprog filename anaRes copystring ios =
  serveGUI msgs (Just ref) statusmode flatprog filename anaRes copystring ios 

serveGUI ((SetStatusMode mode):msgs) editwindow _ flatprog filename anaRes copystring ios = 
  serveGUI msgs editwindow mode flatprog filename anaRes copystring ios 

serveGUI ((SetAna ana):msgs) editwindow statusmode (Just prog) filename _ copystring ios = 
  serveGUI msgs editwindow statusmode (Just prog) filename (Just ((snd (head (filter (\(x,_)->x==ana) anaList))) prog)) copystring ios 

serveGUI ((SetAna _):msgs) editwindow statusmode Nothing filename anaRes copystring ios = 
  serveGUI msgs editwindow statusmode Nothing filename anaRes copystring ios 

serveGUI ((SetCopyString cs):msgs) editwindow statusmode flatprog filename anaRes _ ios =
  serveGUI msgs editwindow statusmode flatprog filename anaRes cs ios 

serveGUI (New:msgs) editwindow statusmode _ _ _ copystring ios =
  serveGUI msgs editwindow statusmode Nothing "" Nothing copystring ios

serveGUI ((Open name contents parserout funs):msgs) editwindow statusmode _ filename anaRes copystring ios =
  let flatprog free
  in send (OpenIO (if name=="" then filename else name) contents flatprog parserout) ios &>
     funs =:= getTopFuns flatprog &>
     serveGUI msgs editwindow statusmode (Just flatprog) (if name=="" then filename else name) Nothing copystring ios

serveGUI ((Save contents):msgs) editwindow statusmode flatprog filename anaRes copystring ios =
  send (SaveIO filename contents) ios &>
  serveGUI msgs editwindow statusmode flatprog filename anaRes copystring ios

serveGUI ((SaveAs name contents):msgs) editwindow statusmode flatprog filename anaRes copystring ios =
  send (SaveIO name contents) ios &>
  serveGUI msgs editwindow statusmode flatprog name anaRes copystring ios

serveGUI ((StartDebugger ans):msgs) editwindow statusmode Nothing filename anaRes copystring ios =
  ans =:= "No program loaded" &>
  serveGUI msgs editwindow statusmode Nothing filename anaRes copystring ios

serveGUI ((StartDebugger ans):msgs) editwindow statusmode (Just (Prog name imps types funs ops tr)) filename anaRes copystring ios =
 let tr_main = trans tr "main" in
  (if tr_main == ""
   then ans =:= "Debugger not started: function \"main\" undefined!"
   else ans =:= "Debugger started" &>
        send (StartDebuggerIO (Prog name imps types funs ops tr)
                              (Comb FuncCall tr_main [])) ios   ) &>
  serveGUI msgs editwindow statusmode (Just (Prog name imps types funs ops tr))
           filename anaRes copystring ios
  where trans [] _ = ""
        trans ((Trans n1 n2):ts) n | n1==n = n2
                                   | otherwise = trans ts n
     
serveGUI ((GetEditWindow ref):msgs) (Just editwindow) statusmode flatprog filename anaRes copystring ios = 
  ref=:=editwindow &>
  serveGUI msgs (Just editwindow) statusmode flatprog filename anaRes copystring ios 

serveGUI ((GetStatusMode mode):msgs) editwindow statusmode flatprog filename anaRes copystring ios = 
  mode=:=statusmode &>
  serveGUI  msgs editwindow statusmode flatprog filename anaRes copystring ios  

serveGUI ((GetExpr expr):msgs) editwindow statusmode (Just (Prog name imps types funs ops tr)) filename anaRes copystring ios =
  expr=:=Comb FuncCall (trans tr "main") [] &>
  serveGUI msgs editwindow statusmode (Just (Prog name imps types funs ops tr)) filename anaRes copystring ios 
  where trans [] n = n
        trans ((Trans n1 n2):ts) n | n1==n = n2
                                   | otherwise = trans ts n

-- get the analysis result for a particular (external) function name:
serveGUI ((GetAnaRes name res color):msgs) editwindow statusmode flatprog filename (Just anaRes) copystring ios = 
  let Just (Prog modname _ _ funs _ trans) = flatprog
      int2extName = stripModName (cutPath modname) trans
      (r,c) = processResult (snd (head (filter (\(x,_)->int2extName x == name) anaRes)))
   in res=:=r &> color=:=c &>
      serveGUI msgs editwindow statusmode flatprog filename (Just anaRes) copystring ios 
  where processResult (Message m c) = (m,c)
        processResult (Graph g c) | send (DisplayGraph g) ios = ("",c)

serveGUI ((GetAnaRes _ res color):msgs) editwindow statusmode flatprog filename Nothing copystring ios =
  res=:="" &> color=:="" &>
  serveGUI msgs editwindow statusmode flatprog filename Nothing copystring ios 

serveGUI ((GetAnaNames names):msgs) editwindow statusmode flatprog filename anaRes copystring ios =
  names=:=(map fst anaList) &>
  serveGUI msgs editwindow statusmode flatprog filename anaRes copystring ios

serveGUI ((GetCopyString cs):msgs) editwindow statusmode flatprog filename anaRes copystring ios = 
  cs=:=copystring &>
  serveGUI msgs editwindow statusmode flatprog filename anaRes copystring ios


----------------------------------------------------------------------------
-- Auxiliary function:

-- Compute the list of (external) function names defined
-- on top-level in the current module:
getTopFuns :: Prog -> [String]
getTopFuns (Prog pname _ _ funs _ trans) =
    filter (isTopFun (cutPath pname) trans)
           (map (revTrans trans) funs)
  where
   revTrans [] _ = ""
   revTrans ((Trans n1 n2):ts) (Func n arity typ rule)
                            | n==n2     = n1
                            | otherwise = revTrans ts (Func n arity typ rule)

   isTopFun _ [] f = False
   isTopFun pname ((Trans n1 n2):ts) f
         | f==n1 && (n2==pname++"."++n1 || pname=="prelude") = True
         | otherwise = isTopFun pname ts f

-- end of GUIServer
