-----------------------------------------------------------------------------
-- IO-Server for execution of io actions
--
-- * the server is started by "startIOServer"
--
-- * the server listens to messages of type IOServerMsg
--
-- * a FlatCurry file is read with all its imports by "readFlatCurryMod"
--
-- Johannes Koj, October 2000
-- Modified by Michael Hanus, December 2000
-----------------------------------------------------------------------------

import Conf
import Flat
import Debugger
import System
import DaVinci
import Ports

-----------------------------------------------------------------------------
-- the server

data IOServerMsg = OpenIO String String Prog String
                 | SaveIO String String
                 | ReadFlat String Prog
                 | StartDebuggerIO Prog Expr
                 | DisplayGraph DvGraph

startIOServer = openNamedPort ioServerPort >>= ioserve

ioserve ((OpenIO name contents flatprog parserout):msgs) =
  if name==""
  then do unifyIO contents ""
          unifyIO flatprog (Prog "" [] [] [] [] [])
          unifyIO parserout ""
          ioserve msgs 
  else do fcont <- readFile name
          unifyIO contents fcont
          (fp,po) <- readFlatCurryMod (cutNameSuffix name)
          unifyIO fp flatprog
          unifyIO po parserout
          ioserve msgs

ioserve ((SaveIO name contents):msgs) =
  do if name=="" then done
                 else writeFile name contents
     ioserve msgs

ioserve ((ReadFlat name flatprog):msgs) = 
  do readFlatCurryModule name >>= unifyIO flatprog
     ioserve msgs

ioserve ((StartDebuggerIO prog expr):msgs) = 
  do startDebugger prog expr
     ioserve msgs

ioserve ((DisplayGraph g):msgs) =
  do dvDisplay g
     ioserve msgs

---------------------------------------------------------------------------
-- auxiliary

connectIOServer = connectPort (ioServerPort++"@localhost")

-- read a FlatCurry module and all its imported modules:
readFlatCurryMod :: String -> IO (Prog,String)
readFlatCurryMod mod =
  do ret <- compileAndTest mod
     if ret then collectMods [mod] [] types ops funs trans
            else do unifyIO types []
                    unifyIO ops []
                    unifyIO funs []
                    unifyIO trans []
     pofile <- getPOFilename
     po <- readFile pofile
     system ("rm "++pofile)
     return (Prog mod [] types funs ops trans, po)

  where types,ops,funs,trans free

        collectMods [] _ ts os fs trs =
           do unifyIO ts []
              unifyIO os []
              unifyIO fs []
              unifyIO trs []
        collectMods (m:ms) implist ts os fs trs =
           if m `elem` implist
           then do collectMods ms implist ts1 os1 fs1 trs1
                   unifyIO ts ts1
                   unifyIO os os1
                   unifyIO fs fs1
                   unifyIO trs trs1
           else do (Prog _ imps types funs ops trans) <- readFlatCurryModule m
                   collectMods ms ((m:imps)++implist) ts1 os1 fs1 trs1
                   collectMods imps (m:implist) ts2 os2 fs2 trs2
                   unifyIO ts (types++ts1++ts2)
                   unifyIO os (ops++os1++os2)
                   unifyIO fs (funs++fs1++fs2)
                   unifyIO trs (trans++trs1++trs2)
          where ts1,ts2,os1,os2,fs1,fs2,trs1,trs2 free

compileAndTest mod =
  do preprocess_module mod
     pakcshome <- getEnviron "PAKCSHOME"
     pofile <- getPOFilename
     system (pakcshome++"/bin/parsecurry -fcy "++mod++" > "++pofile++" 2>&1")
     ret <- system ("test -f "++mod++".fcy")
     return (if ret==0 then True else False)

-- compute name for auxiliary file for parser outputs:
getPOFilename =
  do pid <- getPID
     return ("/tmp/pakcsoutput_"++show pid)

unifyIO :: a -> a -> IO ()
unifyIO a1 a2 | a1=:=a2 = done

-- cut any suffix present in the source file name (e.g., ".curry", ".lcurry"):
cutNameSuffix s = let dropsuf = dropWhile (/='.') (reverse s)
                   in if dropsuf==[] then s
                                     else reverse (tail dropsuf)

-- end of program
