---------------------------------------------------------------------------
-- A parser for (pure) Prolog programs together with a translator into Curry:
--
-- The presentation of this parser should be the FlatCurry representation
-- but currently we use it in a liberal format since we only translate it
-- into a Curry program.
--
-- Current restrictions on the Prolog source:
-- * no floats, no infix/prefix/postfix operators, no "_" in names,...
--
-- Michael Hanus, August 2001
---------------------------------------------------------------------------

import Parser
import Char
import List
import Flat
import System

--------------------------------------------------------------------------
-- scanner for Prolog programs:
-- (main purpose: deletion of white spaces)

data PL_Token = PL_Atom   String  -- Prolog atom
              | PL_Var    String  -- Prolog variable
              | PL_Number Int     -- Prolog number (currently: only Int)
              | PL_Token  String  -- other tokens

pl_scan :: String -> [PL_Token]
pl_scan [] = []
pl_scan (c:cs) | c==' '    = pl_scan cs
               | c=='\n'   = pl_scan cs
               | isDigit c = let (ns,remcs) = pl_scan_num [c] cs
                              in PL_Number (foldr1 (\x y->x+10*y)
                                               (map (\d->ord d - ord '0') ns))
                                 : pl_scan remcs
               | isUpper c = let (ids,remcs) = pl_scan_id [c] cs
                              in PL_Var (reverse ids) : pl_scan remcs
               | isLower c = let (ids,remcs) = pl_scan_id [c] cs
                              in PL_Atom (reverse ids) : pl_scan remcs
               | c `elem` pl_specials
                           = let (ids,remcs) = pl_scan_op [c] cs
                              in PL_Atom (reverse ids) : pl_scan remcs
               | c=='['    = pl_scan_lsquare cs
               | c=='%'    = pl_scan_comment cs
               | otherwise = PL_Token [c] : pl_scan cs -- e.g., ( ) , ! |

-- scan a number:
pl_scan_num ns [] = (ns,[])
pl_scan_num ns (c:cs) = if isDigit c
                        then pl_scan_num (c:ns) cs
                        else (ns,c:cs)

-- scan an identifier:
pl_scan_id ids [] = (ids,[])
pl_scan_id ids (c:cs) = if isAlphaNum c
                        then pl_scan_id (c:ids) cs
                        else (ids,c:cs)

-- scan an operator:
pl_scan_op ids [] = (ids,[])
pl_scan_op ids (c:cs) = if c `elem` pl_specials
                        then pl_scan_op (c:ids) cs
                        else (ids,c:cs)

-- special characters allowed in Prolog operators:
pl_specials =
    ['+','-','*','/','<','>','=','`','\\',':','.','?','@','#','$','&','^','~']

pl_scan_lsquare [] = [PL_Token "["]
pl_scan_lsquare (c:cs) | c==']'    = PL_Atom "[]" : pl_scan cs
                       | otherwise = PL_Token "[" : pl_scan (c:cs)

pl_scan_comment [] = []
pl_scan_comment (c:cs) = if c=='\n' then pl_scan cs
                                    else pl_scan_comment cs

-- example: pl_scan "p(12,f(34 ,[] , b), [x|y]).\n"


--------------------------------------------------------------------------
-- Parser for Prolog programs:
-- the input is the result of the scanner

-- a Prolog program is a sequence of clauses:
prolog_program  =  star clause

-- a clause is a fact or a rule:
clause   =  lit l <*> clause_body b >>> (l,b)
 where l,b free

clause_body =  pl_atom "."              >>> [Comb FuncCall "success" []]
          <||> pl_atom ":-" <*>
               lit b1 <*> more_lits bs  >>> (b1:bs)
 where b1,bs free

-- more comma separated literals:
more_lits   =  pl_atom "."                              >>> []
          <||> pl_token "," <*> lit l <*> more_lits ls  >>> (l:ls)
 where l,ls free

lit     =  pl_atom p                       >>> (Comb FuncCall (trans p) [])
      <||> pl_atom p <*> pl_token "(" <*>
           term t <*> more_terms tl        >>> (Comb FuncCall (trans p) (t:tl))
 where
   p,t,tl free

   trans n = if n=="=" then "=:=" else n


term    =  terminal (PL_Number v)                >>> (Lit (Intc v))
      <||> terminal (PL_Var name)                >>> (Var (varname2int name))
      <||> pl_atom a                             >>> (Comb ConsCall a [])
      <||> pl_atom a <*> pl_token "(" <*>
           term t <*> more_terms tl              >>> (Comb ConsCall a (t:tl))
      <||> pl_token "[" <*> term e <*>
           more_listelems e fl                   >>> fl
 where v,name,a,e,t,tl,fl free

more_listelems e =
       pl_token "|" <*> term t <*> pl_token "]"  >>> (Comb ConsCall ":" [e,t])
  <||> pl_token "," <*> term t <*>
       more_listelems t l                        >>> (Comb ConsCall ":" [e,l])
  <||> pl_token "]"          >>> (Comb ConsCall ":" [e, Comb ConsCall "[]" []])
 where t,l free

-- assign unique number to a variable name:
varname2int name = foldl1 ((+) . (62*)) (map char2int name)
 where char2int c | isLower c = ord c - ord 'a'
                  | isUpper c = ord c - ord 'A' + 26
                  | otherwise = ord c - ord '0' + 52

-- more comma separated terms:
more_terms  =  pl_token ")"                               >>> []
          <||> pl_token "," <*> term t <*> more_terms tl  >>> (t:tl)
 where t,tl free

pl_atom cs = terminal (PL_Atom cs)

pl_token cs = terminal (PL_Token cs)

-- example application:
-- clause ast (pl_scan "p(1,f(3,b),c):-q(1),q(2).\n") =:= []


--------------------------------------------------------------------------
-- transform clauses (i.e., pair of lhs term / rhs terms)
-- into left-linear clauses by introducing equalities and add list
-- of extra variables in the right-hand side:

trans_clause (Comb ct f ts, body) =
  let (nts,eqs,lvs) = lin_terms [] ts
   in (Comb ct f nts, eqs++body, extravars_in_terms lvs body)

-- "linearize" a sequence of terms:
lin_terms vs [] = ([],[],vs)
lin_terms vs (t:ts) = (nt:nts,teqs++tseqs,vsts)
  where (nt, teqs, vst ) = lin_term vs t
        (nts,tseqs,vsts) = lin_terms vst ts

lin_term vs (Lit l) = (Lit l, [], vs)
lin_term vs (Var i) =
   if i `elem` vs
   then let vi = unused_index vs
         in (Var vi, [Comb FuncCall "=:=" [Var vi, Var i]],vi:vs)
   else (Var i, [], i:vs)
lin_term vs (Comb ct f ts) = (Comb ct f nts, eqs, nvs)
  where (nts,eqs,nvs) = lin_terms vs ts

unused_index vs = unused 0
 where unused i = if i `elem` vs then unused (i+1) else i

-- compute all extra variables in a sequence of terms:
extravars_in_terms _ [] = []
extravars_in_terms vs (t:ts) =
  union (extravars_in_term vs t) (extravars_in_terms vs ts)

extravars_in_term _  (Lit _) = []
extravars_in_term vs (Var i) = if i `elem` vs then [] else [i]
extravars_in_term vs (Comb _ _ ts) = extravars_in_terms vs ts


--------------------------------------------------------------------------
-- pretty printer into Curry notation:
pp_clause (hd,body,extravars) =
  pp_head hd ++ " = " ++ concat (intersperse " &> " (map pp_term body))
  ++ "\n" ++
  (if extravars==[]
   then ""
   else "  where " ++
        concat (intersperse "," (map (\i->"x"++show i) extravars)) ++
        " free\n")

pp_head (Comb _ f ts) = pp_name f++(concatMap (\t->" "++pp_term t) ts)

pp_term (Lit (Intc v)) = show v
pp_term (Var i) = "x"++show i
pp_term (Comb _ f []) = pp_name f
pp_term (Comb _ f [t]) = "("++pp_name f++" "++pp_term t++")"
pp_term (Comb _ f [t1,t2])
  | f==":"           = "("++pp_term t1++":"++pp_term t2++")"
  | isAlpha (head f) = "("++pp_name f++" "++pp_term t1++" "++pp_term t2++")"
  | otherwise        = "("++pp_term t1++" "++f++" "++pp_term t2++")"
pp_term (Comb _ f (t1:t2:t3:ts))
  = "("++pp_name f++(concatMap (\t->" "++pp_term t) (t1:t2:t3:ts))++")"

pp_name cs | isAlpha (head cs) = cs
           | cs == "[]"        = cs
           | otherwise         = ('(':cs)++")"


-- translate a Prolog program into a Curry program:
prolog2curry file =
  do protest <- system ("test -f "++file++".pro")
     if protest/=0
      then done
      else readFile (file++".pro") >>= \plprog ->
           doSolve (plprog =:= plprog) >>
           writeFile
             (file++".curry")
             (concatMap (pp_clause . trans_clause)
                (findfirst (\ast->prolog_program ast (pl_scan plprog) =:= [])))


main1 = prolog2curry "app"


-- read a Prolog file and translate into FlatCurry:
readPrologFlatCurry prog =
  do prolog2curry prog
     readFlatCurry prog

main2 = readPrologFlatCurry "app"
