---------------------------------------------------------------------------
-- A parser for Toy programs together with a translator into Curry:
--
-- The presentation of this parser should be the FlatCurry representation
-- but currently we use it in a liberal format since we only translate it
-- into a Curry program.
--
-- Current restrictions on the Toy source:
-- * no floats, no infix/prefix/postfix operators, no "_" in names,...
--
-- Michael Hanus, August 2001
---------------------------------------------------------------------------

import Parser
import Char
import List
import Flat
import System

--------------------------------------------------------------------------
-- scanner for Toy programs:
-- (main purpose: deletion of white spaces)

data Toy_Token = Toy_Atom   String  -- Toy atom
               | Toy_Var    String  -- Toy variable
               | Toy_Number Int     -- Toy number (currently: only Int)
               | Toy_EOR            -- end of Toy rule marker
               | Toy_Token  String  -- other tokens

toy_scan :: String -> [Toy_Token]
toy_scan [] = []
toy_scan (c:cs) | c==' '    = toy_scan cs
                | c=='\n'   = if cs/=[] && (head cs == ' ' || head cs == '\t')
                              then toy_scan cs
                              else Toy_EOR : toy_scan cs
                | isDigit c = let (ns,remcs) = toy_scan_num [c] cs
                               in Toy_Number (foldr1 (\x y->x+10*y)
                                               (map (\d->ord d - ord '0') ns))
                                  : toy_scan remcs
                | isUpper c = let (ids,remcs) = toy_scan_id [c] cs
                               in Toy_Var (reverse ids) : toy_scan remcs
                | isLower c = let (ids,remcs) = toy_scan_id [c] cs
                               in Toy_Atom (reverse ids) : toy_scan remcs
                | c `elem` toy_specials
                            = let (ids,remcs) = toy_scan_op [c] cs
                               in (if ids `elem` ["=","==<"]
                                   then Toy_Token (reverse ids)
                                   else Toy_Atom (reverse ids))
                                  : toy_scan remcs
                | c=='['    = toy_scan_lsquare cs
                | c=='%'    = toy_scan_comment cs
                | otherwise = Toy_Token [c] : toy_scan cs -- e.g., ( ) , ! |

-- scan a number:
toy_scan_num ns [] = (ns,[])
toy_scan_num ns (c:cs) = if isDigit c
                        then toy_scan_num (c:ns) cs
                        else (ns,c:cs)

-- scan an identifier:
toy_scan_id ids [] = (ids,[])
toy_scan_id ids (c:cs) = if isAlphaNum c
                        then toy_scan_id (c:ids) cs
                        else (ids,c:cs)

-- scan an operator:
toy_scan_op ids [] = (ids,[])
toy_scan_op ids (c:cs) = if c `elem` toy_specials
                        then toy_scan_op (c:ids) cs
                        else (ids,c:cs)

-- special characters allowed in Toy operators:
toy_specials =
    ['+','-','*','/','<','>','=','`','\\',':','.','?','@','#','$','&','^','~']

toy_scan_lsquare [] = [Toy_Token "["]
toy_scan_lsquare (c:cs) | c==']'    = Toy_Atom "[]" : toy_scan cs
                       | otherwise = Toy_Token "[" : toy_scan (c:cs)

toy_scan_comment [] = []
toy_scan_comment (c:cs) = if c=='\n' then toy_scan cs
                                    else toy_scan_comment cs

-- example: toy_scan "p 12 (f 34 []  b) [X|Y]\n"


--------------------------------------------------------------------------
-- Parser for Toy programs:
-- the input is the result of the scanner

-- a Toy program is a sequence of clauses:
toy_program  =  terminal Toy_EOR <*> toy_program p  >>> p
           <||> equation e       <*> toy_program p  >>> (e:p)
           <||> empty                               >>> []
 where e,p free

-- a clause is a fact or a rule:
equation =  term lhs <*> toy_token "=" <*> equation_rhs rhs >>> (lhs,rhs)
 where lhs,rhs free

equation_rhs =  term rhs >>> (Comb ConsCall "success" [], rhs)
           <||> term rhs <*> toy_token "<==" <*> term cond >>> (cond,rhs)
 where rhs,cond free

term  =  toy_atom a <*> (star bterm) args      >>> (Comb ConsCall a args)
    <||> bterm t <*> (star bterm) ts >>> foldl1 Apply (t:ts)
 where a,args,t,ts free

bterm   =  terminal (Toy_Number v)                >>> (Lit (Intc v))
      <||> terminal (Toy_Var name)                >>> (Var (varname2int name))
      <||> toy_token "(" <*> toy_atom a <*>
           (star term) tl <*> toy_token ")"       >>> (Comb ConsCall a tl)
      <||> toy_atom a                             >>> (Comb ConsCall a [])
      <||> toy_token "[" <*> term e <*>
           more_listelems e fl                   >>> fl
 where v,name,a,e,tl,fl free

more_listelems e =
       toy_token "|" <*> term t <*> toy_token "]"  >>> (Comb ConsCall ":" [e,t])
  <||> toy_token "," <*> term t <*>
       more_listelems t l                        >>> (Comb ConsCall ":" [e,l])
  <||> toy_token "]"          >>> (Comb ConsCall ":" [e, Comb ConsCall "[]" []])
 where t,l free

-- assign unique number to a variable name:
varname2int name = foldl1 ((+).(62*)) (map char2int name)
 where char2int c | isLower c = ord c - ord 'a'
                  | isUpper c = ord c - ord 'A' + 26
                  | otherwise = ord c - ord '0' + 52

-- inversion of varname2int:
int2varname :: Int -> String
int2varname i = if i==0 then []
                else int2varname (i `div` 62) ++ [int2char (i `mod` 62)]
 where int2char c | c<26 = chr (c + ord 'a')
                  | c<52 = chr (c + ord 'A' - 26)
                  | otherwise = chr (c + ord '0' - 52)


toy_atom cs = terminal (Toy_Atom cs)

toy_token cs = terminal (Toy_Token cs)

-- example application:
-- toy_program ast (toy_scan "append [] Ys = Ys\n") =:= []
-- toy_program ast (toy_scan "p 1 (f 3 b) c = (q 1 (q 2))\np 1 (f 3 b) c = [1,2]\n") =:= []
-- clause ast (toy_scan "p 1 (f 3 b) c = [1,2]\n") =:= []
-- term ast (toy_scan "p 1 (f 3 b) c") =:= []


--------------------------------------------------------------------------
-- pretty printer into Curry notation:
pp_clause (lhs,(cond,rhs)) =
  pp_head lhs ++
  (if cond==(Comb ConsCall "success" []) then "" else " | " ++ pp_term cond)
  ++ " = " ++ pp_term rhs ++ "\n" ++
  (if extravars==[]
   then ""
   else "  where " ++
        concat (intersperse "," (map (\i->"x"++show i) extravars)) ++
        " free\n")
 where
   extravars = diff (union (vars_in_term cond) (vars_in_term rhs))
                    (vars_in_term lhs)

   diff [] _  = []
   diff (x:xs) ys = if x `elem` ys then diff xs ys else x : diff xs ys

-- compute all variables in a term:
vars_in_term (Lit _) = []
vars_in_term (Var i) = [i]
vars_in_term (Comb _ _ ts) = foldr union [] (map vars_in_term ts)
vars_in_term (Apply e1 e2) = union (vars_in_term e1) (vars_in_term e2)


pp_head (Comb _ f ts) = pp_name f++(concatMap (\t->" "++pp_term t) ts)

pp_term (Lit (Intc v)) = show v
pp_term (Var i) = int2varname i
pp_term (Comb _ f []) = f --pp_name f
pp_term (Comb _ f [t]) = "("++pp_name f++" "++pp_term t++")"
pp_term (Comb _ f [t1,t2])
  | f==":"           = "("++pp_term t1++":"++pp_term t2++")"
  | isAlpha (head f) = "("++pp_name f++" "++pp_term t1++" "++pp_term t2++")"
  | otherwise        = "("++pp_term t1++" "++f++" "++pp_term t2++")"
pp_term (Comb _ f (t1:t2:t3:ts))
  = "("++pp_name f++(concatMap (\t->" "++pp_term t) (t1:t2:t3:ts))++")"
pp_term (Apply e1 e2) = "("++pp_apply e1++" "++pp_term e2++")"

pp_apply (Lit l) = pp_term (Lit l)
pp_apply (Var i) = pp_term (Var i)
pp_apply (Comb ct f args) = pp_term (Comb ct f args)
pp_apply (Apply e1 e2) = pp_apply e1++" "++pp_term e2

pp_name cs | isAlpha (head cs) = cs
           | cs == "[]"        = cs
           | otherwise         = ('(':cs)++")"


-- translate a Toy program into a Curry program:
toy2curry file =
  do toytest <- system ("test -f "++file++".toy")
     if toytest/=0
      then done
      else readFile (file++".toy") >>= \toyprog ->
           doSolve (toyprog =:= toyprog) >>
           writeFile
             (file++".curry")
             (concatMap pp_clause
               (findfirst (\ast->toy_program ast (toy_scan toyprog) =:= [])))


main1 = toy2curry "app"


-- read a Toy file and translate into FlatCurry:
readToyFlatCurry prog =
  do toy2curry prog
     readFlatCurry prog

main2 = readToyFlatCurry "app"

-- translate a Toy program into a Curry program:
main =
  do readFile ("app.toy") >>= \toyprog ->
      doSolve (toyprog =:= toyprog) >>
      putStr (show
               (findfirst (\ast->toy_program ast (toy_scan toyprog) =:= [])))

