%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Definition of external functions of module IO:
%
% Note: the Prolog term '$stream'('$inoutstream'(In,Out)) represents a handle
% for a stream that is both readable (on In) and writable (on Out)
% Otherwise, handles are represented by Prolog terms of the form '$stream'(N).
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

:- use_module('../prologbasics').
:- use_module(library(system)).
:- ensure_loaded(user:prim_ports). % to implement IO.prim_hWaitForInputsOrMsg

prim_stdin(Stream) :- prolog_flag(user_input,Stream).

prim_stdout(Stream) :- prolog_flag(user_output,Stream).

prim_stderr(Stream) :- prolog_flag(user_error,Stream).

prim_openFile(A,Mode,_,'$io'(Stream)) :-
	string2Atom(A,FName),
	curryFileMode2plmode(Mode,PMode),
	open(FName,PMode,Stream).

curryFileMode2plmode('IO.ReadMode',read).
curryFileMode2plmode('IO.WriteMode',write).
curryFileMode2plmode('IO.AppendMode',append).


prim_hClose('$stream'('$inoutstream'(In,Out)),_,'$io'('Prelude.()')) :- !,
	flush_output(Out),
	close(Out),
	(In==Out -> true ; close(In)).
prim_hClose(Stream,_,'$io'('Prelude.()')) :-
	(isOutputStream(Stream) -> flush_output(Stream) ; true),
	close(Stream).


prim_hFlush('$stream'('$inoutstream'(_,Out)),_,'$io'('Prelude.()')) :- !,
	flush_output(Out).
prim_hFlush(Stream,_,'$io'('Prelude.()')) :-
	(isOutputStream(Stream) -> flush_output(Stream) ; true).


prim_hIsEOF('$stream'('$inoutstream'(In,_)),_,'$io'(B)) :- !,
	(at_end_of_stream(In) -> B='Prelude.True' ; B='Prelude.False').
prim_hIsEOF(Stream,_,'$io'(B)) :-
	(at_end_of_stream(Stream) -> B='Prelude.True' ; B='Prelude.False').


prim_hSeek(Handle,SeekMode,Pos,_,'$io'('Prelude.()')) :-
	currySeekMode2plmode(SeekMode,PlSM),
	seek(Handle,Pos,PlSM,_).

currySeekMode2plmode('IO.AbsoluteSeek',bof).
currySeekMode2plmode('IO.RelativeSeek',current).
currySeekMode2plmode('IO.SeekFromEnd',eof).


?- block prim_hWaitForInputs(?,?,?,?,-,?).
prim_hWaitForInputs(RStreams,RTO,_,'$io'(N),E0,E) :-
	user:derefAll(RStreams,Streams),
	selectInstreams(Streams,InStreams),
	user:deref(RTO,TimeOut),
	(TimeOut<0 -> TO=off
	            ; TOSec is TimeOut//1000,
	              TOMSec is (TimeOut mod 1000) * 1000, % in milliseconds
	              TO = TOSec:TOMSec),
	waitForInputDataOnStreams(InStreams,TO,N),
	!, E0=E.

selectInstreams([],[]).
selectInstreams(['$stream'('$inoutstream'(In,_))|Streams],[In|InStreams]) :- !,
	selectInstreams(Streams,InStreams).
selectInstreams([Stream|Streams],[Stream|InStreams]) :-
	selectInstreams(Streams,InStreams).


prim_hGetChar('$stream'('$inoutstream'(In,_)),_,'$io'(C)) :- !,
	get_code(In,N), char_int(C,N).
prim_hGetChar(Stream,_,'$io'(C)) :-
	get_code(Stream,N), char_int(C,N).


prim_hPutChar('$stream'('$inoutstream'(_,Out)),C,_,'$io'('Prelude.()')) :- !,
	char_int(C,N), put_code(Out,N).
prim_hPutChar(Stream,C,_,'$io'('Prelude.()')) :-
	char_int(C,N), put_code(Stream,N).


prim_hIsReadable('$stream'('$inoutstream'(_,_)),_,'$io'('Prelude.True')) :- !.
prim_hIsReadable(Stream,_,'$io'(B)) :-
	(isInputStream(Stream) -> B='Prelude.True' ; B='Prelude.False').


prim_hIsWritable('$stream'('$inoutstream'(_,_)),_,'$io'('Prelude.True')) :- !.
prim_hIsWritable(Stream,_,'$io'(B)) :-
	(isOutputStream(Stream) -> B='Prelude.True' ; B='Prelude.False').


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% choice on a stream and an external port message stream:
?- block prim_hWaitForInputsOrMsg(?,-,?,?,?,?),
         prim_hWaitForInputsOrMsg(?,?,?,?,-,?).
prim_hWaitForInputsOrMsg(Handles,share(M),World,Result,E0,E) :-
        !,
	get_mutable(V,M),
	(V='$eval'(R) % external message stream has been already evaluated
	 -> E0=E, Result='$io'('Prelude.Right'(R))
	  ; prim_hWaitForInputsOrMsg(Handles,V,World,CResult,E0,E),
	    (CResult='$io'('Prelude.Left'(_))
  	     -> Result=CResult
	      ; CResult='$io'('Prelude.Right'(S)),
	        (compileWithSharing(variable) -> propagateShare(S,TResult) ; S=TResult),
	        Result='$io'('Prelude.Right'(TResult)),
	        update_mutable('$eval'(TResult),M))).
prim_hWaitForInputsOrMsg(_,[M|Ms],_,'$io'('Prelude.Right'([M|Ms])),E0,E) :-
	!, E0=E. % stream already evaluated
prim_hWaitForInputsOrMsg(RHandles,[],_,'$io'('Prelude.Left'(N)),E0,E) :- !,
	% message stream is empty, so anything must be received from the handles.
	user:derefAll(RHandles,Handles),
	selectInstreams(Handles,InStreams),
	waitForInputDataOnStreams(InStreams,off,N),
	!, E0=E.
prim_hWaitForInputsOrMsg(Handles,'Ports.basicServerLoop'(Port),World,
			Result,E0,E) :-
        Port='Ports.internalPort'(_,_,PNr,_),
	checkIncomingPortStreams(PNr,InPortStream,OutPortStream),
	!,
	readStreamUntilEndOfTerm(InPortStream,MsgString),
	(parse_received_message(InPortStream,OutPortStream,MsgString,Msg)
	 -> ifTracePort((write(user_error,'TRACEPORTS: Received message: '),
		         write(user_error,Msg), nl(user_error))),
	    Result='$io'('Prelude.Right'([Msg|'Ports.basicServerLoop'(Port)])), E0=E
	  ; write(user_error,'ERROR: Illegal message received (ignored): '),
	    putChars(user_error,MsgString), nl(user_error),
	    prim_hWaitForInputsOrMsg(Handles,'Ports.basicServerLoop'(Port),World,
	                                   Result,E0,E)),
	!.
prim_hWaitForInputsOrMsg(RHandles,'Ports.basicServerLoop'(Port),World,Result,E0,E):-
	user:derefAll(RHandles,Handles),
	selectInstreams(Handles,InStreams),
        Port='Ports.internalPort'(_,_,PNr,Socket),
	waitForSocketOrInputStreams(Socket,Client,InPortStream,OutPortStream,
				    InStreams,Index),
	(Client=no
	 -> % there is input on Handles:
	    Result='$io'('Prelude.Left'(Index)), E0=E
	  ; % there is a client connection:
	    ifTracePort((write(user_error,'TRACEPORTS: Connection to client: '),
		         write(user_error,Client), nl(user_error))),
	    (readPortMessage(PNr,InPortStream,OutPortStream,MsgString)
	     -> (parse_received_message(InPortStream,OutPortStream,MsgString,Msg)
	         -> ifTracePort((write(user_error,'TRACEPORTS: Received message: '),
		                 write(user_error,Msg), nl(user_error))),
		    Result='$io'('Prelude.Right'([Msg|'Ports.basicServerLoop'(Port)])), E0=E
	          ; write(user_error,'ERROR: Illegal message received (ignored): '),
		    putChars(user_error,MsgString), nl(user_error),
		    prim_hWaitForInputsOrMsg(Handles,
                                 'Ports.basicServerLoop'(Port),World,Result,E0,E))
	      ; % try reading next message:
		prim_hWaitForInputsOrMsg(Handles,
		                 'Ports.basicServerLoop'(Port),World,Result,E0,E))),
	!.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
