%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% basic predicates related to the SICStus-Prolog system

:- module(prologbasics,
	  [prolog/1, prologMajorVersion/1, prologMinorVersion/1, pakcsrc/2,
	   sicstus37orHigher/0, sicstus38orHigher/0,
	   sicstus310orHigher/0, generatePrologBasics/0,
%SICS37	   atom_codes/2, number_codes/2,
%SICS37	   put_code/1, put_code/2, put_byte/2, get_code/1, get_code/2, get_byte/2,
	   unifyWithOccursCheck/2,
	   waitConcurrentConjunction/6,
	   append/3, appendAtom/3, map1/2, map2/3, map1partial/2, map2partial/3,
	   getProgramArgs/1,
	   noSingletonWarnings/0, noRedefineWarnings/0, noDiscontiguousWarnings/0,
	   getRunTime/1, getElapsedTime/1, garbageCollectorOn/0, garbageCollectorOff/0,
	   fileModTime/2, fileSize/2, existsFile/1, existsDirectory/1,
	   deleteFile/1,
	   fileExistsAndNewer/2, canWriteFile/1, currentPID/1, currentTime/6,
	   getHostname/1, execCommand/4, forkProcessForGoal/1,
	   isInputStream/1, isOutputStream/1,
	   newSocket/1, socketConnect/5, closeSocketStream/2,
	   socketBind/3, socketListen/2, socketAccept/4,
	   waitForInputDataOnStreams/3,
	   waitForSocketClientStream/5, waitForSocketOrInputStreams/6,
	   try_save_program/1, saveprog_entry/2, try_save_predicates/2,
	   ensure_lib_loaded/1, compilePrologFile/1,
	   compilePrologFileAndSave/1, consultPrologorPOFile/2,
	   getNewPrologFileName/1, mainPrologFileName/2,
	   callAndReturnSuspensions/2, writeqWithVars/1,
	   genBlockDecl/4]).

:- use_module(library(system)).
:- use_module(library(sockets)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Principle kind of Prolog system and version used for this implementation.
prolog(sicstus).

prologMajorVersion(MV) :-
	prolog_flag(version,V,V),
	atom_chars(V,Vs),
	append("SICStus ",[MC|_],Vs),
	MV is MC-48, !.

prologMinorVersion(MV) :-
	prolog_flag(version,V,V),
	atom_chars(V,Vs),
	append("SICStus 3.",Vs1,Vs),
	append(MVs,[46|_],Vs1),
	number_chars(MV,MVs), !.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
:- multifile pakcsrc/2. % relevant for createSavedState

:- dynamic pakcsrc/2.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

unifyWithOccursCheck(T1,T2) :-
	sicstus38orHigher -> unify_with_occurs_check(T1,T2)
	                   ; T1=T2. % could be improved...

append([],Xs,Xs).
append([X|Xs],Ys,[X|Zs]) :- append(Xs,Ys,Zs).

% concatenate two atoms:
appendAtom(A1,A2,A3) :-
	atom_codes(A1,L1), atom_codes(A2,L2),
	append(L1,L2,L3),
	atom_codes(A3,L3).

%SICS37	atom_codes(A,L) :- atom_chars(A,L).
%SICS37	number_codes(A,L) :- number_chars(A,L).
%SICS37 put_code(C) :- put(C).
%SICS37 put_code(S,C) :- put(S,C).
%SICS37 put_byte(S,C) :- put(S,C).
%SICS37 get_code(C) :- get0(C).
%SICS37 get_code(S,C) :- get0(S,C).
%SICS37 get_byte(S,C) :- get0(S,C).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% get Sicstus version (e.g., '3.9'):
% Currently, we support only 3.5, 3.7, 3.8, 3.9, 3.10, 3.11, 3.12:
getSicstusVersion(SV) :-
	prolog_flag(version,V,V),
	atom_chars(V,Vs),
	(append("SICStus 3.5",_,Vs) -> SV='3.5' ;
	 append("SICStus 3.7",_,Vs) -> SV='3.7' ;
	 append("SICStus 3.8",_,Vs) -> SV='3.8' ;
	 append("SICStus 3.9",_,Vs) -> SV='3.9' ;
	 append("SICStus 3.10",_,Vs) -> SV='3.10' ;
	 append("SICStus 3.11",_,Vs) -> SV='3.11' ;
	 append("SICStus 3.12",_,Vs) -> SV='3.12' ;
	 write(user_error,'ERROR: UNKNOWN SICSTUS PROLOG VERSION:'),
	 nl(user_error),
	 write(user_error,'PLEASE MODIFY pakcs/curry2prolog/sicstusbasics.pl'),
	 nl(user_error), halt(1)).

sicstus37orHigher :-
	getSicstusVersion(SV),
	(SV = '3.7' ; sicstus38orHigher).

sicstus38orHigher :-
	getSicstusVersion(SV),
	(SV = '3.8' ; sicstus39orHigher).

sicstus385orHigher :- sicstus39orHigher, !.
sicstus385orHigher :-
	prolog_flag(version,V,V),
	atom_chars(V,Vs),
	append("SICStus 3.8.",[MV|_],Vs),
	MV>=53.

sicstus39orHigher :-
	getSicstusVersion(SV),
	(SV = '3.9' ; sicstus310orHigher).

sicstus310orHigher :-
	getSicstusVersion(SV),
	(SV = '3.10' ; SV = '3.11' ; SV = '3.12').


generatePrologBasics :-
	sicstus38orHigher
	-> system('cp sicstusbasics.pl prologbasics.pl')
	 ; system('sed "s/%SICS37/ /g" < sicstusbasics.pl > prologbasics.pl').


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Implementation of suspension for concurrent conjunction (&)

?- block waitConcurrentConjunction(?,?,?,-,-,?).
waitConcurrentConjunction(S1,S2,R,E1,E2,E) :- nonvar(E1), !,
	(S1='FAIL'(_) -> R=S1, E=E1 ; waitForEval(S2,R,E2,E)).
waitConcurrentConjunction(S1,S2,R,E1,E2,E) :- % E2 must be nonvar
	(S2='FAIL'(_) -> R=S2, E=E2 ; waitForEval(S1,R,E1,E)).

?- block waitForEval(?,?,-,?).
waitForEval(R,R,E,E).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% meta-predicates for higher-order programming:

% map a unary predicate on a list:
:- meta_predicate map1(:,?).
map1(_,[]).
map1(M:P,[X|Xs]) :- C =.. [P,X], call(M:C), map1(M:P,Xs).

% map a binary predicate on two lists:
:- meta_predicate map2(:,?,?).
map2(_,[],[]).
map2(M:P,[X|Xs],[Y|Ys]) :- C =.. [P,X,Y], call(M:C), map2(M:P,Xs,Ys).

% map a unary predicate (which might be a partial application) on a list:
:- meta_predicate map1partial(:,?).
map1partial(_,[]).
map1partial(M:P,[X|Xs]) :-
	P =.. [Pred|PartialArgs],
	append(PartialArgs,[X],Args),
	C =.. [Pred|Args], call(M:C),
	map1partial(M:P,Xs).

% map a binary predicate (which might be a partial application) on two lists:
:- meta_predicate map2partial(:,?,?).
map2partial(_,[],[]).
map2partial(M:P,[X|Xs],[Y|Ys]) :-
	P =.. [Pred|PartialArgs],
	append(PartialArgs,[X,Y],Args),
	C =.. [Pred|Args], call(M:C),
	map2partial(M:P,Xs,Ys).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% get program arguments:
getProgramArgs(Args) :- prolog_flag(argv,Args,Args).

% no warnings for singleton variables:
noSingletonWarnings :- prolog_flag(single_var_warnings,_,off).

% no warnings for redefining predicates:
noRedefineWarnings :- prolog_flag(redefine_warnings,_,off).

% turn off discontiguous clauses warnings:
noDiscontiguousWarnings :-
	sicstus38orHigher -> prolog_flag(discontiguous_warnings ,_,off) ; true.

% get current run time in msecs:
getRunTime(Time) :- statistics(runtime,[Time,_]).

% get current elapsed time in msecs:
getElapsedTime(Time) :- statistics(walltime,[Time,_]).

% turn on garbage collector:
garbageCollectorOn :- prolog_flag(gc,_,on).
% turn off garbage collector:
garbageCollectorOff :- prolog_flag(gc,_,off).

% get modification time of a file:
fileModTime(File,Time) :- file_property(File,mod_time(Time)).

% get modification time of a file:
fileSize(File,Size) :- file_property(File,size(Size)).

% does a file exist and is a regular file?
existsFile(File) :- file_exists(File), file_property(File,type(regular)).

% does a directory exist?
existsDirectory(Dir) :- file_exists(Dir), file_property(Dir,type(directory)).

% remove a file from the file system:
deleteFile(File) :- delete_file(File,[]).

% fileExistsAndNewer(f1,f2) is true if file f1 exists and is newer than f2:
fileExistsAndNewer(File1,File2) :-
	file_exists(File1),
	fileModTime(File1,MT1),
	fileModTime(File2,MT2),
	MT1>=MT2.

% can I write a file (i.e., write and immediately close it)?
canWriteFile(File) :-
	on_exception(_ErrorMsg,
	             (open(File,write,Stream), close(Stream)),
		     fail).

% process of identifer of current Prolog process:
currentPID(Pid) :- pid(Pid).

% get current time (year,month,day,hour,minute,second):
currentTime(Year,Month,Day,Hour,Minute,Second) :-
	datime(datime(Year,Month,Day,Hour,Minute,Second)).

% get name of current host:
getHostname(Name) :- host_name(Name).

% execute a shell command in background and return the input, output, and
% error stream connected to this command (if the corresponding stream argument
% is not already instantiated to 'std'):
execCommand(Cmd,InWrite,OutRead,ErrRead) :-
	(var(ErrRead) -> ErrReadArg=pipe(ErrRead) ; ErrReadArg=ErrRead),
	exec(Cmd,[pipe(InWrite),pipe(OutRead),ErrReadArg],_).


% fork the current program state with an initial goal to execute
% (without producing any output):
:- meta_predicate forkProcessForGoal(:).
forkProcessForGoal(Goal) :-
	currentPID(PID),
	number_codes(PID,PIDS),
	append("/tmp/pakcs_fork_",PIDS,StateP),
	append(StateP,".state",StateS),
	atom_codes(StateName,StateS),
        saveprog_entry(StateName,Goal),
	append("(",StateS,FC0),
	append(FC0," ; rm -f ", FC1),
	append(FC1,StateS,FC2),
	append(FC2,") > /dev/null 2> /dev/null &",ForkCmdS),
	atom_codes(ForkCmd,ForkCmdS),
	system(ForkCmd).


% is a stream a readable stream?
isInputStream(Stream) :-
	sicstus38orHigher -> stream_property(Stream,input)
	                   ; current_stream(_,input,Stream).

% is a stream a writable stream?
isOutputStream(Stream) :-
	sicstus38orHigher -> stream_property(Stream,output)
	                   ; current_stream(_,output,Stream).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Interface to sockets.

% Create a new socket structure.
newSocket(Socket) :- socket('AF_INET',Socket).

% Create a connection to a socket and return instream and outstream:
socketConnect(Socket,Host,Port,Stream,Stream) :-
	socket_connect(Socket,'AF_INET'(Host,Port),Stream).

% close socket stream connection:
closeSocketStream(InStream,_OutStream) :- close(InStream).

% Bind a socket to a port (return hostname and, possibly, port number):
socketBind(Socket,Hostname,Port) :-
	socket_bind(Socket,'AF_INET'(Hostname,Port)).

% Define backlog on socket:
socketListen(Socket,N) :- socket_listen(Socket,N).

% return the read and write stream of a first connection to a socket:
socketAccept(Socket,Client,Stream,Stream) :-
        socket_accept(Socket,Client,Stream).

% Wait for input data on a list of streams (Timeout = TOSec:TOMSec or off),
% returns -1 if no data available within Timeout limit, otherwise the index of
% the corresponding stream:
waitForInputDataOnStreams(InStreams,Timeout,Index) :-
	socket_select([],_,_,Timeout,InStreams,SelStreams), !,
	(SelStreams = [] -> Index = -1
                          ; SelStreams = [Stream|_],
                            streamIndex(InStreams,Stream,Index)).

streamIndex([S|_],S,0) :- !.
streamIndex([_|Streams],S,I) :- streamIndex(Streams,S,I1), I is I1+1.


% Wait for a client connection at a socket (Timeout = TOSec:TOMSec or off),
% fails if no client connection available within Timeout limit.
waitForSocketClientStream(Socket,Timeout,Client,InStream,OutStream) :-
	socket_select([Socket],PortStreams,Clients,Timeout,[],_), !,
	PortStreams=[InStream], Clients=[Client], OutStream = InStream.


% Wait for a client connection at a socket or available stream input data.
% If a client connection is established, InPortStream and OutPortStream
% are instantiated to the client stream connection and Client is
% instantiated to the clients address, otherwise Client is instantiated
% to 'no' and the last argument is instantiated to the InStreams index with
% available data.
%
% Note that this implementation works only with
% Sicstus-Prolog 3.8.5 or higher (due to a bug in previous versions
% of Sicstus-Prolog).
waitForSocketOrInputStreams(Socket,Client,InPortStream,OutPortStream,InStreams,Index) :-
	sicstus385orHigher, !,
	socket_select([Socket],PortStreams,Clients,off,InStreams,SelStreams),
	(PortStreams=[InPortStream]
	 -> Clients=[Client],
	    OutPortStream = InPortStream
	  ; Client=no,
	    SelStreams = [Stream|_],
	    streamIndex(InStreams,Stream,Index)).
waitForSocketOrInputStreams(_,_,_,_,_,_) :-
	raise_exception('You need Sicstus-Prolog 3.8.5 or higher to execute this program (due to a bug in the socket library of previous versions of Sicstus)').


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Due to incompatibilities between the various versions of
% Sicstus-Prolog, we have to use different methods to save states etc.

:- meta_predicate saveprog_entry(?,:).
saveprog_entry(State,Entry) :-
	pakcsrc(standalone,yes), !,
	save_standalone_executable(State,Entry).
saveprog_entry(State,Entry) :-
	getSicstusVersion(SV), SV = '3.5',
	!, % if we use Sicstus 3.5
	save(State), Entry.
saveprog_entry(State,Entry) :-
	sicstus37orHigher, !,
	save_program(State,Entry).   % if we use Sicstus 3.7 or higher
% Note for Sicstus 3.9 and higher:
% It works only if the name of the state contains a suffix with a dot,
% like 'xxx.state', otherwise Sicstus adds automatically
% the suffix '.sav'!

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% for stand-alone executables:
:- dynamic rt_entry/1.

% use this instead of saveprog_entry:
save_standalone_executable(State,Entry) :-
	(sicstus310orHigher -> true
	 ; write(user_error,'ERROR: stand-alone executables require Sictus 3.10 or higher!'),
	   nl(user_error), fail),
	(retract(rt_entry(_)) -> true ; true),
	asserta(rt_entry(Entry)),
	save_program(State).

user:runtime_entry(start) :-
	rt_entry(Entry),
	call(Entry).


% try to save a user predicate in a .po file if it is supported by this
% Sicstus version:
try_save_predicates(P/N,POFile) :-
	sicstus38orHigher, !,
	save_predicates(user:P/N,POFile).
try_save_predicates(_,_).


% try to save an already compiled Prolog program in a .po file if it is supported by this
% Sicstus version:
try_save_program(PrologFileName) :-
	sicstus38orHigher, !,
	atom_codes(PrologFileName,PrologFileNameS),
	append(FileNameS,".pl",PrologFileNameS),
	append(FileNameS,".po",POFileNameS),
	atom_codes(POFileName,POFileNameS),
	((append("/tmp/",_,PrologFileNameS) ; \+ canWriteFile(POFileName)) -> true
	 ; save_files(PrologFileName,POFileName)).
try_save_program(_).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Auxiliaries for compiling programs and loading run-time libraries:

% compile a Prolog file:
compilePrologFile(PrologFileName) :- compile(user:PrologFileName).

% compile a Prolog file and try to save it in fast load format:
compilePrologFileAndSave(PrologFileName) :-
	atom_codes(PrologFileName,PrologFileNameS),
	append(FileNameS,".pl",PrologFileNameS),
	append(FileNameS,".po",POFileNameS),
	atom_codes(POFileName,POFileNameS),
	(fileExistsAndNewer(POFileName,PrologFileName)
	 -> load_files(user:POFileName) % for faster compilation
	  ; compilePrologFile(PrologFileName),
	    try_save_program(PrologFileName)).

% consult a Prolog file or a .po file if it exists
consultPrologorPOFile(PrologFileName,POFileName) :-
	(fileExistsAndNewer(POFileName,PrologFileName)
	 -> load_files(user:POFileName) % for faster compilation
	  ; consult(user:PrologFileName)).


% directory containing the system run-time modules:
moduleDir(MD) :-
        environ('PAKCSHOME',TCP),
        appendAtom(TCP,'/curry2prolog/lib_src/',MD).

% ensure that run-time library is loaded:
ensure_lib_loaded(Lib) :- % first, look into working directory:
	working_directory(WDir,WDir),
	appendAtom(WDir,'/',Dir),
	appendAtom(Dir,Lib,DirLib),
	appendAtom(DirLib,'.pl',DirLibPl),
	file_exists(DirLibPl), !,
	ensure_loaded(user:DirLib).
ensure_lib_loaded(Lib) :-
	moduleDir(Dir),
	appendAtom(Dir,Lib,DirLib),
	ensure_loaded(user:DirLib).


% get name of temporary Prolog file:
getNewPrologFileName(PrologFile) :-
	pid(PID),
	number_codes(PID,PIDS),
	append("/tmp/pakcsprog",PIDS,P1), append(P1,".pl",ProgS),
	atom_codes(PrologFile,ProgS),
	append("rm -f ",ProgS,RmCmdS),
	atom_codes(RmCmd,RmCmdS),
	system(RmCmd).


% determine for a given Prolog file name (of the main module) a file name
% where the clauses for the main predicates (hnf, constrEq,...) should be stored:
mainPrologFileName(_PrologFile,MainPrologFile) :-
	getNewPrologFileName(NewPrologFile),
	appendAtom(NewPrologFile,'.main',MainPrologFile).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% call a goal and return list of suspended goals:
:- meta_predicate callAndReturnSuspensions(:,?).
callAndReturnSuspensions(Goal,Suspensions) :- call_residue(Goal,Suspensions).

% write a Prolog term possibly containing variables:
writeqWithVars(T) :- \+ \+ (numbervars(T,0,_), writeq(T)).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Write a block declaration for predicate PredName of arity PredArity
% where the positions in the non-empty(!) list GroundPositions must be instantiated.
% Furthermore, the last argument is a possibly new predicate name corresponding
% to PredName which should be coded instead of PredName (this depends on
% the implementation scheme for block declarations).
genBlockDecl(PredName,PredArity,BoundPositions,PredName) :-
	map2partial(genBlockLiteral(PredName,PredArity),BoundPositions,Literals),
	basics:foldr1(',',Literals,LiteralsGoal),
	compiler:writeClause((:- block(LiteralsGoal))).

genBlockLiteral(PredName,PredArity,BlockPos,Literal) :-
	functor(Literal,PredName,PredArity),
	setBlockArgs(PredArity,BlockPos,Literal).
setBlockArgs(0,_,_) :- !.
setBlockArgs(I,P,Literal) :-
	arg(I,Literal,Arg),
	(I=P -> Arg='-' ; Arg='?'),
	I1 is I-1, setBlockArgs(I1,P,Literal).


