%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% basic predicates related to the SWI-Prolog system

:- module(prologbasics,
	  [prolog/1, prologMajorVersion/1, prologMinorVersion/1, pakcsrc/2,
	   sicstus37orHigher/0, sicstus38orHigher/0, sicstus310orHigher/0,
	   unifyWithOccursCheck/2,
	   waitConcurrentConjunction/6,
	   append/3, appendAtom/3, map1/2, map2/3, map1partial/2, map2partial/3,
	   getProgramArgs/1,
	   noSingletonWarnings/0, noRedefineWarnings/0, noDiscontiguousWarnings/0,
	   getRunTime/1, getElapsedTime/1, garbageCollectorOn/0, garbageCollectorOff/0,
	   fileModTime/2, fileSize/2, existsFile/1, existsDirectory/1,
	   deleteFile/1,
	   fileExistsAndNewer/2, canWriteFile/1, currentPID/1, currentTime/6,
	   getHostname/1, execCommand/4, forkProcessForGoal/1,
	   isInputStream/1, isOutputStream/1,
	   newSocket/1, socketConnect/5, closeSocketStream/2,
	   socketBind/3, socketListen/2, socketAccept/4,
	   waitForInputDataOnStreams/3, waitForSocketClientStream/5,
	   waitForSocketOrInputStreams/6,
	   try_save_program/1, saveprog_entry/2, runtime_entry/0,
	   try_save_predicates/2,
	   compilePrologFile/1, compilePrologFileAndSave/1,
	   consultPrologorPOFile/2, ensure_lib_loaded/1,
	   getNewPrologFileName/1, mainPrologFileName/2,
	   callAndReturnSuspensions/2, writeqWithVars/1,
	   genBlockDecl/4,
	   environ/2, prolog_flag/2, prolog_flag/3, system/1, system/2,
	   create_mutable/2, get_mutable/2, update_mutable/2]).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Principle kind of Prolog system and version used for this implementation.
prolog(swi).

prologMajorVersion(MV) :-
	current_prolog_flag(version,VN),
	MV is VN//10000.

prologMinorVersion(MV) :-
	current_prolog_flag(version,VN),
	MV is (VN mod 10000)//100.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
:- multifile pakcsrc/2. % relevant for createSavedState

:- dynamic pakcsrc/2.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

unifyWithOccursCheck(T1,T2) :- unify_with_occurs_check(T1,T2).

append([],Xs,Xs).
append([X|Xs],Ys,[X|Zs]) :- append(Xs,Ys,Zs).

% concatenate two atoms:
appendAtom(A1,A2,A3) :-
	atom_codes(A1,L1), atom_codes(A2,L2),
	append(L1,L2,L3),
	atom_codes(A3,L3).

sicstus37orHigher :- fail.

sicstus38orHigher :- fail.

sicstus310orHigher :- fail.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Implementation of suspension for concurrent conjunction (&)

waitConcurrentConjunction(S1,S2,R,E1,E2,E) :-
	when((nonvar(E1) ; nonvar(E2)),
	     waitConcurrentConjunctionBlocked(S1,S2,R,E1,E2,E)).
waitConcurrentConjunctionBlocked(S1,S2,R,E1,E2,E) :- nonvar(E1), !,
	(S1='FAIL'(_) -> R=S1, E=E1 ; waitForEval(S2,R,E2,E)).
waitConcurrentConjunctionBlocked(S1,S2,R,E1,E2,E) :- % E2 must be nonvar
	(S2='FAIL'(_) -> R=S2, E=E2 ; waitForEval(S1,R,E1,E)).

waitForEval(R0,R,E0,E) :- freeze(E0,(R0=R, E0=E)).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% meta-predicates for higher-order programming:

% map a unary predicate on a list:
:- meta_predicate map1(:,?).
map1(_,[]).
map1(P,[X|Xs]) :- C =.. [P,X], call(C), map1(P,Xs).

% map a binary predicate on two lists:
:- meta_predicate map2(:,?,?).
map2(_,[],[]).
map2(P,[X|Xs],[Y|Ys]) :- C =.. [P,X,Y], call(C), map2(P,Xs,Ys).

% map a unary predicate (which might be a partial application) on a list:
:- meta_predicate map1partial(:,?).
map1partial(_,[]).
map1partial(P,[X|Xs]) :-
	P =.. [Pred|PartialArgs],
	append(PartialArgs,[X],Args),
	C =.. [Pred|Args], call(C),
	map1partial(P,Xs).

% map a binary predicate (which might be a partial application) on two lists:
:- meta_predicate map2partial(:,?,?).
map2partial(_,[],[]).
map2partial(P,[X|Xs],[Y|Ys]) :-
	P =.. [Pred|PartialArgs],
	append(PartialArgs,[X,Y],Args),
	C =.. [Pred|Args], call(C),
	map2partial(P,Xs,Ys).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% get program arguments:
getProgramArgs(Args) :-
	current_prolog_flag(argv,[_|AllArgs]),
	(AllArgs=['-x',_,'--'|Args] -> true ; Args=AllArgs).

% no warnings for singleton variables:
noSingletonWarnings :- style_check(-singleton).

% turn off discontiguous clauses warnings:
noDiscontiguousWarnings :- style_check(-discontiguous).

% no warnings for redefining predicates:
noRedefineWarnings :- true. % TODO, no solution yet

% get current run time in msecs:
getRunTime(Time) :- statistics(runtime,[Time,_]).

% get current elapsed time in msecs:
getElapsedTime(Time) :-
	statistics(runtime,[RTime,_]), statistics(system_time,[STime,_]),
	Time is RTime+STime.

% turn on garbage collector:
garbageCollectorOn :- set_prolog_flag(gc,true).
% turn off garbage collector:
garbageCollectorOff :- set_prolog_flag(gc,false).

% get modification time of a file:
fileModTime(File,Time) :- time_file(File,FTime), Time is float_integer_part(FTime).

% get modification time of a file:
fileSize(File,Size) :- size_file(File,Size).

% does a file exist?
existsFile(File) :- exists_file(File).

% does a file exist?
existsDirectory(Dir) :- exists_directory(Dir).

% remove a file from the file system:
deleteFile(File) :- delete_file(File).

% fileExistsAndNewer(f1,f2) is true if file f1 exists and is newer than f2:
fileExistsAndNewer(File1,File2) :-
	existsFile(File1),
	fileModTime(File1,MT1),
	fileModTime(File2,MT2),
	MT1>=MT2.

% can I write a file (i.e., write and immediately close it)?
canWriteFile(File) :-
	on_exception(_ErrorMsg,
	             (open(File,write,Stream), close(Stream)),
		     fail).

% process of identifer of current Prolog process:
currentPID(Pid) :- current_prolog_flag(pid,Pid).

% get current time (year,month,day,hour,minute,second):
currentTime(Year,Month,Day,Hour,Minute,Second) :-
	get_time(T), convert_time(T,Year,Month,Day,Hour,Minute,Second,_).

% get name of current host:
getHostname(Name) :- gethostname(Name).

% execute a shell command in background and return the input, output, and
% error stream connected to this command:
execCommand(Cmd,InWrite,OutRead,ErrRead) :-
	pipe(InRead,InWrite),
	pipe(OutRead,OutWrite),
	(ErrRead==std -> true ; pipe(ErrRead,ErrWrite)),
	fork(Pid),
	(Pid=child
	 -> close(InWrite), dup(InRead,user_input),
	    close(OutRead), dup(OutWrite,user_output),
	    (ErrRead==std -> true ; close(ErrRead), dup(ErrWrite,user_error)),
	    exec(sh('-c',Cmd))
	  ; close(InRead),
	    close(OutWrite),
	    (ErrRead==std -> true ; close(ErrWrite))).


% fork the current program state with an initial goal to execute
% (without producing any output):
:- meta_predicate forkProcessForGoal(:).
forkProcessForGoal(Goal) :-
	fork(Pid),
	(Pid=child
	 -> open('/dev/null',write,Null),
	    dup(Null,user_output), dup(Null,user_error),
	    call(Goal)
	  ; true).


% is a stream a readable stream?
isInputStream(Stream) :- stream_property(Stream,input).

% is a stream a writable stream?
isOutputStream(Stream) :- stream_property(Stream,output).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Interface to sockets.

% Create a new socket structure.
newSocket(Socket) :- tcp_socket(Socket).

% Create a connection to a socket and return instream and outstream:
socketConnect(Socket,Host,Port,InStream,OutStream) :-
	tcp_connect(Socket,Host:Port),
	tcp_open_socket(Socket,InStream,OutStream).

% close socket stream connection:
closeSocketStream(InStream,OutStream) :- close(InStream), close(OutStream).

% Bind a socket to a port (return hostname and, possibly, port number):
socketBind(Socket,Hostname,Port) :-
	tcp_bind(Socket,Port), gethostname(Hostname).

% Define backlog on socket:
socketListen(Socket,N) :- tcp_listen(Socket,N).

% return the read and write stream of a first connection to a socket:
socketAccept(Socket,Client,InStream,OutStream) :-
        tcp_open_socket(Socket,AcceptStream,_),
	tcp_accept(AcceptStream,SocketConnection,ClientIP),
	tcp_host_to_address(Client,ClientIP),
	tcp_open_socket(SocketConnection,InStream,OutStream).

% Wait for input data on a list of streams (Timeout = TOSec:TOMSec or off),
% returns -1 if no data available within Timeout limit, otherwise the index of
% the corresponding stream:
waitForInputDataOnStreams(InStreams,Timeout,Index) :-
	timeoutAsSWI(Timeout,TimeoutSWI),
	wait_for_input(InStreams,ReadyStreams,TimeoutSWI), !,
	(ReadyStreams = [] -> Index = -1
                            ; ReadyStreams = [Stream|_],
                              streamIndex(InStreams,Stream,Index)).

streamIndex([S|_],S,0) :- !.
streamIndex([_|Streams],S,I) :- streamIndex(Streams,S,I1), I is I1+1.

% Wait for a client connection at a socket (Timeout = TOSec:TOMSec or off),
% fails if no data available within Timeout limit
waitForSocketClientStream(Socket,Timeout,Client,InStream,OutStream) :-
	timeoutAsSWI(Timeout,TimeoutSWI),
	tcp_open_socket(Socket,AcceptStream,_),
	wait_for_input([AcceptStream],ReadyStreams,TimeoutSWI), !,
	ReadyStreams = [AcceptStream],
	tcp_accept(AcceptStream,Slave,Client),
	tcp_open_socket(Slave,InStream,OutStream).

% Wait for a client connection at a socket or available stream input data.
% If a client connection is established, InPortStream and OutPortStream
% are instantiated to the client stream connection and Client is
% instantiated to the clients address, otherwise Client is instantiated
% to 'no' and the last argument is instantiated to the InStreams index with
% available data.
waitForSocketOrInputStreams(Socket,Client,InPortStream,OutPortStream,InStreams,Index) :-
	tcp_open_socket(Socket,AcceptStream,_),
	wait_for_input([AcceptStream|InStreams],ReadyStreams,infinite), !,
	(member(AcceptStream,ReadyStreams)
	 -> tcp_accept(AcceptStream,Slave,Client),
	    tcp_open_socket(Slave,InPortStream,OutPortStream)
	  ; Client=no,
	    ReadyStreams = [Stream|_],
	    streamIndex(InStreams,Stream,Index)).

timeoutAsSWI(off,infinite) :- !.
timeoutAsSWI(Secs:MSecs,S) :- S is Secs+(MSecs/1000).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Saving program states

:- meta_predicate saveprog_entry(?,:).
saveprog_entry(State,Entry) :-
	pakcsrc(standalone,yes), !,
	(retract(rt_entry(_)) -> true ; true),
	asserta(rt_entry(Entry)),
	qsave_program(State,[toplevel(runtime_entry),stand_alone(true)]).
saveprog_entry(State,Entry) :-
	(retract(rt_entry(_)) -> true ; true),
	asserta(rt_entry(Entry)),
	%export(user:functiontype(_,_,_,_,_,_)),
	qsave_program(State,[toplevel(runtime_entry)]).

:- dynamic user:rt_entry/1.

runtime_entry :-
	user:rt_entry(Entry),
	call(Entry).


% try to save a user predicate in a .po file if it is supported by this
% Sicstus version:
try_save_predicates(_,_).


% try to save an already compiled Prolog program in a .po file if it is supported by this
% Sicstus version:
try_save_program(_).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Auxiliaries for compiling programs and loading run-time libraries:

% compile a Prolog file:
compilePrologFile(PrologFileName) :- compile(user:PrologFileName).

% compile a Prolog file and try to save it in fast load format:
compilePrologFileAndSave(PrologFileName) :-
	compilePrologFile(PrologFileName), try_save_program(PrologFileName).

% consult a Prolog file or a .po file if it exists (not in SWI)
consultPrologorPOFile(PrologFileName,_POFileName) :-
	consult(user:PrologFileName).


% directory containing the system run-time modules:
moduleDir(MD) :-
        environ('PAKCSHOME',TCP),
        appendAtom(TCP,'/curry2prolog/libswi/',MD).

% ensure that run-time library is loaded:
ensure_lib_loaded(Lib) :- % first, look into working directory:
	working_directory(WDir,WDir),
	appendAtom(WDir,'/',Dir),
	appendAtom(Dir,Lib,DirLib),
	appendAtom(DirLib,'.pl',DirLibPl),
	existsFile(DirLibPl), !,
	ensure_loaded(user:DirLib).
ensure_lib_loaded(Lib) :-
	moduleDir(Dir),
	appendAtom(Dir,Lib,DirLib),
	ensure_loaded(user:DirLib).


% get name of temporary Prolog file:
getNewPrologFileName(PrologFile) :-
	tmp_file(curry_main,Main),
	atom_codes(Main,MainS),
	append(MainS,".pl",ProgS),
	atom_codes(PrologFile,ProgS),
	append("rm -f ",ProgS,RmCmdS),
	atom_codes(RmCmd,RmCmdS),
	system(RmCmd).


% determine for a given Prolog file name (of the main module) a file name
% where the clauses for the main predicates (hnf, constrEq,...) should be stored:
mainPrologFileName(_PrologFile,MainPrologFile) :-
	mainPrologFileName(MainPrologFile) -> true
	 ; tmp_file(pakcs_main,MainPrologFile),
	   assertz(mainPrologFileName(MainPrologFile)).

% for storing the file name during the Prolog session:
:- dynamic mainPrologFileName/1.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% call a goal and return list of suspended goals (here always empty):
:- meta_predicate callAndReturnSuspensions(:,?).
callAndReturnSuspensions(Goal,Suspensions) :- call(Goal), Suspensions=[].

% write a Prolog term possibly containing variables:
writeqWithVars(T) :- writeq(T).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Write a block declaration for predicate PredName of arity PredArity
% where the positions in the non-empty(!) list GroundPositions must be instantiated.
% Furthermore, the last argument is a possibly new predicate name corresponding
% to PredName which should be coded instead of PredName (this depends on
% the implementation scheme for block declarations).
genBlockDecl(PredName,PredArity,BoundPositions,NewPredName) :-
	appendAtom('blocked_',PredName,NewPredName),
	functor(Literal,PredName,PredArity),
	Literal =.. [_|Args],
	NewLiteral =.. [NewPredName|Args],
	genFreezeLiteral(BoundPositions,NewLiteral,NewLiteral,FreezeLiteral),
	compiler:writeClause((Literal :- FreezeLiteral)).

genFreezeLiteral([],_,Literal,Literal) :- !.
genFreezeLiteral([P|Ps],Literal,FreezeLiteral,NewFreezeLiteral) :-
	arg(P,Literal,Var),
	genFreezeLiteral(Ps,Literal,freeze(Var,FreezeLiteral),NewFreezeLiteral).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% mutable values (simulated by open-ended lists)

create_mutable(V,'$mutable'([V|_])).

get_mutable(V,'$mutable'(L)) :- get_mutable_aux(V,L).
get_mutable_aux(V,[H|T]) :- var(T) -> V=H ; get_mutable_aux(V,T).

update_mutable(V,'$mutable'(L)) :- update_mutable_aux(V,L).
update_mutable_aux(V,[_|T]) :- var(T) -> T=[V|_] ; update_mutable_aux(V,T).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

environ(Var,Val) :- getenv(Var,Val), !.

prolog_flag(user_input,user_input) :- !.
prolog_flag(user_output,user_output) :- !.
prolog_flag(user_error,user_error) :- !.
prolog_flag(F,_) :- write('Warning: unknown prolog_flag: '), write(F), nl.

prolog_flag(F,V,V) :- write('Warning: unknown prolog_flag: '), write(F), nl.

system(C) :- shell(C,0).

system(C,S) :- shell(C,S).
