--------------------------------------------------------------------------------
--- This module contains functions to obtain information concerning the current
--- distribution of the Curry implementation.
--- Most of the information is based on the external constants <b>curryCompiler...</b>.
---
--- @author Bernd Brassel, Michael Hanus
--- @version August 2006
--------------------------------------------------------------------------------

module Distribution (
  curryCompiler,curryCompilerMajorVersion,curryCompilerMinorVersion,
  curryRuntime,curryRuntimeMajorVersion,curryRuntimeMinorVersion,
  
  rcFileName,rcFileContents,getRcVar,getRcVars,

  findFileInLoadPath,readFirstFileInLoadPath,getLoadPath,getLoadPathForFile,

  FrontendTarget(..), FrontendParam(..), callFrontendWithParams
  ) where

import List(intersperse)
import System
import IO
import Directory
import FileGoodies
import PropertyFile

-----------------------------------------------------------------
-- Compiler and run-time environment name and version
-----------------------------------------------------------------

-- if you do not use other functions but 
-- if-then-else, and the _Prelude_ functions
-- (<), (>), (<=), (>=), (==)
-- directly on the following constants, 
-- the compiler might be able to eliminate
-- them at compile time.

--- The name of the Curry compiler (e.g., "pakcs" or "kics").
curryCompiler :: String
curryCompiler external

--- The major version number of the Curry compiler.
curryCompilerMajorVersion :: Int
curryCompilerMajorVersion external

--- The minor version number of the Curry compiler.
curryCompilerMinorVersion :: Int
curryCompilerMinorVersion external

--- The name of the run-time environment (e.g., "sicstus", "swi", or "ghc")
curryRuntime :: String
curryRuntime external

--- The major version number of the Curry run-time environment.
curryRuntimeMajorVersion :: Int
curryRuntimeMajorVersion external

--- The minor version number of the Curry run-time environment.
curryRuntimeMinorVersion :: Int
curryRuntimeMinorVersion external

---------------------------------------------------
-- retrieving user specified options from rc file
---------------------------------------------------

--- The name of the file specifying configuration parameters of the
--- current distribution. This file must have the usual format of
--- property files (see description in module PropertyFile).
rcFileName :: IO String
rcFileName = getEnviron "HOME" >>= return . (++"/."++curryCompiler++"rc")

--- Returns the current configuration parameters of the distribution.
--- This action yields the list of pairs (var,val).
rcFileContents :: IO [(String,String)]
rcFileContents = rcFileName >>= readPropertyFile

--- Look up a specific configuration variable as specified by user in his rc file.
getRcVar :: String -> IO (Maybe String)
getRcVar var = getRcVars [var] >>= return . head

--- Look up configuration variables as specified by user in his rc file.
getRcVars :: [String] -> IO [Maybe String]
getRcVars vars = do
  rcs <- rcFileContents 
  return (map (flip lookup rcs) vars)


-----------------------------------------------------------
--- finding files in correspondence to compiler load path
-----------------------------------------------------------

--- Adds a directory name to a file by looking up the current load path.
--- An error message is delivered if there is no such file.
findFileInLoadPath :: String -> IO String
findFileInLoadPath fn = 
  getLoadPathForFile fn >>= getFileInPath (baseName fn) [""] 

--- Returns the contents of the file found first in the current load path.
--- An error message is delivered if there is no such file.
readFirstFileInLoadPath :: String -> IO String
readFirstFileInLoadPath fn = findFileInLoadPath fn >>= readFile 

--- Returns the current path (list of directory names) that is
--- used for loading modules.
getLoadPath :: IO [String]
getLoadPath = getLoadPathForFile "xxx"

--- Returns the current path (list of directory names) that is
--- used for loading modules w.r.t. a given main module file.
--- The directory prefix of the module file (or "." if there is
--- no such prefix) is the first element of the load path and the
--- remaining elements are determined by the environment variables
--- CURRYRPATH, PAKCSHOME, and PAKCSLIBPATH when using pakcs resp.
--- the entry of kicsrc when using kics.
getLoadPathForFile :: String -> IO [String]
getLoadPathForFile modfile =
  let fileDir = dirName modfile in
  if curryCompiler == "pakcs" then
    do pakcshome <- getEnviron "PAKCSHOME"
       pakcspath <- getEnviron "PAKCSLIBPATH"
       currypath <- getEnviron "CURRYPATH"
       return $ fileDir :
              (if null currypath then [] else splitPath currypath) ++
              (if null pakcspath
               then if null pakcshome
                    then []
                    else [pakcshome++"/lib",pakcshome++"/lib/meta"]
               else splitPath pakcspath)
  else if curryCompiler == "kics" then
    getRcVar "Libraries" >>=
    maybe (rcErr "library path" []) (return . (fileDir:) . splitPath)
  else error "Distribution:getLoadPathForFile: unknown curryCompiler"


-------------------------------------------------------------------
-- calling the front end
-------------------------------------------------------------------

--- Data type for representing the different target files that can be produced
--- by the front end of the Curry compiler.
--- @cons FCY - FlatCurry file ending with .fcy
--- @cons FINT - FlatCurry interface file ending with .fint
--- @cons ACY - AbstractCurry file ending with .acy
--- @cons UACY - Untyped (without type checking) AbstractCurry file ending with .uacy
--- @cons HTML - colored HTML representation of source program
data FrontendTarget = FCY | FINT | ACY | UACY | HTML

--- Data type for representing parameters supported by the front end
--- of the Curry compiler.
--- @cons Quiet - work silently
--- @cons FullPath dirs - the complete list of directory names for loading modules
--- @cons OutFile file - output file (currently, only relevant for HTML target)
--- @cons LogFile file - store all output (including errors) of the front end in file
data FrontendParam = Quiet | FullPath [String] | OutFile String | LogFile String

--- In order to make sure that compiler generated files (like .fcy, .fint, .acy)
--- are up to date, one can call the front end of the Curry compiler with this action.
--- @param target - the kind of target file to be generated
--- @param params - parameters for the front end
--- @param progname - the name of the main module of the application to be compiled
callFrontendWithParams :: FrontendTarget -> [FrontendParam] -> String -> IO ()
callFrontendWithParams target params progname = do
  parsecurry <- callParseCurry
  let logfile = logFileOfParams params
      syscall = parsecurry ++ " " ++ showFrontendTarget target
                     ++ concatMap showFrontendParam params
                     ++ " " ++ progname
  if null logfile
    then system syscall
    else system (syscall++" > "++logfile++" 2>&1")
  return ()
 where
   callParseCurry =
     if curryCompiler == "pakcs" then
      do pakcshome <- getEnviron "PAKCSHOME"
         if null pakcshome
          then error "Environment variable PAKCSHOME undefined: cannot start front end"
          else return (pakcshome++"/bin/parsecurry")
     else if curryCompiler == "kics" then
      do [mcall,mlibs] <- getRcVars ["FrontendCall","Libraries"]
         call <- maybe (rcErr "frontend call" "") return mcall
         libs <- maybe (rcErr "library paths" "") return mlibs
         return ("PAKCSLIBPATH="++libs++" "++call)
     else error "Distribution:callFrontendWithParams: unknown curryCompiler"

   showFrontendTarget FCY  = "-fcy"
   showFrontendTarget FINT = "-fcy"
   showFrontendTarget ACY  = "-acy"
   showFrontendTarget UACY = "-uacy"
   showFrontendTarget HTML = "-html"

   showFrontendParam Quiet = " -quiet"
   showFrontendParam (FullPath path) = " -fullpath " ++ concat (intersperse ":" path)
   showFrontendParam (OutFile f) = " -o " ++ f
   showFrontendParam (LogFile _) = ""

   logFileOfParams [] = ""
   logFileOfParams (LogFile f : _) = f
   logFileOfParams (OutFile _  : ps) = logFileOfParams ps
   logFileOfParams (Quiet      : ps) = logFileOfParams ps
   logFileOfParams (FullPath _ : ps) = logFileOfParams ps

rcErr :: String -> a -> IO a
rcErr s x = hPutStr stderr (s ++ " undefined in rc file") >> return x
