------------------------------------------------------------------------------
--- A HtmlServer manages the event handlers used in
--- dynamic web pages on the server side.
--- Each event handler is encoded as a string and stored on the server side
--- with a unique key. Only this key is sent in the actual web page
--- to the client.
--- Event handlers are only valid for a particular time period
--- specified by <code>eventHandlerExpiration</code>, i.e., after that time
--- event handlers will be deleted.

module HtmlServer(storeEnvHandlersWithCgiKey,retrieveEnvHandlersWithCgiKey)
  where

import Ports
import ReadNumeric(readNat)
import ReadShowTerm(readsQTerm)
import Unsafe(readAnyQTerm,showAnyQTerm)
import Time
import System

--- Should each event handlers be used only once, i.e., deleted after each use?
--- If this is true, one cannot use the back button in the server and submit
--- the same form again (which is usually a reasonable behavior to avoid
--- double submissions of data). If this is false, the event server
--- might overflow with unused events (which are usually deleted only
--- after the timout period specified by eventHandlerExpiration below).
singleUseEventHandlers = True

--- Stores a list of new event handlers for a given cgi program and
--- the corresponding arguments with a new key.
storeEnvHandlersWithCgiKey :: String -> [(String,String)] -> [(a,String)]
                              -> IO ()
storeEnvHandlersWithCgiKey cgikey env handlerkeys = do
  time <- getClockTime
  let (msgs,unifs) = generateEventServerMessages
                              (toDate (eventHandlerExpiration time))
                               handlerkeys
  sendEventServerMessages (CleanOldEventHandlers:msgs)
  doSolve (foldr (&) success unifs)
 where
   generateEventServerMessages _ [] = ([],[])
   generateEventServerMessages expiredate ((handlerstring,newkey):evhs) =
     let (msgs,unifs) = generateEventServerMessages expiredate evhs
      in (StoreNewEnvEventWithCgiKey key expiredate cgikey env
                                     (showAnyQTerm handlerstring)  : msgs,
          (show key ++ ' ':show expiredate =:= newkey) : unifs)
    where key free

-- Retrieves a previously stored event handler for a cgi program.
-- Returns Nothing if the handler is no longer available, i.e., expired.
retrieveEnvHandlersWithCgiKey :: String -> String
                                 -> IO (Maybe ([(String,String)],a))
retrieveEnvHandlersWithCgiKey cgikey key = 
  let (numstring,datestring) = break (==' ') key
      dateps = readsQTerm datestring
      num    = maybe (-1) (\(i,s)->if null s then i else -1) (readNat numstring)
   in if null datestring || null dateps || num < 0
      then return Nothing
      else catchFail
             (sendEventServerMessages
                   [GetEnvEventWithCgiKey num (fst (head dateps)) cgikey info] >>
              return (decode info))
             (return Nothing)
 where
   info free

   decode Nothing = Nothing
   decode (Just (env,str)) = Just (env, readAnyQTerm str)

-- Define for a given date a new date when the event handler expires.
eventHandlerExpiration :: ClockTime -> ClockTime
eventHandlerExpiration = addHours 1
--eventHandlerExpiration = addMinutes 1

-- Define for a given date a new date when the next cleanup of event handlers
-- should be done.
nextCleanup :: ClockTime -> ClockTime
nextCleanup = addMinutes 5

---------------------------------------------------------------------------
-- The implementation of the event handler server via ports:

type Date = (Int,Int,Int,Int,Int,Int)

-- Messages for event handler server:
data EHSMsg =
   StoreNewEnvEventWithCgiKey Int Date String [(String,String)] String
 | GetEnvEventWithCgiKey Int Date String (Maybe ([(String,String)],String))
 | CleanOldEventHandlers
 | ShowAllHandlers String
 | SketchAllHandlers Int String
 | Stop

-- Start the event handler server via script:
startServer = do
  pakcshome <- getEnviron "PAKCSHOME"
  system (pakcshome++"/www/start")

-- Stop the event handler server (usually, not required):
stopServer = sendEventServerMessages [Stop]

-- show all currently stored handlers:
showAllHandlers = do
  sendEventServerMessages [ShowAllHandlers s]
  putStr s
 where s free

-- sketch all currently stored handlers up to a particular length:
sketchAllHandlers len = do
  sendEventServerMessages [SketchAllHandlers len s]
  putStr s
 where s free


-- Open the external port "ehserver" and start the event handler server:
serve = do
  msgs <- openNamedPort "ehserver"
  time <- getClockTime
  ehServer 0 (toDate time) [] msgs

-- The event handler server loop:
-- Argument 1: Current index for numbering new events
-- Argument 2: Next date when cleanup is necessary
-- Argument 3: The current event handlers
--             (index,expiration date,cgikey,env,contents)
ehServer :: Int -> Date -> [(Int,Date,String,([(String,String)],String))] ->
            [EHSMsg] -> IO ()

ehServer _ _ _ (Stop:_) = done

ehServer maxkey cleandate ehs
         (StoreNewEnvEventWithCgiKey key time cgikey cenv info : msgs) = do
  doSolve $ key =:= maxkey
  ehServer incMaxKey cleandate ((key,time,cgikey,(cenv,info)):ehs) msgs
 where
  incMaxKey = if maxkey>30000 then 0 else maxkey+1 -- to avoid integer overflows

ehServer maxkey cleandate ehs
         (GetEnvEventWithCgiKey key time cgikey mbinfo : msgs) =
  if singleUseEventHandlers
  then deleteEv ehs >>= \newehs ->
       ehServer maxkey cleandate newehs msgs
  else (mbinfo =:= searchEv ehs) &> ehServer maxkey cleandate ehs msgs
 where
   deleteEv [] | mbinfo=:=Nothing = return []
   deleteEv (ev@(n,t,c,i):es) =
     if key==n && time==t
     then if c==cgikey then (mbinfo =:= Just i)  &> return es
                       else (mbinfo =:= Nothing) &> return (ev:es)
     else deleteEv es >>= \des -> return (ev:des)

   searchEv [] = Nothing
   searchEv ((n,t,c,i):es) = if key==n && time==t
                             then if c==cgikey then Just i else Nothing
                             else searchEv es

ehServer maxkey cleandate ehs (CleanOldEventHandlers : msgs) = do
  time <- getClockTime
  if compareDate (toDate time) cleandate == LT
   then ehServer maxkey cleandate ehs msgs
   else putStrLn (toDateString time ++": cleanup (current number of handlers: "++
                  show (length ehs) ++")") >>
        ehServer maxkey (toDate (nextCleanup time))
                 (filter (isNotExpired (toDate time)) ehs) msgs
 where
  isNotExpired time (_,etime,_,_) = compareDate time etime == LT

-- bind argument to a string representation of all event handlers
ehServer maxkey cleandate ehs (ShowAllHandlers s : msgs)
  | s =:= "Next cleanup: " ++ toDateString (dateToClockTime cleandate) ++"\n"++
          "Current event handlers:\n" ++ concatMap showWithoutCgiKey ehs
  = ehServer maxkey cleandate ehs msgs
 where
  showWithoutCgiKey (key,time,_,(_,info)) =
    "No. "++ show key ++" "++ show time ++": "++info++"\n"
  showWithCgiKey (key,time,cgi,(env,info)) = -- only for testing
    "No. "++ show key ++" / date: "++ show time ++" / key: \"" ++cgi++"\":\n  "++
    show env ++ "\n  "++info++"\n"

-- bind argument to a short string representation of all event handlers
-- where from each event handler only the first len characters are shown:
ehServer maxkey cleandate ehs (SketchAllHandlers len s : msgs)
  | s =:= "Next cleanup: " ++ toDateString (dateToClockTime cleandate) ++"\n"++
          "Current event handlers:\n" ++ concatMap showWithoutCgiKey ehs
  = ehServer maxkey cleandate ehs msgs
 where
  showWithoutCgiKey (key,time,_,(_,info)) =
    let infolength = length info in
    "No. "++ show key ++" "++ show time ++": "++
    take len info ++
    (if len<infolength then "...("++show infolength++" bytes)" else "") ++"\n"

-- The timeout (in msec) of the event handler server.
-- If the event handler server does not answer within the timeout period,
-- we assume that the server does not exist and we start a new one:
eventHandlerServerTimeOut = 2000

-- send a list of messages to local event handler server:
sendEventServerMessages :: [EHSMsg] -> IO ()
sendEventServerMessages msgs =
  connectPortRepeat eventHandlerServerTimeOut done 1 "ehserver@localhost" >>=
  maybe (startServer >> sendEventServerMessages msgs)
        (\port->mapIO_ (\m->doSend m port) msgs)

