------------------------------------------------------------------------------
--- Library for handling date and time information.
---
--- @author Michael Hanus
--- @version Tue Sep 13 12:22:24 CEST 2005
------------------------------------------------------------------------------

module Time(ClockTime,getClockTime,toDate,toTimeString,toDateString,
            dateToClockTime,
            addSeconds,addMinutes,addHours,addDays,addMonths,addYears,
            daysOfMonth,validDate,compareDate,compareClockTime) where


--- ClockTime represents a clock time in some internal representation.
data ClockTime = Time Int Int Int Int Int Int


--- Returns the current clock time.
getClockTime :: IO ClockTime
getClockTime external

--- Transforms a clock time into a tupel of integers containing
--- year, month, day, hour, minute, second.
toDate :: ClockTime -> (Int,Int,Int,Int,Int,Int)
toDate (Time y mo d h mi s) = (y,mo,d,h,mi,s)

--- Transforms a clock time into a string containing the time.
toTimeString :: ClockTime -> String
toTimeString (Time _ _ _ h mi s) = digit2 h ++":"++ digit2 mi ++":"++ digit2 s
  where digit2 n = if n<10 then ['0',chr(ord '0' + n)]
                           else show n

--- Transforms a clock time into a readable form.
toDateString :: ClockTime -> String
toDateString (Time y mo d h mi s) =
    shortMonths!!(mo-1) ++ " " ++ show d ++ " " ++
    toTimeString (Time y mo d h mi s) ++ " " ++ show y
  where shortMonths = ["Jan","Feb","Mar","Apr","May","Jun",
                       "Jul","Aug","Sep","Oct","Nov","Dec"]

--- Transforms a tupel of integers (containing year, month, day,
--- hour, minute, second) into a clock time.
dateToClockTime :: (Int,Int,Int,Int,Int,Int) -> ClockTime
dateToClockTime (y,mo,d,h,mi,s) = (Time y mo d h mi s)

--- Adds seconds to a given time.
addSeconds :: Int -> ClockTime -> ClockTime
addSeconds n (Time y mo d h mi s) = let ns = (s+n) `mod` 60 in
 if ns>=0
 then addMinutes ((s+n) `div` 60) (Time y mo d h mi ns)
 else addMinutes ((s+n) `div` 60 - 1) (Time y mo d h mi (ns+60))

--- Adds minutes to a given time.
addMinutes :: Int -> ClockTime -> ClockTime
addMinutes n (Time y mo d h mi s) = let nmi = (mi+n) `mod` 60 in
 if nmi>=0
 then addHours ((mi+n) `div` 60) (Time y mo d h nmi s)
 else addHours ((mi+n) `div` 60 - 1) (Time y mo d h (nmi+60) s)

--- Adds hours to a given time.
addHours :: Int -> ClockTime -> ClockTime
addHours n (Time y mo d h mi s) = let nh = (h+n) `mod` 24 in
 if nh>=0
 then addDays ((h+n) `div` 24) (Time y mo d nh mi s)
 else addDays ((h+n) `div` 24 - 1) (Time y mo d (nh+24) mi s)

--- Adds days to a given time.
addDays :: Int -> ClockTime -> ClockTime
addDays n (Time y mo d h mi s) =
 if n>=0
 then let monthdays = daysOfMonth mo y in
      if d+n <= monthdays
      then Time y mo (d+n) h mi s
      else addDays (n-monthdays) (addMonths 1 (Time y mo d h mi s))
 else if d+n>0
      then Time y mo (d+n) h mi s
      else let monthdays = if mo==1 then daysOfMonth 12 (y-1)
                                    else daysOfMonth (mo-1) y
            in addDays (n+monthdays) (addMonths (-1) (Time y mo d h mi s))

--- Adds months to a given time.
addMonths :: Int -> ClockTime -> ClockTime
addMonths n (Time y mo d h mi s) = let nmo = (mo-1+n) `mod` 12 + 1 in
 if nmo>0
 then addYears ((mo-1+n) `div` 12) (Time y nmo d h mi s)
 else addYears ((mo-1+n) `div` 12 - 1) (Time y (nmo+12) d h mi s)

--- Adds years to a given time.
addYears :: Int -> ClockTime -> ClockTime
addYears n (Time y mo d h mi s) = Time (y+n) mo d h mi s

--- Gets the days of a month in a year.
daysOfMonth :: Int -> Int -> Int
daysOfMonth mo yr =
  if mo/=2 
  then [31,28,31,30,31,30,31,31,30,31,30,31] !! (mo-1)
  else if yr `mod` 4 == 0 && (yr `mod` 100 /= 0 || yr `mod` 400 == 0)
       then 29
       else 28

--- Is a date consisting of year/month/day valid?
validDate y m d = m > 0 && m < 13 && d > 0 && d <= daysOfMonth m y

--- Compares two dates.
compareDate :: (Int,Int,Int,Int,Int,Int)
            -> (Int,Int,Int,Int,Int,Int) -> Ordering
compareDate (year1, month1, day1, hour1, min1, sec1)
            (year2, month2, day2, hour2, min2, sec2) =
    if (year1 < year2) then LT
    else if (year1 > year2) then GT
    else
     if (month1 < month2) then LT
     else if (month1 > month2) then GT
     else
      if (day1 < day2) then LT
      else if (day1 > day2) then GT
      else
       if (hour1 < hour2) then LT
       else if (hour1 > hour2) then GT
       else
        if (min1 < min2) then LT
        else if (min1 > min2) then GT
        else
         if (sec1 < sec2) then LT
         else if (sec1 > sec2) then GT
         else EQ

--- Compares two clock times. Since the current clock time is not
--- very precise, the comparison is based on comparing the corresponding dates.
compareClockTime :: ClockTime -> ClockTime -> Ordering
compareClockTime time1 time2 = compareDate (toDate time1) (toDate time2)
