% this module contains the implementation of primitive functions:

external_call(+(X,Y),Result) :- Result is X+Y.
external_call(-(X,Y),Result) :- Result is X-Y.
external_call(*(X,Y),Result) :- Result is X*Y.
external_call(div(X,Y),Result) :- Result is X // Y.
external_call(mod(X,Y),Result) :- Result is X mod Y.
external_call(<(X,Y),Result) :- X<Y -> Result=true ; Result=false.
external_call(>(X,Y),Result) :- X>Y -> Result=true ; Result=false.
external_call(<=(X,Y),Result) :- X=<Y -> Result=true ; Result=false.
external_call(>=(X,Y),Result) :- X>=Y -> Result=true ; Result=false.


external_call(printToScreen(X),'$io'(unit)):-
	\+ \+ (numbervars(X,0,_),write(X)),flush_output.

external_call(putChar(X),'$io'(unit)) :- put(X), flush_output.
external_call(putStr(L),'$io'(unit)):-writeOut(L),flush_output.
  writeOut([]).
  writeOut([X|Xs]):-put(X),writeOut(Xs).
external_call(putStrLn(L),'$io'(unit)):-writeOut(L),nl,flush_output.
  
external_call(getChar,'$io'(X)) :- get0(X).
external_call(done,'$io'(unit)).
external_call(return(X),'$io'(X)).

external_call(readFile(X),'$io'('#readFileContents'(FName))) :- name(FName,X).
external_call('#readFileContents'(FName),Result) :-
	seeing(CStr),
	see(FName),
	get0(Char),
	(Char = -1 -> Result=[], seen
                    ; Result=[Char|'#readFileContents'(FName)]),
	see(CStr).
external_call(writeFile(Name,Str),'#writeFileContents'(FName,Str)) :-
	name(FName,Name).
external_call('#writeFileContents'(FName,[]),'$io'(unit)) :-
	telling(CStr),
	tell(FName),
	told,
	tell(CStr).
external_call('#writeFileContents'(FName,[C|Cs]),'#writeFileContents'(FName,Cs)) :-
	telling(CStr),
	tell(FName),
	put(C),
	tell(CStr).


external_call(showConstructor(X),Result):- showConstructor(X,Result1),!,extractShow(Result1,[],Result).


external_call(error(Msg),_) :-
	atom_chars(AMsg,Msg),
	raise_exception(AMsg).

external_call(ensureNotFree(T),T).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% definitions for port communication:
:- use_module(library(sockets)).

% internal representation of the handle of a port:
% internalPort(Host,PNr,Socket) where:
% - for external ports:
%   Host: Internet name of the host (an atom)
%   PNr: socket number of the port at Host (an integer)
%   Socket: Sicstus-Prolog handle of this socket
% - for internal ports:
%   Host: not used
%   PNr:  0
%   Socket: a list with an open end (the stream)

%%%%% open
  
external_call(openPortOnce(_,_,Done),{}) :-
	nonvar(Done), !.
external_call(openPortOnce(_,Stream,_),{}) :-
	nonvar(Stream), !,
	write('*** Error in openPort: stream is not a free variable'), nl.
external_call(openPortOnce(Port,Stream,Done),{}) :-
	Port = internalPort('localport',0,Stream),
	Done=true.

external_call(openPortSocketOnce(_,_,_,Done),{}) :-
	nonvar(Done), !.
external_call(openPortSocketOnce(_,Stream,_,_),{}) :-
	nonvar(Stream), !,
	write('*** Error in openPort: stream is not a free variable'), nl.
external_call(openPortSocketOnce(Port,Stream,SocketNr,Done),{}) :-
	(var(SocketNr) -> true ; name(PNr,SocketNr)),
	socket('AF_INET',Socket),
	socket_bind(Socket,'AF_INET'(Host,PNr)),
	socket_listen(Socket,5),
	Port = internalPort(Host,PNr,Socket),
	%write('>>>Ports>>> Listen on Port "'),
	port2String(Port,Str),
	%name(PortName,Str),
	%write(PortName), write('"...'), nl,
	(var(SocketNr) -> SocketNr=Str ; true),
	Stream=basicserverloop(Port,_),
	Done=true.


%%%% send
  
external_call(sendPortOnce(_,_,Done),{}) :-
	nonvar(Done), !.
external_call(sendPortOnce(internalPort(_,0,Stream),Msg,Done),{}) :-
	!,
	add2Stream(Stream,Msg),
	Done=true.
external_call(sendPortOnce(internalPort(Host,PNr,Socket),Msg,Done),C) :-
	% external port: evaluate the message to ground normal form
	% before sending:
	C = sendPortOnceEval(internalPort(Host,PNr,Socket),groundnf(Msg),Done).

external_call(sendPortOnceEval(_,_,Done),{}) :-
	nonvar(Done), !.
external_call(sendPortOnceEval(internalPort(Host,PNr,_),Msg,Done),{}) :-
	% catch connection errors:
	on_exception(Exc,send2Socket(Host,PNr,Msg),
                     (print_error(Exc)->true;true)),
	Done=true.

% append a message to the open end of a stream:
add2Stream(Str,Item) :- var(Str), !, Str = [Item|_].
add2Stream([],_) :-
	write('*** Error in send: stream has not a free variable at the end'),
	nl.
add2Stream([_|Str],Item) :- add2Stream(Str,Item).


%%%% read

external_call(basicserverloop(_P,Result),Result):-nonvar(Result),!.
external_call(basicserverloop(P,Result),Result):-
  readPortOnce(P,ReadResult),
  if(var(ReadResult),
	 Result=basicserverloop(P,_),
	 Result=[ReadResult|basicserverloop(P,_)]).
 

  
external_call(basicStdInLoop(Result),Result):-nonvar(Result),!.
external_call(basicStdInLoop(Result),Result):-
  readStdInOnce(ReadResult),
  if(var(ReadResult),
	 Result=basicStdInLoop(_),
	 Result=[ReadResult|basicStdInLoop(_)]).


%%%% connect
  
external_call(connectPortExt(Name,internalPort(Host,Pnr,_)),{}) :-
	append(SPnr,[64|SHost],Name), % split the port name
	name(Pnr,SPnr), name(Host,SHost), !.
external_call(connectPortExt(Name,_),{}) :-
	write('*** Error in connectPort: non-valid port name: '),
	name(A,Name), write(A), nl.
    

%%%% timeout
  
external_call(afterExt(WMSec,Start,_),afterExt(WMSec,CurrTime,_)) :-
	var(Start), !,  % 1st call to after
	statistics(walltime,[CurrTime,_]).
external_call(afterExt(WMSec,Start,_),{}) :- % 2nd call to after
	statistics(walltime,[CurrTime,_]),
	WMSec < CurrTime - Start, !.
external_call(afterExt(WMSec,Start,_),afterExt(WMSec,Start,_)). % wait further

  
%%%%%% Hilfsfunktionen
  
readPortOnce(_,Result):-
	nonvar(Result), !.
readPortOnce(internalPort(Host,Pnr,Socket),_):-
	var(Socket), !,
	write('*** Error in readPort: Port '),
	write(Pnr), write('@'), write(Host),
	write(' only opened for writing!'), nl.
readPortOnce(internalPort(_,_,Socket),Msg) :-
	% catch connection errors:
	on_exception(Exc,readFromSocket(Socket,Msg),
                     (print_error(Exc)->Msg=[];Msg=[])), !.
% try later again if readFromSocket failed due to timeout:
readPortOnce(_Port,_).


readStdInOnce(Result):-nonvar(Result),!.
readStdInOnce(Result):-
  stream_select([user_input],0:50,[_]), % to prevent the input prompt before
  prompt(Old,''),
  readStdInStream(Result),                           % input started
  prompt(_,Old).
% try later again if readStdInOnce failed due to timeout:
readStdInOnce(_).



%readStdInStream([]):-at_end_of_line(user_input),!.
%readStdInStream([C|Cs]):-get0(user_input,C),readStdInStream(Cs).
readStdInStream(Result):-
  get0(user_input,C),
  if(C==10,
	 Result=[],
	 (readStdInStream(Result1),
	  Result=[C|Result1])).
  

% convert port into an external string representation:
port2String(internalPort(Host,PNr,_),S) :-
	name(PNr,LPnr),
	%LHost="localhost",
	name(Host,LHost),
	append("@",LHost,AtHost),
	append(LPnr,AtHost,S).

send2Socket(Host,PNr,Msg) :-
	socket('AF_INET',Socket),
	socket_connect(Socket,'AF_INET'(Host,PNr),PortStream),
	name(AMsg,Msg),
	write(PortStream,AMsg),
	close(PortStream).
	%write('>>>Ports>>> Sent to "'),
	%write(PNr), write('@'), write(Host), write('": '),
	%write(AMsg), nl.

readFromSocket(Socket,Msg) :-
%	socket_accept(Socket,Client,PortStream),
    socket_select([Socket],PortStreams,Clients,0:50,[],_), % timeout: 1 sec
	PortStreams=[PortStream], Clients=[_Client],
	%write('>>>Ports>>> Connection to client: '), write(Client), nl,
	readStream(PortStream,Msg),
	%write('>>>Ports>>> Received message: '), name(A,Msg), write(A), nl,
	!.
%readFromSocket(Socket,Msg) :- write('>>>Ports>>> Timeout!'), nl, fail.

readStream(Str,[]) :-
	at_end_of_stream(Str), !.
readStream(Str,[C|Rest]) :-
	get0(Str,C), readStream(Str,Rest).


  
%external_call(waitOnStdInOnce(_),{}):-write(1),
%  socket_select([],_,0:1,[user_input],[user_input]),write(2).
 % ReadStreams==[] -> Result=waitOnStdInOnce(_);Result={}.
%external_call(waitOnStdInOnce(_),waitOnStdInOnce(_)).
  
  
%%%%% Tools needed by the external functions

% showConstructor(Term,Term as String)
%
% transform all possible parts of a term into a string.
% function calls and variables cause showConstructor to suspend.
% Note, that in the initial call of showConstructor, Head can
% only be a constructor. The case distinction is
% only for the recursive evaluation of the arguments.
% Any arguments that are not constructors are transformed
% into show(..) again.
% Then "extractShow" removes them from the list
% of numbers representing the created string and
% put's them into "BeforeString++show[List](..)++RestString",
% which is a valid Curry goal.
% By this, a term wichout lists, functions and variables is
% transformed in one single step.

showConstructor(X,Result):- atom(X),!,name(X,Result).
showConstructor(X,Result):- integer(X),!,name(X,Result).
showConstructor(X,[show(X)]):- var(X),!.


showConstructor(X,Result):-
	X=..[Head|Args],
	showFunctionOrCons(X,Head,Args,Result).

%Head symbol is a constructor, then transform it into a string,
%except that it is a list,
showFunctionOrCons(_X,Head,Args,Result):-
	isCons(Head,_),
	\+ Head=='.',
	makeHead(Head,HeadString),
	append(HeadString,"(",String1),
	showArgs(Args,String2),
	append(String2,")",String3),
	append(String1,String3,Result).
%Head symbol is a function or a variable or a list constructor: suspend
showFunctionOrCons(X,_Head,_Args,[show(X)]).

makeHead(',',[]):-!.
makeHead(X,Head):-name(X,Head).

% transform arguments
showArgs(X,X):-var(X),!.
showArgs([],[]).
showArgs([X],Result):-showConstructor(X,Result).
showArgs([X|Xs],Result):-showConstructor(X,Result1),
	showArgs(Xs,Result2),!,
	append(",",Result2,Result3),
	append(Result1,Result3,Result).

extractShow([],List,List).
extractShow([X|Xs],List,Result):-
	integer(X),!,append(List,[X],List1),extractShow(Xs,List1,Result).
extractShow([X|Xs],[],Result):-
	extractShow(Xs,[],Result1),connectXXs(X,Result1,Result).
extractShow([X|Xs],[L|List],++([L|List],Result)):-
	extractShow(Xs,[],Result1),connectXXs(X,Result1,Result).

% note: List in the second clause can be a ...++... expression!!!
connectXXs(X,[],X):-!.
connectXXs(X,List,++(X,List)).


