s:-[search].

:- dynamic searchLambdaList/1.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%% Search-stuff %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% createSearchSpace(Lambda-Abtraction,SearchSpace)
%
% - creates a local computation space in which the lambda-
%   abstraction will be computed.
% - according to S19. 
%
% - attach the missing parameter to the Lambda-function.
% - if we Lambda has a rule-Tree, as it will always be if Lambda
%   is really al lambda-abstraction, we call csX to get the
%   body of the function and create the searchSpace for the body.
%   This saves one computation step.
%   But if we do not have a rule tree, csX might evoke a ND, so
%   we create the searchSpace for the function call and leave
%   the call itself to the later cseXlocal-computation.

createSearchSpace($Lambda,searchSpace(FreshX,[FreshX],[],Term)):-
	Lambda=..[Head|Args],
	append(Args,[FreshX],NewArgs),
	LambdaCall=..[Head|NewArgs],
	if(tree_of_root(Lambda,rule(L,R)),
           csX([],LambdaCall,rule(L,R),false,[(_V,_Sigma,Term)]),
	   Term=LambdaCall).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% cseXSearch(searchSpace(X,Var,Sigma,Term),ResultOf_cseX(Term),Result).
%
% - according to semantics on S52 ff., S63 ff.
% - if the searchspace is suspended, there are several possibilities in
%   going on. Call computeSuspendedSearch.
% - If it is not suspended, we either performed a deterministic step. Return
%   a searchSpace with the new term and append the old substitution to the new one
%   as we also do in general (in rn).
%   Or we performed a non-deterministic step. Then we must test if the searchSpace
%   was stable, because only then it is allowed to return several new spaces.
%   We know that no deterministic step was possible, if csX returns a ND. So we
%   only have to test whether the searchSpace contained global variables.
%   If so, suspend it. Otherwise return all the new branches
%   in a list of lambda-abstractions.

cseXSearch(_SearchSpace,[],[]).

cseXSearch(SearchSpace,CseXResult,Result):-
	suspended(CseXResult),
	computeSuspendedSearch(SearchSpace,Result).

cseXSearch(searchSpace(X,_V,Sigma,_Term),[(NewV,Sigma1,NewTerm)],searchSpace(X,NewV,NewSigma,NewTerm)):-
	appendSubst(Sigma,Sigma1,NewSigma).


cseXSearch(searchSpace(X,V,Sigma,Term),CseXNDResult,Result):-
	globalVars(V,Term,GlobalVars),
	if(GlobalVars==[],
	   return(X,Sigma,CseXNDResult,Result),
	   Result=unboundVars(GlobalVars,searchSpace(X,V,Sigma,Term))).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% computeSuspendedSearch(searchSpace(X,Var,Sigma,Term),Result)
%
% - The searchspace suspended, i.e. Term cannot be reduced
%   deterministic anymore. There are two possibilities now:
% - If the searchSpace was reduced to "{}", cseX resulted in suspend({}),
%   so now we are completely reduced
%   and give back the local space as lambda-abstraction. 
%   To do so, we call tl  immediately, because return does not 
%   create a lambda-abstraction, but tl does. V is passed
%   to tl for comparison with the set of variables that might
%   appear in the substitiution of X (see tl and makeLambda for
%   explanation)
% - Otherwise, the term was not reduced to {} but suspended.
%   If Term contains global variables, mark the space as instable.
%   Note: a space can also suspend, if it tries to bind a global
%         variable. Then it is not instable! We use the term
%         "instable" in the rest of the file for this case,
%         but we mean instable spaces AND spaces which suspended
%         because they tried to bind a global variable.
%         Therefore we use "unboundVars" as marker for those spaces.
%   Otherwise suspend it.


computeSuspendedSearch(searchSpace(X,V,Sigma,{}),[Result]):-!,
	tl(X,V,Sigma,{},Result).

computeSuspendedSearch(searchSpace(X,V,Sigma,Term),Result):-
	globalVars(V,Term,GlobalVars),
	if(GlobalVars==[],
	   Result=suspend(searchSpace(X,V,Sigma,Term)),
	   Result=unboundVars(GlobalVars,searchSpace(X,V,Sigma,Term))).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% return(X,Sigma,Result,Lambda-Searchspace) 
%
% - If the searchSpace was stable, it was either reduced to {} or to several
%   solutions. Return shell put the solutions into bodies of lambda-expressions
%   and return the partial calls in a list. 
%   E.g. searchSpace(X,[A],[X/s(A)],A=f(1)) shell return
%   [$searchLambda1] and add the definition 
%   searchLambda1(X) = {local A in X=s(A)/\A=f(1)}
% - But we have choosen a more efficient way:
%   return packs the soultions in "searchLambda(X,V,Sigma,Body)"
%   If we meet searchLambda(..) in cseX, we create the lambda-abstraction.
%   But if we meet searchLambda(..) as parameter of "try", we can immediately
%   call cseXSearch. So we must not put the searchSpace into a lambda-abstraction
%   and unpack this abstraction in the next step. Very useful, for example,
%   with the all-Search-example.
%   But we can also meet a searchLambda(..)@X-call, and of course searchLambda(..)
%   must be transformed into "$searchLambdaN" then, too. The apply-predicate will
%   compare the arity and number of parameters of searchLambda. Therefore we 
%   must declare searchLambda as 4ary function internally, so that searchLambda will
%   be evaluated before applying X to it.. 

return(_X,_Sigma,[],[]).

return(X,Sigma,[(V,Sigma1,R)|Rs],[LambdaR|LambdaRs]):-
	appendSubst(Sigma,Sigma1,NewSigma),
	stl2(NewSigma,X,NewSigma1),
        LambdaR=searchLambda(X,V,NewSigma1,R),
	return(X,Sigma,Rs,LambdaRs).

% minimize the substitution, only the value for X is needed in future.
stl2([],_,[]).
stl2([(Y/T)|_R],X,[X/T]):-X==Y.
stl2([_|R],X,Result):-stl2(R,X,Result).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% tl(X,V,Sigma,Body,NewTerm)
%
% - according to the semantics on S26,S58
%
% - tl=toLambda
% - tl packs a stable searchSpace into a lambda-expression
% - first call stl (sigmaToLambda), which transforms the substitution of the
%   searchSpace into an expression. This expression can be "{}" if the
%   substitution does not contain a substitution for X, the searchvariable
%   (see stl below). Call minimizeBody then with the computed expression of stl
%   and Body, which is the computed expression of the searchSpace. If one
%   of these expressions is "{}", it is dropped to avoid a "{}/\t" or
%   "t/\{}"-result, otherwise we connect both expression with /\. 
% - stl also returns a set of variables. These are the variables that appear in the
%   substitution for X. If they are not in V, they must be declared as parameters
%   of the lambda-abstraction, so we pass them to makelambda. See example there.
% - finally call makeLambda to put the NewBody-Expression into a lambda-abstraction.

tl(X,V,Sigma,Body,NewTerm):-
	stl(Sigma,X,SubstBody,SubstVars),
	minimizeBody(Body,SubstBody,NewBody),
	removeVars(V,SubstVars,GlobalVars),
	makeLambda(X,V,GlobalVars,NewBody,NewTerm).


minimizeBody({},B,B).
minimizeBody(B,{},B).
minimizeBody(B1,B2,B1/\B2).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% stl(Sigma,X,Term,SubstVars)
%
% - according to the semantics given on S26, S59
%
% - If Sigma containts a binding for X, i.e. (X/t), result in X=T.
%   Otherwise result in "{}". So we give back only the binding of
%   X, because bindings of local variables are not of interest outside
%   the local space, so drop them here.
% - We also give back the variables that appear in the substitution for X, because
%   they can be global variables!!! So we must declare them as parameters of
%   the lambda-abstraction that we create in makeLambda.

stl([],_,{},[]).
stl([(Y/T)|_R],X,(X=T),Vars):-X==Y,freeVars(T,Vars).
stl([_|R],X,Result,SubstVars):-stl(R,X,Result,SubstVars).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% makeLambda(X,V,GlobalVars,Body,$Lambda).
%
% - Theorie: Return \X.local V in Body, with V the variables that appear in Body.
% - In our implementation, we cannot return \X.local V in Body, but must return
%   $searchLambdaN, and add a rule "searchLambdaN(X)=local V in Body".
% - X is the variable thats value we are interested in. In general X should
%   appear in Body, the best case would be "X=t" with t reduced completely.
%   So as we defive "searchLambdaN(X)=...", an application of $searchLambdaN
%   to any variable Y, will return the value computed for X from Body and bind
%   it to Y.
% - Notice: Although a searchSpace can only reduce of it does not contain global
%   variables, such variables can yet appear, if they were in the substitution
%   created for X! For example try(\X->X=add(s(s(z)),N)) will compute the 
%   substitution X/s(s(N)) and the searchSpaces can reduce. We will then
%   transform the substitution into a term again by stl, so N will appear
%   in the lambda-abstraction that we create here. As it is a global variable
%   it must be declared as parameter: "searchLambda(N,X)={X=s(s(N))}".
%   These global variables are found by stl and passed as GlobalVars.
% - Concrete:
%   * first construct the name searchLambdaN, return it in $Lambda
%   * X might appear V, if it has not been bound yet. But X is not a
%     normal local variable but the search-variable. So remove it from V.
%   * Then create the left and right side of the searchLambdaN-Rule.
%     Left side is searchLambdaN(GlobalVars,X), right side is Body with the free
%     variables declared local to it.
%   * assert and retract the rule to create it with fresh and unique
%     variables as it is necessary for a function definition.
%   * finally assert the function-predicate, the definitional tree(which
%     is easy for a lambda-abstraction, just a rule), and the tree-flag.
% - we also collect the names of the lambda-abstractions we create, so that
%   after the computation we can remove them.

makeLambda(X,V,GlobalVars,Body,$LambdaCall):-!,
	getlambdacounter(N),
	name(N,StringN),
	append("searchLambda",StringN,StringLambda),
	name(Lambda,StringLambda),
	LambdaCall=..[Lambda|GlobalVars],

	removeVarFromSet(X,V,VarList),
        append(GlobalVars,[X],Parameters),
	L=..[Lambda|Parameters],
	declareVarsListInTerm(VarList,Body,R),

	assertz(fresh(L,R)),
	retract(fresh(NL,NR)),

        length([X|GlobalVars],Arity),
        createType(Arity,Type),
	assertz(function(Lambda,Arity,Type)),

	assertz(tree(Lambda,rule(NL,NR))),
	assertz(functionTreeFlag(Lambda,standard)),

	adjustSearchLambdaList(Lambda).

createType(0,constraint).
createType(N,(_A->Type)):-N1 is N-1,createType(N1,Type).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% adjustSearchLambdaList(Lambda)
%
% - add Lambda to the list of already created lambda-abstractions
%   or create a new list, of no lambda-abstractions have been
%   created before.

adjustSearchLambdaList(Lambda):-
	if(retract(searchLambdaList(X)),
	   assertz(searchLambdaList([Lambda|X])),
	   assertz(searchLambdaList([Lambda]))).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% clearSearchLambdas
%
% - if lambda-abstractions were created by makeLambda, remove them
%   together with their flags (function,tree,rule,functionTreeFlag)
% - also remove lambdacounter, so that we start with searchLambda1
%   at every computation.

clearSearchLambdas:-
	if(retract(searchLambdaList(X)),
	   (removeSearchLambdas(X),retractall(lambdacounter(_))),
           true).

removeSearchLambdas([]).
removeSearchLambdas([Lambda|Xs]):-
	retractall(function(Lambda,_,_)),
	retractall(tree(Lambda,_)),
	retractall(functionTreeFlag(Lambda,_)),
	removeSearchLambdas(Xs).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%% Choice-stuff %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% createChoiceSpaces(Args,Spaces)
%
% - Args ist the list of (ci,ei). Transform them into a quadrupel,
%   (LokalVars,Substitution,NewC,E) where Substitution is empty
%   at the beginning and a possibly local-declaration of ci is
%   already reduced and the variables put into LokalVars (for
%   reasons of efficiency)

createChoiceSpaces([],[]).
createChoiceSpaces([(C,E)|Xs],[(VarC,[],NewC,E)|XsSpaces]):-
	getLocalDecl(C,VarC,NewC),
	createChoiceSpaces(Xs,XsSpaces).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% getLocalDecl(Term,LocalList,NewTerm)
%
% - extracts a local declaration from C, if one exists

getLocalDecl(List localVars C,List,C).
getLocalDecl(C,[],C).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% cseXChoice(SpacesToCompute,Result)
% calls
% cseXChoice(SpacesToCompute,InstableFlag,SuspendedFlag,ComputedSpaces,Result).
%
% - Idea: Step through the first argument, the list of spaces to compute.
%   If we find suspended or instable spaces, put them into ComputedSpaces,
%   set the according flag so signalize what kind of spaces we have found,
%   and go on, trying to find a spaces that is neither suspended nor instable
%
% - If first argument is empty, all spaces have been computed and
%   that means:
%   1) We had only one computable space at all and this one failed
%   2) All spaces were either instable or suspended with at least
%      one instable. Then we mark the whole space as instable, because
%      only the instable spaces can be recomputed later, and attach to
%      it the set of ALL global variables that appear in the single spaces.
%      So we will only enter the choiceSpaces-construct again, if at least
%      one of the contained spaces has one of its global variables got bound.
%   3) All spaces suspended, suspend the whole space then.
%
% - the reverse is neccessary to put the spaces into their original order.
%   This is more efficient than keeping the order while computing through
%   an append-operation each time we move a space from SpacesToCompute
%   to ComputedSpaces. 
%
% - If the first argument is not empty, we try to find a computable space.
%   * a suspended one remains suspended
%   * an instable is tested if one of the global variables has been bound.
%     If not, let it instable.
%   * If one has been bound, compute the space, hoping it can go on now.
%   * Otherwise, we have a normal computable space. Compute it.


cseXChoice(Spaces,Result):-cseXChoice(Spaces,false,false,[],Result).

cseXChoice([],false,false,[],[]).
cseXChoice([],true,_SuspendFlag,All,[([],[],unboundVars(V,choiceSpaces(NewAll)))]):-
	collectVFromInstables(All,V),
	reverse(All,NewAll).
cseXChoice([],false,true,All,[([],[],suspend(choiceSpaces(NewAll)))]):-
	reverse(All,NewAll).


cseXChoice([(V,Sigma,suspend(Space),E)|Xs],Instables,_Suspended,Computed,Result):-
	cseXChoice(Xs,Instables,true,[(V,Sigma,suspend(Space),E)|Computed],Result).

cseXChoice([unboundVars(V,Space)|Xs],_Instables,Suspended,Computed,Result):-
	nobound(V),!,
	cseXChoice(Xs,true,Suspended,[unboundVars(V,Space)|Computed],Result).
	
cseXChoice([unboundVars(_,(V,Sigma,C,E))|Xs],_Instables,_Suspended,Computed,Result):-
	cseX(V,C,true,CResult),
	returnChoice((V,Sigma,C,E),CResult,Xs,Computed,Result).


cseXChoice([(V,Sigma,C,E)|Xs],_Instables,_Suspended,Computed,Result):-
	cseX(V,C,true,CResult),
	returnChoice((V,Sigma,C,E),CResult,Xs,Computed,Result).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% collectVFromInstables(Spaces,V)
%
%  - Spaces is a list of instables or suspended choicespaces.
%    Collect the V's of all the instable spaces.

collectVFromInstables([],[]).
collectVFromInstables([unboundVars(V,_Space)|Spaces],ErgV):-
	collectVFromInstables(Spaces,VSpaces),
	appendSet(V,VSpaces,ErgV).	
collectVFromInstables([_Space|Spaces],ErgV):-
	collectVFromInstables(Spaces,ErgV).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% returnChoice(Space,ResultOfcseX(C),RestSpaces,ComputedSpaces,Result)
%
% - If cseX returns [], the space failed. So remove it and give back
%   the other ones. If there are no other one, give back [] (means: failed)
% - If cseX returns a suspended result, examine it closer
% - If cseX returns a normal result, that means the space was reduced
%   deterministic. Just give back the new space then, together with
%   the other, unchanged spaces, but put it at the end, behind all the
%   other spaces to guarantee a fair strategy!
% - Otherwise, cseX returned a ND-Result, i.e. more than one results. Do the
%   same as with the searchSpaces: If the space was stable, i.e. no global
%   variables appeared in the condition, return the results. Otherwise
%   mark the space as instable.

returnChoice(_Space,[],[],[],[]):-!.
returnChoice(_Space,[],Xs,Computed,[([],[],choiceSpaces(All))]):-!,
	reverse(Computed,Computed1),
	append(Computed1,Xs,All).

returnChoice(Space,CseXResult,Xs,Computed,Result):-
	suspended(CseXResult),
	computeSuspendedChoice(Space,Xs,Computed,Result).

returnChoice((_V,Sigma,_C,E),[(NewV,Sigma1,NewC)],Xs,Computed,[([],[],choiceSpaces(All))]):-
	appendSubst(Sigma,Sigma1,NewSigma),
	reverse(Computed,Computed1),

% 1)normal choice
%	append(Computed1,[(NewV,NewSigma,NewC,E)|Xs],All).

% 2) fair choice
% just put the reduced space to the end of all spaces. This guarantees
% a complete search strategie
	append(Xs,[(NewV,NewSigma,NewC,E)],Xs1),
	append(Computed1,Xs1,All).




returnChoice((V,Sigma,C,E),CseXNDResult,Xs,Computed,[([],[],choiceSpaces(All))]):-
	globalVars(V,C,GlobalVars),
        if(GlobalVars==[],
           transformResultIntoChoiceSpaces(Sigma,E,CseXNDResult,NewChoiceSpaces),
           NewChoiceSpaces=[unboundVars(GlobalVars,(V,Sigma,C,E))]),
        reverse(Computed,Computed1),
% 1) normal choice
%	append(Computed1,NewChoiceSpaces,Spaces1),
%	append(Spaces1,Xs,All).

% 2) fair choice
% just put the reduced space to the end of all spaces. This guarantees
% a complete search strategie
	append(Xs,NewChoiceSpaces,Xs1),
	append(Computed1,Xs1,All).





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% computeSuspendedChoice(Space,Restspaces,ComputedSpaces,Result)
% 
% - If the condition is reduced to {}, return the body sigma(e)
%   and V, because these variables can appear in e!!! We do not give
%   back Sigma, because we are not interested on bindings of local
%   variables after leaving choice.
% - Otherwise, the condition was not reduced to {} but suspended.
%   If C contains global variables, mark the space as instable.
%   Otherwise suspend it.

computeSuspendedChoice((V,Sigma,{},E),_Xs,_Computed,[(V,[],NewE)]):-
	substitute(E,Sigma,NewE).

computeSuspendedChoice((V,Sigma,C,E),Xs,Computed,[([],[],choiceSpaces(All))]):-
	globalVars(V,C,GlobalVars),
	if(GlobalVars==[],
	   Space=(V,Sigma,suspend(C),E),
           Space=unboundVars(GlobalVars,(V,Sigma,C,E))),
	reverse(Computed,Computed1),
	append(Xs,[Space],Xs1),
	append(Computed1,Xs1,All).





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% transformResultIntoChoiceSpaces(Sigma,Body,cseXResult,NewChoiceSpaces)
%
% - the results of cseXResult must be transformed into the choice-space-form,
%   that means attach the body E to each of the results that were computed
%   from the according C.

transformResultIntoChoiceSpaces(_Sigma,_E,[],[]).
transformResultIntoChoiceSpaces(Sigma,E,[(V,Sigma1,C)|Xs],[(V,NewSigma,C,E)|NewXs]):-
	appendSubst(Sigma,Sigma1,NewSigma),
	transformResultIntoChoiceSpaces(Sigma,E,Xs,NewXs).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% globalVars(Vars,Term,GlobalVars,NotGlobalVars)
%
% - according to the semantics given onS63 ff. It's called "stable" there.
%
% - return all variables that appear free in Term and are not in Vars.

globalVars(Vars,Term,GlobalVars):-
	freeVars(Term,FreeVars),
	removeVars(Vars,FreeVars,GlobalVars).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%
% altes return

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% return(X,Sigma,Result,Lambda-Expressions) 
%
% - If the searchSpace was stable, it was either reduced to {} or to several
%   solutions. Return shell put the solutions into bodies of lambda-expressions
%   and return the partial calls in a list. 
%   E.g. searchSpace(X,[A],[X/s(A)],A=f(1)) shell return
%   [$searchLambda1] and add the definition 
%   searchLambda1(X) = {local A in X=s(A)/\A=f(1)}
% - we can have several solutions, so return works on a list. Each single
%   solution is computed by tl, which creates the lambda-abstraction and
%   returns the partial call of it.
% - This goes all according to the semantics given on S54


%return(_X,_Sigma,[],[]).

%return(X,Sigma,[(V,Sigma1,R)|Rs],[LambdaR|LambdaRs]):-
%	appendSubst(Sigma,Sigma1,NewSigma),
%        tl(X,V,NewSigma,R,LambdaR),
%	return(X,Sigma,Rs,LambdaRs).
