%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHANGES since July 98:
%
%%%%%
% 8.12.98 - Frank Steiner
% - for the transformation of rules with several guards:
%   - bugfix made in typeTypedFunction
%   - typeRule(s) additionally returns a list of typevariables,
%     the ones that were found for the guards of unknown type
%   - type(Un)TypedFunction passes this list upwards
%   - typeComponent processes this list
%   - setGuardTypes added
%   - checkGuardType added
%   - errorMessageForRule added

%%%%%
% 12.9.98 - Frank Steiner
% in typeTypedFunction bugfix added

%%%%%
% 8.1.99 - Frank Steiner
%
% for enabling the new choice annotation:
% typeTerm(Cx,(T1<--T2),constraint,(NT1<--NT2),Cx3) :- !,...   added
  
rt:-consult('typecheck').

:-dynamic writetermfact/1,lhs_Start/0,lhs_Args/0,tc_shell/0,lambdacounter/1,wherecounter/1.

typeCheckAll:-!,typeAll.

typeCheckAll(SymbTab):-assert(symbtab(SymbTab)),typeCheckAll,retractall(symbtab(_)).
  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% perform type-checking for a curry-script
%
  
typeAll:-
    retractall(tc_shell),
    typeUndefined,					% - type all funcs with type but no rules
    collectAllRules(FuncRules),				% - collect all rule-defs in FuncRules
    createWheres(FuncRules,FuncRules2),			% - create rules for local definitions (where)
    createLambdas(FuncRules2,FuncRules2a),		% - create rules for lambda-expressions
    createNormalForms(FuncRules2a,FuncRules3),		% - transform rules into their normal-forms
    buildCallGraph(FuncRules3,CGraph),!,		% - create dependency-call-graph
    testCallGraph(CGraph,CGraph,CGraph2),!,		% - test call-graph (all used funcs defined ?)
    buildSuperStructGraph(CGraph2,SGraph),!,		% - create strong call-graph
    typeComponents(FuncRules3,SGraph).			% - type all rules




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% transform the right sides of each rule into its normal-form

createNormalForms([],[]).
createNormalForms([(F,FRules)|Rules],[(F,NewFRules)|NewRules]):-
	createNormalFormsForFunc(FRules,NewFRules),
	createNormalForms(Rules,NewRules).

createNormalFormsForFunc([],[]).
createNormalFormsForFunc([(L,R)|Rules],[(L,NewR)|NewRules]):-
	createNF(L=R,NewR),
	createNormalFormsForFunc(Rules,NewRules).



% typeUndefined
%   type all funcs with a defined type but no defined rules
%   I suppose they are external funcs. But if they are not primitive, the user will see a warning
%

typeUndefined:-temptype(_,_,_),!,typeUndefined2.
typeUndefined.

typeUndefined2:-
    temptype(F,N,Type),
    \+ (temprule(L,_),functor(L,F,_)),
    !,retract(temptype(F,N,Type)),assertz(function(F,N,Type)),
    ifc(isExternal(F),
       true,
       (nl,\+ \+ (numbervars(Type,0,_),
       format('Warning: type declared, but no rules found: ~w/~w::~w',[F,N,Type])))
      ),
    typeUndefined2.
typeUndefined2:-nl.

% typeComponents(FuncRules,Graph)
%    create for all funcs in the graph a type
%    the type will be asserted: function(Funcname,Arity,Type)

typeComponents(_,[]):-!.
typeComponents(FuncRules,Graph):-
    findBasicComponent(Graph,Component),!,			% find component with no dependency
    typeComponent(FuncRules,Component),				% type the funcs of this component
    removeComponentRules(Component,FuncRules,FuncRules2),	% remove component
    removeComponentNodes(Component,Graph,Graph2),!,
    typeComponents(FuncRules2,Graph2).

% typeComponent(FuncRules,Component)
%    type all funcs in the component
%    the types in the type-context created by typeFunctions(...) will be asserted
%   unknownGuardTypes is a list of (Rule,Variable) where Variable is a typevariable
%   that has been derived as type for the guards in Rule with several guards.
%   They are detected and collected in typeRule.  
%   The types of these guards will be set to constraint after the component has been typed.
%   With that, the type of a function like
  

  
typeComponent(FuncRules,Component):-!,
    typeFunctions(FuncRules,Component,[],Context,UnknownGuardTypes),!,
    setGuardTypes(UnknownGuardTypes),
    addtypedContext(Context).

% if during typechecking the component the type of the guards for a rule with
% several guards could not be detected, the typevariables derived for the type
% are collected and returned in the list UnknownGuardTypes, together with the
% the according rule. Possibly these
% typevariables have been bound afterwards, but if not, set them to constraint.
% if they have been bound in the meantime, check that they are either constraint
% or boolean. Otherwise report an error message,
    setGuardTypes([]).
    setGuardTypes([(_Rule,TypVar)|Xs]):-var(TypVar),!,TypVar=constraint,
      %write('Setting unknown Guard type to constraint for rule '),nl,
      %write(Rule),nl,
      setGuardTypes(Xs).
    setGuardTypes([(_Rule,constraint)|Xs]):-!,setGuardTypes(Xs).
    setGuardTypes([(_Rule,bool)|Xs]):-!,setGuardTypes(Xs).
    setGuardTypes([((L,condList(CList,RList)),T)|_Xs]):-nl,
      write('*** Type-Error: Illegal type '),write(T),
      write(' derived for the guards in the rule: '),nl,
      numbervars((L,condList(CList,RList)),0,_),
      write(L),nl,!,
      errorMessageForRule(CList,RList).
  
      errorMessageForRule([],[]):-!,fail.
	  errorMessageForRule([C|CList],[R|RList]):-
        write(' |'),write(C),write(' = '),write(R),nl,
        errorMessageForRule(CList,RList).
  

  addtypedContext([]).
  addtypedContext([(F,(N,T))|Cx]):-assertz(function(F,N,T)),addtypedContext(Cx).

% typeFunctions(FuncRules,FuncNames,Typecontext,New typecontext,UnknownGuardTypes)
%   creates a new typecontext with the types of the functions in the FuncNames-list
%   if user has defined a type for a function (in temptype(...)) this type will be compared with
%   the type inferred by the typechecker.
%   The typechecker produces for each checked func-rule a new rule (asserted to rule(Left,Right)).
%   unknownGuardTypes is a list of (Rule,Variable) where Variable is a typevariable
%   that has been derived as type for the guards in Rule with several guards.
%   The types of these guards will be set to constraint after the component has been typed.
%   Therefore we must collect and return them. They are detected and collected in typeRule.


typeFunctions(_FuncRules,[],Context,Context,[]).
typeFunctions(FuncRules,[F|Fs],Context,NewContext,UnknownGuardTypes):-
  write('.'),flush_output,!,
  if(temptype(F,N,T),
	 (typeTypedFunction(FuncRules,Context,F,N,T,UnknownGuardTypes1),
	  Context2=Context
	 ),
	 (typeUntypedFunction(FuncRules,Context,F,InferredN,InferredType,UnknownGuardTypes1),
	  Context2=[(F,(InferredN,InferredType))|Context]
	 )
	),
  typeFunctions(FuncRules,Fs,Context2,NewContext,UnknownGuardTypes2),
  append(UnknownGuardTypes1,UnknownGuardTypes2,UnknownGuardTypes).
  
typeTypedFunction(FuncRules,Context,F,N,FuncType,UnknownGuardTypes):-
    retract(temptype(F,N,FuncType)),assertz(function(F,N,FuncType)),
    typeUntypedFunction(FuncRules,Context,F,_InferredN,InferredType,UnknownGuardTypes),
    assert(bugfix(InferredType)),retract(bugfix(InferredTypeCopy)),
% this is neccessary because otherwise when having f(X) = g(X) and for g the type
% list(_123) has been found, the inferred type of f(X) is list(_123),too. Now
% FuncType is list(_345) but with numbervars _345 is replaced by A, which is
% not a variable.
% Now, by (originally) "ifc(FuncType=InferredType" _123 is bound to A, and thus,
% the type of g suddenly is list(A) where A is not a variable. This would lead
% to an error.
    assert(bugfix(FuncType)),retract(bugfix(FuncTypeCopy)),
% this only for prevention of later errors. Prevents FuncType vom beeing
% numbervar'ed and asserted with these numbervar'ed variables
    numbervars(FuncTypeCopy,0,_),!,
    ifc(FuncTypeCopy=InferredTypeCopy,
        true,
        writeTypeError("Type is too general~ndeclared: ~w~nderived: ~w",[FuncTypeCopy,InferredTypeCopy])
    ).

typeUntypedFunction(FuncRules,Context,F,N,FuncType,UnknownGuardTypes):-!,
    findRulesOfF(FuncRules,F,Rules),!,
    typeRules([(F,(_,FuncType))|Context],Rules,NewRules,UnknownGuardTypes),
    findArity(FuncType,N),
    expandArgs(NewRules,N,NewRules2),
    addrules(NewRules2).

    addrules([]).
    addrules([(L,R)|Rules]):-
	retransformLocal(R,NewR),
%	writeterm(L=NewR),nl,
        assertz(rule(L,NewR)),addrules(Rules).


    % Expandiert die Anzahl der Parameter einer Regel auf N (z.B. z = $s auf 1: z(X)=$s @ X)

    expandArgs([],_,[]).
    expandArgs([Rule|Rules],N,[NewRule|NewRules]):-
    	expandRule(Rule,N,NewRule),expandArgs(Rules,N,NewRules).
    expandRule((L,R),N,(L,R)):-
    	functor(L,_,N),!.
    expandRule((L,R),N,(NL,condList(NC,NR))):-nonvar(R),
        R=condList(CList,RList),!,
    	L=..[Left|LArgs],append(LArgs,[X],NewLArgs),NewL=..[Left|NewLArgs],
		expandCondList(X,RList,NewRList),
    	expandRule((NewL,condList(CList,NewRList)),N,(NL,condList(NC,NR))).
    expandRule((L,R),N,(NL,NR)):-
    	L=..[Left|LArgs],append(LArgs,[X],NewLArgs),NewL=..[Left|NewLArgs],
    	expandRule((NewL,(R @ X)),N,(NL,NR)).

    expandCondList(_,[],[]).
	expandCondList(X,[R|Rs],[(R@X)|NRs]):- expandCondList(X,Rs,NRs).
			   

typeRules(_Context,[],[],[]).
typeRules(Context,[Rule|Rules],[NewRule|NewRules],UnknownGuardTypes):-
    typeRule(Context,Rule,NewRule,UnknownGuardTypes1),!,
    typeRules(Context,Rules,NewRules,UnknownGuardTypes2),
	append(UnknownGuardTypes1,UnknownGuardTypes2,UnknownGuardTypes).


typeRule(Context,(UnfixedL,UnfixedR),NewRule,UnknownGuardTypes):-
  fixvars((UnfixedL,UnfixedR),(L,R),FixList),
  retract(currline(_)),assertz(currline(L=R)),!,
  
  typeLhs(Context,L,LType,NL,Context2),!,
  ifc((nonvar(R),R=condList(CList,RList)), % if we have a rule with several guards, do a
	                                       % little trick. check condList2 which has type
	                                       % list(a) -> list(b) -> pair(a,b). So the type
	                                       % derived for the guards will be returned, too.
	  (%nl,write('found a rule with several guards: '),write(L),write(' = '),write(R),nl,
	   typeTerm(Context2,condList2(CList,RList),pair(CType,RType),condList2(NCL,NRL),_),
	   NR=condList(NCL,NRL),!,
	   ifc(var(CType), % if type of guard is unknown, safe the typevariable derived for it
		  %(write('GuardType unknown'),nl,
		   UnknownGuardTypes=[((UnfixedL,UnfixedR),CType)],%),  % together with the rule
		  (%write('GuardType known: '),write(CType),nl,
		   checkGuardType(CType,L=condList(CList,RList)),
		   UnknownGuardTypes=[]))
	   ),
	  (typeTerm(Context2,R,RType,NR,_),
	   UnknownGuardTypes=[])),
	  
  !,
  
  ifc(unifytype(LType,RType),
	  unfixvars((NL,NR),NewRule,FixList),
	  writeTypeError("Type of lhs does not match type of rhs",[])
	 ).


checkGuardType(constraint,_).
checkGuardType(bool,_).
checkGuardType(CType,L=condList(CList,RList)):-nl,
      write('*** Type-Error: Illegal type '),write(CType),
      write(' derived for the guards in the rule: '),nl,
      numbervars((L,condList(CList,RList)),0,_),
      write(L),nl,!,
      errorMessageForRule(CList,RList).
  
       
fixvars(Term,NewTerm,FixList):-fixvars(Term,NewTerm,0,_,[],FixList).

fixvars(Var,'$VAR'(N1),N,N,FixList,FixList):- var(Var),findVar(Var,FixList,N1),!.
fixvars(Var,'$VAR'(N),N,N2,FixList,[(N,Var)|FixList]):- var(Var),!,N2 is N+1.

fixvars(Term,NewTerm,N,N2,FixList,NewFixList):-
    Term=..[H,Arg|Args],!,fixvars(Arg,NewArg,N,N3,FixList,FixList2),
    Term2=..[H|Args],fixvars(Term2,Term3,N3,N2,FixList2,NewFixList),
    Term3=..[H|NewArgs],NewTerm=..[H,NewArg|NewArgs].
fixvars(Term,Term,N,N,FixList,FixList).

    findVar(Var,[(N,V)|_],N):-Var==V,!.
    findVar(Var,[_|Ls],N):-findVar(Var,Ls,N).

unfixvars(Term,NewTerm,FixList):-unfixvars(Term,NewTerm,FixList,_).

unfixvars('$VAR'(N),Var,VarList,VarList):-findValue(N,VarList,Var),!.
unfixvars('$VAR'(N),Var,VarList,[(N,Var)|VarList]):-!.		% kann dieser Fall auftreten ?
unfixvars(Term,NewTerm,VarList,NewVarList):-
    Term=..[H,Arg|Args],!,unfixvars(Arg,NewArg,VarList,VarList2),
    Term2=..[H|Args],unfixvars(Term2,NewTerm2,VarList2,NewVarList),
    NewTerm2=..[H|NewArgs],NewTerm=..[H,NewArg|NewArgs].
unfixvars(Term,Term,VarList,VarList).
       
typeLhs(Cx,L,LType,NL,Cx2):-
    retractall(lhs_Start),retractall(lhs_Args),
    functor(L,_F,_),assertz(lhs_Start),
    typeTerm(Cx,L,LType,NL,Cx2),
    retract(lhs_Args).

%
% this typeTerm/3 is for typechecks initiated from the curry-shell
%    all functions are "external"

typeTerm(UnfixedTerm,Type,NewTerm2):-
    fixvars(UnfixedTerm,Term,FixList),
    retractall(tc_shell),assertz(tc_shell),
    retractall(currline(_)),assertz(currline(Term)),
    typeTerm([],Term,Type,NewTerm,_),
    unfixvars(NewTerm,NewTerm2,FixList).

%%%% typeTerm(Context,Term,Typ,NewTerm,NewContext) %%%%%
%
% Flags:
%   lhs_Start. : typeTerm enters in the lhs of a rule
%                 (typeTerm doesnt take a new instance of the type for the func-symbol on the lhs)
%   lhs_Args. : typeTerm enters in an argument of the lhs of a rule
%	  	  (no func-calls allowed on lhs)
%   tc_shell. : typeTerm has been called from shell
%		  (all funcs are external)

% Term is Variable
typeTerm(Cx,Var,Type,Var,Cx2) :- Var='$VAR'(_),!,
	searchVarType(Var,Cx,Type,Cx2).

% Term is Integer
typeTerm(Cx,IntConst,int,IntConst,Cx):-integer(IntConst),!.

% Term is Float
typeTerm(Cx,FloatConst,float,FloatConst,Cx):- float(FloatConst),!.

% Term is Constructor
typeTerm(Cx,Term,ResType,NT,Cx2) :- Term=..[C|Args],constructor(C,ConsN,ConsType,_),!,
    typeFuncCall(Cx,C,ConsType,Args,ResType,NT,ConsN,Cx2),!.	

% function-call on lhs !
typeTerm(_,_,_,_,_) :- lhs_Args,!,
    writeTypeError("Function-calls on lhs of rule are not allowed",[]).

% Term is t1==t2
typeTerm(Cx,(T1==T2),bool,(NT1==NT2),Cx3) :- !,
	typeTerm(Cx,T1,Type1,NT1,Cx2),!,
	typeTerm(Cx2,T2,Type2,NT2,Cx3),!,
	ifc(unifytype(Type1,Type2),
		true,
		writeTypeError("Type of lhs and rhs of == do not match in term ~w",[T1==T2])
	).

% Term is t1=t2
typeTerm(Cx,(T1=T2),constraint,(NT1=NT2),Cx3) :- !,
	typeTerm(Cx,T1,Type1,NT1,Cx2),!,
	typeTerm(Cx2,T2,Type2,NT2,Cx3),!,
	ifc(unifytype(Type1,Type2),
		true,
		writeTypeError("Type of lhs and rhs of = do not match in term ~w",[T1=T2])
	).

% Term is t1<--t2
typeTerm(Cx,(T1<--T2),constraint,(NT1<--NT2),Cx3) :- !,
	typeTerm(Cx,T1,Type1,NT1,Cx2),!,
	typeTerm(Cx2,T2,Type2,NT2,Cx3),!,
	ifc(unifytype(Type1,Type2),
		true,
		writeTypeError("Type of lhs and rhs of <-- do not match in term ~w",[T1<--T2])
	).

% Term is F @ Arg (application)
typeTerm(Cx,(F @ Arg),Type,NT,Cx3) :- !,
    typeTerm(Cx,F,FType,NF,Cx2),
    typeTerm(Cx2,Arg,ArgType,NA,Cx3),!,
    ifc(unifytype(FType,(ArgType->ResType)),
       (Type=ResType,
	NT=(NF @ NA)),
       writeTypeError("Cannot apply '~w' with type '~w' to '~w' with type '~w' in term '~w'",[F,FType,Arg,ArgType,(F@Arg)])
       ).

% Term is function-call of a function that was not defined in the same component
%    We take a new instance of the func-type
typeTerm(Cx,Term,ResType,NT,Cx2) :-
    \+lhs_Start,       							% not for lhs !        
    Term=..[F|Args],function(F,FuncN,FuncType),!,
    typeFuncCall(Cx,F,FuncType,Args,ResType,NT,FuncN,Cx2).

% Term is function-call of a function that was defined in the same component
%   do *not* take a new instance of the func-type
typeTerm(Cx,Term,ResType,NT,Cx2) :-
    \+ tc_shell,								% not for shell-calls
    Term=..[F|Args],searchVarType(F,Cx,(_FuncN,FuncType),_),!,
    ifc(lhs_Start,(retract(lhs_Start),assertz(lhs_Args)),true),!,		% entering in an arg. of the lhs ?
    ifc(var(FuncType),
       (length(Args,FuncN),
	createNType(FuncN,FuncType)
	),
       true
       ),
    typeFuncCall(Cx,F,FuncType,Args,ResType,NT,FuncN,Cx2),!.

    createNType(0,_):-!.
    createNType(N,(_->Type)):-N2 is N-1,createNType(N2,Type).

% the head-symbol is unknown -> error
typeTerm(_Cx,Term,Type,_,_) :-
    functor(Term,H,_N),!,
    ifc(var(Type),
       writeTypeError("Function/constructor ~w is undefined in term ~w",[H,Term]),
       writeTypeError("Function/constructor ~w with resulttype ~w is undefined in term ~w",[H,Type,Term])
    ).

typeFuncCall(Cx,F,FuncType,Args,Type,NT,DefinedN,Cx2) :-!,
  typeFuncCall2(Cx,F,FuncType,Args,Type,NT2,Cx2),!,
  length(Args,Length),
  ifc(Length=DefinedN,
	  NT=NT2,
	  ifc(Length<DefinedN,
		  NT='$'(NT2),
          makeApplyTerm(NT2,DefinedN,NT))
      ).


makeApplyTerm(Term,Argnumber,ApplyTerm):-
  Term=..[Head|Args],
  split(Argnumber,Args,FirstN,Rest),
  Term1=..[Head|FirstN],
  addApplys(Term1,Rest,ApplyTerm).

  split(0,Xs,[],Xs).
  split(N,[X|Xs],[X|Firsts],Lasts):-N1 is N-1,split(N1,Xs,Firsts,Lasts).

  addApplys(Term,[],Term).
  addApplys(Term,[X|Xs],NewTerm):-addApplys(Term@X,Xs,NewTerm).


typeFuncCall2(Cx,F,T,[Arg|Args],Type,NT,Cx3) :-
	\+ var(T),T=(T1->T2),!,
	typeTerm(Cx,Arg,ArgType,NArg,Cx2),!,
	ifc(unifytype(ArgType,T1),
		(typeFuncCall2(Cx2,F,T2,Args,Type,NT2,Cx3),
		 NT2=..[F|NArgs],
	         NT=..[F,NArg|NArgs]),
	writeTypeError("Cannot apply '~w' to '~w' with type '~w'",[F,Arg,ArgType])
	).

typeFuncCall2(Cx,F,Type,[],Type,F,Cx) :- !.

typeFuncCall2(_Cx,F,_Type,_Args,_,_,_) :- !,
	writeTypeError("Too many arguments for '~w'",[F]).

%----------------------------------------------------------------
% searchVarType(Var,Context,Type,NewContext)
%   searchs the Type of Var in Context
%   if the Var doesnt exist in Context a new entry will be added to the Context

searchVarType(Var,[],Type,[(Var,Type)]) :- !.
searchVarType(Var,[(Var2,Type)|Cx],Type,[(Var2,Type)|Cx]) :-
	Var==Var2,!,
        ifc(lhs_Args,
		writeTypeError("Lhs of rule is nonlinear",[]),
		true
	).
searchVarType(Var,[Cx|Context],Type,[Cx|Context2]) :- searchVarType(Var,Context,Type,Context2).

% some help-functions

findRulesOfF(FuncRules,F,Rules):-findValue(F,FuncRules,Rules).

findBasicComponent([(Component,[])|_],Component):-!.
findBasicComponent([_|Components],Component):-findBasicComponent(Components,Component).

removeComponentRules(_,[],[]).
removeComponentRules(Component,[(Function,_)|OldRules],NewRules):-member(Function,Component),!,
		removeComponentRules(Function,OldRules,NewRules).
removeComponentRules(Component,[R|OldRules],[R|NewRules]):-
		removeComponentRules(Component,OldRules,NewRules).

removeComponentNodes(_Component,[],[]).
removeComponentNodes(Component,[(Component,[])|Nodes],Nodes2):-!,
		removeComponentNodes(Component,Nodes,Nodes2).
removeComponentNodes(Component,[(N,Edges)|Nodes],[(N,Edges2)|Nodes2]):-
		removeEdges(Component,Edges,Edges2),
		removeComponentNodes(Component,Nodes,Nodes2).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Where-Statements
%   create new functions for where-statements
%
%   createWheres(FuncRules,NewFuncRules)

createWheres([],[]):-!.
createWheres([(F,FRules)|Rules],[(F,NewFRules)|NewRules]):-
        name(F,FuncName),retractall(wherecounter(_)),
        createWheres2(FuncName,FRules,NewFRules,WhereRules),
        collectFuncRules(WhereRules,WhereRules2),       % <-------
        append(WhereRules2,Rules,ExtendedRules),        % <-------
        createWheres(ExtendedRules,NewRules).

collectFuncRules(OldRules,NewRules):-
        collectFuncRules(OldRules,[],NewRules).

collectFuncRules([],Result,Result).
collectFuncRules([(F,FRules)|Rules],ProcessedRules,Result):-
        appendFuncRules(F,FRules,ProcessedRules,NewProcessedRules),
        collectFuncRules(Rules,NewProcessedRules,Result).

appendFuncRules(F,FRules,[],[(F,FRules)]).
appendFuncRules(F,FRules,[(F,OldFRules)|Rules],[(F,NewFRules)|Rules]):-!,
        append(FRules,OldFRules,NewFRules).
appendFuncRules(F,FRules,[OtherRules|Rules],[OtherRules|Result]):-
        appendFuncRules(F,FRules,Rules,Result).



createWheres2(_FuncName,[],[],[]).
createWheres2(FuncName,[(L,R)|Rules],[(L,NewR)|NewRules],WhereRules):-
	retractall(currline(_)),assertz(currline(L=R)),
	createWheres3(FuncName,L,R,NewR,WhereRules1),
	createWheres2(FuncName,Rules,NewRules,WhereRules2),
	append(WhereRules1,WhereRules2,WhereRules).

createWheres3(_,_L,R,R,[]):- var(R),!.

createWheres3(FuncName,L,R2,NewR,WhereRules):- R2=whererule(R,Locals),!,
	sortWheres(Locals,PatWheres,FuncWheres),
	ifc(PatWheres=(PatWhereL=PatWhereR),
	     (		% new func-name for the where-rule
		append(FuncName,"_where",WhereString1),getwherecounter(N),name(N,StringN),
		append(WhereString1,StringN,WhereString),name(WhereName,WhereString),
			% new rhs for the old rule (now calling the where-rule)
	      	freeVars(L,ArgVars),NewR=..[WhereName,PatWhereR|ArgVars],
			% new where-rule
		LWhere=..[WhereName,PatWhereL|ArgVars],
		freeVars(LWhere,AllVars),
		createWheresForFunc(FuncName,AllVars,R,FuncWheres,RWhere,FuncWhereRules),
		assertz(fresh((LWhere,RWhere))),retract(fresh(FreshRule)),
		assertz(functionTreeFlag(WhereName,standard)),
		WhereRules=[(WhereName,[FreshRule])|FuncWhereRules]
             ),
	     (
		freeVars(L,AllVars),
                createWheresForFunc(FuncName,AllVars,R,FuncWheres,NewR,WhereRules)
	     )
         ).        
createWheres3(_,_L,R,R,[]).

sortWheres(Var,_,_):- var(Var),!,writeTypeError("Incorrect local definition",[]).
sortWheres(L=Right,PatWheres,FuncWheres):-
	% \+ var(Right),Right=((R;L2)=R2),!, % changed by MH into:
	nonvar(Right), Right=(RL2=R2), nonvar(RL2), RL2=(R;L2),!,
	sortWheres(L=R,PatWhere,FuncWhere),
	sortWheres(L2=R2,PatWheres2,FuncWheres2),
	ifc(PatWhere=empty,
	   PatWheres=PatWheres2,
           (PatWheres2=(L3=R3),PatWheres=((L,L3)=(R,R3)))
        ),
	append(FuncWhere,FuncWheres2,FuncWheres).
sortWheres(Left=Right,Left=Right,[]):- var(Left),!.
sortWheres(Left=Right,Left=Right,[]):- Left=..[C|_],constructor(C,_,_,_),!.
sortWheres(Left=Right,empty,[Left=Right]):- !.
sortWheres(_,_,_):- writeTypeError("Incorrect local definition",[]).


createWheresForFunc(FuncName,AllVars,Expr,AllLocalRules,NewExpr,WhereRules):-
	lambdaLifting(AllVars,Expr,AllLocalRules,AllLocalRules,NewExpr2,NewLocalRules),
	createFuncWheres(FuncName,NewLocalRules,NewExpr2,NewExpr,[],WhereRules).

lambdaLifting(_AllVars,Expr,AllLocalRules,[],Expr,AllLocalRules):-!.
lambdaLifting(AllVars,Expr,AllLocalRules,[L=R|LocalRules],LiftedExpr,LiftedRules):-
	freeVars(L,LeftVars),freeVars(R,RightVars),
	removeVars(LeftVars,RightVars,FreeVars1),commonVars(FreeVars1,AllVars,FreeVars),
	ifc(FreeVars=[],
	   lambdaLifting(AllVars,Expr,AllLocalRules,LocalRules,LiftedExpr,LiftedRules),
	   (L=..[F|_],
            addArgs(F,F,FreeVars,Expr,Expr2),
	    addArgs(F,F,FreeVars,AllLocalRules,AllLocalRules2),
            lambdaLifting(AllVars,Expr2,AllLocalRules2,AllLocalRules2,LiftedExpr,LiftedRules)
           )
        ).

createFuncWheres(_FuncName,[],Expr,Expr,WhereRules,WhereRules):- !.
createFuncWheres(FuncName,[L=R|LocalRules],Expr,NewExpr,WhereRules,NewWhereRules):-
		% new func-name for local function
	L=..[F|_],
	append(FuncName,"_where_",WhereString1),name(F,LocalString),
	append(WhereString1,LocalString,WhereString),name(WhereName,WhereString),
		% replace the old name by the new where-name
	addArgs(F,WhereName,[],[L=R|LocalRules],[NewL=NewR|NewLocalRules]),
	addArgs(F,WhereName,[],Expr,NewExpr2),
	addArgs(F,WhereName,[],WhereRules,NewWhereRules2),
		% the new where-rule
	assertz(fresh((NewL,NewR))),retract(fresh(FreshRule)),assertz(functionTreeFlag(WhereName,standard)),
	createFuncWheres(FuncName,NewLocalRules,NewExpr2,NewExpr,[(WhereName,[FreshRule])|NewWhereRules2],NewWhereRules).


   addArgs(_F,_NewF,_NewVars,Term,Term):- var(Term),!.
   addArgs(F,NewF,NewVars,Term,NewTerm):- Term=..[Head|Args],
	addArgsToList(F,NewF,NewVars,Args,NewArgs),
	ifc(Head=F,
	   (append(NewVars,NewArgs,NewArgs2),NewTerm=..[NewF|NewArgs2]),
	   NewTerm=..[Head|NewArgs]
        ).

   addArgsToList(_F,_NewF,_NewVars,[],[]):-!.
   addArgsToList(F,NewF,NewVars,[Arg|Args],[NewArg|NewArgs]):-
        addArgs(F,NewF,NewVars,Arg,NewArg),
        addArgsToList(F,NewF,NewVars,Args,NewArgs).

   commonVars([],_,[]):-!.
   commonVars([X|Xs],Ys,[X|Cs]):-varmember(X,Ys),!,commonVars(Xs,Ys,Cs).
   commonVars([_|Xs],Ys,Cs):-commonVars(Xs,Ys,Cs).

   removeVars(_Vars,[],[]):-!.
   removeVars(Vars,[V|Vs],Vs2):-varmember(V,Vars),!,removeVars(Vars,Vs,Vs2).
   removeVars(Vars,[V|Vs],[V|Vs2]):-removeVars(Vars,Vs,Vs2).

   getwherecounter(X):-
       ifc(wherecounter(X),
    	    (retract(wherecounter(X)),X1 is X+1,assertz(wherecounter(X1))),
	    (X=1,assertz(wherecounter(2)))
         ).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% Lambda-Expressions
%   create new functions for lambda-expressions
%
%   createLambdas(FuncRules,NewFuncRules)

createLambdas([],[]):-!.
createLambdas([(F,FRules)|Rules],[(F,NewFRules)|NewRules]):-
	name(F,FuncName),retractall(lambdacounter(_)),
	createLambdas2(FuncName,FRules,NewFRules,LambdaRules),
	append(LambdaRules,Rules,ExtendedRules),
	createLambdas(ExtendedRules,NewRules).
	
createLambdas2(_FuncName,[],[],[]).
createLambdas2(FuncName,[(L,R)|Rules],[(L,NewR)|NewRules],LambdaRules):-
	retractall(currline(_)),assertz(currline(L=R)),
	createLambdas3(FuncName,R,NewR,LambdaRules1),
	createLambdas2(FuncName,Rules,NewRules,LambdaRules2),
	append(LambdaRules1,LambdaRules2,LambdaRules).

% createLambdas3(FuncName,R,NewR,LambdaRules)

createLambdas3(_,Var,Var,[]):-var(Var),!.

createLambdas3(_,\Var,_,_):-var(Var),!,writeTypeError("Wrong definition of lambda-expression",[]).

createLambdas3(_,\ (Var->_Term),_,_):- \+ var(Var),!,writeTypeError("Variable after \\ expected",[]).


createLambdas3(FuncName,\ (Var->Term),NewTerm,LambdaRules):-!,
    splitUp(FuncName,Before,Behind),
    getFNPrefix(FuncName,Before,Behind,Prefix),
	   % neuen Funktionsnamen basteln in "Lambda"
    append(Prefix,"lambda",S0),
	getlambdacounter(N),name(N,StringN),append(S0,StringN,S1),
	append(S1,"_in_",S2),append(S2,FuncName,StringLambda),name(Lambda,StringLambda),
	   % Liste der freien Variablen in "VarList"
	freeVars(\ (Var->Term),VarList),
           % statt Lambdaausdruck -> Funktionsaufruf
        NewTerm=..[Lambda|VarList],
	   % neue Lambda-Regel (mit frischen Variablen)
	appendVar(VarList,Var,VarList2),L=..[Lambda|VarList2],R=Term,
	assertz(fresh(L,R)),retract(fresh(NL,NR)),LambdaRules=[(Lambda,[(NL,NR)])],
	assertz(functionTreeFlag(Lambda,standard)).

    splitUp([],[],[]).
    splitUp([95|Xs],[95],Xs).
    splitUp([X|Xs],[X|Before],Behind):-splitUp(Xs,Before,Behind).

    getFNPrefix(_FN,_Before,[],[]). % no "_" in the name at all
    getFNPrefix(FN,Before,_Behind,Prefix):-symbtab(T),
      name(FName,FN),
      if(member((FName,FName),T),  % Function is from the prelude... so leave it unchanged
         Prefix=[],
         Prefix=Before).


  
createLambdas3(_,\_Term,_,_):-!,writeTypeError("Wrong definition of lambda-expression",[]).

createLambdas3(FuncName,Term,NewTerm,LambdaRules):-
	Term=..[H,Arg|Args],!,createLambdas3(FuncName,Arg,NewArg,LambdaRules1),
	Term2=..[H|Args],createLambdas3(FuncName,Term2,Term3,LambdaRules2),
	Term3=..[H|NewArgs],
	NewTerm=..[H,NewArg|NewArgs],append(LambdaRules1,LambdaRules2,LambdaRules).

createLambdas3(_,Term,Term,[]).

   getlambdacounter(X):-
       ifc(lambdacounter(X),
    	    (retract(lambdacounter(X)),X1 is X+1,assertz(lambdacounter(X1))),
	    (X=1,assertz(lambdacounter(2)))
         ).
    
   appendVar([],Var,[Var]).
   appendVar([X|Xs],Var,[X|NXs]):-appendVar(Xs,Var,NXs).

   removeVar(_Var,[],[]).
   removeVar(Var,[V|Vs],Vs2):-Var==V,!,removeVar(Var,Vs,Vs2).
   removeVar(Var,[V|Vs],[V|NVs]):-removeVar(Var,Vs,NVs).


%%%%%%%%%%%%%%%%%%%%%%%%
% freeVars(Term,FreeVars) 
% freeVars(Term,LocalVars,UpToNowFreeVars,FreeVars) 
%
% - collects all free variables of Term
% - if a variable is found a) look if it is a local-variable in the
%   term we actually scan (then it will be in LocalVars)
%   b) look if has already been found as a free variable. Then it will
%   be in UpToNoFreeVars. If a) or b) don't collect it, otherwise do.
% - Local variables can appear a) by "local [X1,..Xn] in t" or
%   b) by \X->T (X is local to T)
%   As LocalVars is not given back, it will only work for the term for
%   which the variables are really local.
%   In a) report an error if local is not followed by either a single
%   variable or by a list of variables


   freeVars(Term,FreeVars):-freeVars(Term,[],[],FreeVars).

   freeVars(Var,Locals,UpToNow,NewV) :- var(Var),!,
	ifc(strictmember(Var,Locals),
            NewV=UpToNow,
            ifc(strictmember(Var,UpToNow),
		NewV=UpToNow,
                NewV=[Var|UpToNow])).
	 
  freeVars(choice([]),_Locals,UpToNow,UpToNow):-!.
  freeVars(choice([((L localVars C),B)|Choices]),Locals,UpToNow,V):-!,
  	freeVars(L localVars (C,B),Locals,UpToNow,V1),
    freeVars(choice(Choices),Locals,V1,V).
  freeVars(choice([(C,B)|Choices]),Locals,UpToNow,V):-!,
  	freeVars((C,B),Locals,UpToNow,V1),
    freeVars(choice(Choices),Locals,V1,V).


  
    freeVars(\ (Var->Term),Locals,UpToNow,V):-!,
	freeVars(Term,[Var|Locals],UpToNow,V).

   freeVars(List localVars Term,Locals,UpToNow,V):-var(List),!,
% appears during typechecking
	   freeVars(Term,[List|Locals],UpToNow,V).

   freeVars(List localVars Term,Locals,UpToNow,V):-!,
% appears while creating NF and during runtime
           append(List,Locals,NewLocals),
	   freeVars(Term,NewLocals,UpToNow,V).

   freeVars(Term,Locals,UpToNow,NewV) :-
         Term=..[H,Arg|Args],!,freeVars(Arg,Locals,UpToNow,V2),
         Term2=..[H|Args],freeVars(Term2,Locals,V2,NewV).

   freeVars(_Term,_Locals,V,V).



   varmember(Var,[V|_]):-Var==V,!.
   varmember(Var,[_|Vs]):-varmember(Var,Vs).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% erstellt zu einem Graphen eine Liste Kanten der Form (Quelle,Liste der Ziele)
% wobei Quelle und Ziel starke Zusammenhangskomponenten des Graphen sind
%

buildSuperStructGraph(G,SuperG):-
    buildSuperNodes(G,SuperNodes),
    buildSuperStructGraph(G,SuperNodes,SuperNodes,SuperG).

buildSuperStructGraph(_G,_SuperNodes,[],[]).
buildSuperStructGraph(G,SuperNodes,[Node|Nodes],[(Node,Edges2)|Edges3]):-
    buildEdgesofSuperNode(G,SuperNodes,Node,Node,Edges),makeSet(Edges,Edges2),
    buildSuperStructGraph(G,SuperNodes,Nodes,Edges3).

buildEdgesofSuperNode(_G,_SuperNodes,_SuperNode,[],[]).
buildEdgesofSuperNode(G,SuperNodes,SuperNode,[Node|Nodes],Edges):-
    findNeighs(Node,G,Neighs),
    buildEdgesofNode(G,SuperNodes,SuperNode,Node,Neighs,Edges2),
    buildEdgesofSuperNode(G,SuperNodes,SuperNode,Nodes,Edges3),
    append(Edges2,Edges3,Edges).


buildEdgesofNode(_G,_SuperNodes,_SuperNode,_Node,[],[]).
buildEdgesofNode(G,SuperNodes,SuperNode,Node,[Neigh|Neighs],Edges):-
    buildEdgesofNode(G,SuperNodes,SuperNode,Node,Neighs,Edges2),
    findSuperNode(Neigh,SuperNodes,NeighSuperNode),
    ifc(NeighSuperNode=SuperNode,                             % gleiche Komponente ?
       Edges=Edges2,
       Edges=[NeighSuperNode|Edges2]                    % nein -> Kante dahin ziehen
       ).

% gibt zu einem Graphen eine Liste der Zusammenhangskomponenten aus

buildSuperNodes(G,SuperNodes):-
    buildSuperNodes2(G,G,SuperNodes).

buildSuperNodes2([],_,[]).
buildSuperNodes2(_G,[],[]).
buildSuperNodes2(G,[(Node,_Edges)|Nodes],[SuperNode|SuperNodes]):-
    buildSuperStructure(Node,G,SuperNode),!,
    removeNodes(SuperNode,G,G2),
    buildSuperNodes2(G2,Nodes,SuperNodes).
buildSuperNodes2(G,[(_Node,_Edges)|Nodes],SuperNodes):-	   % dieser Fall bedeutet, dass Node nicht mehr vorkommt
    buildSuperNodes2(G,Nodes,SuperNodes).

% buildSuperStructure(Node,Graph,Result)
%    berechnet in Result die strenge Zusammenhangskomponente von Node bezueglich Graph

buildSuperStructure(Node,Graph,Result):-
    length(Graph,NumV),
    visit(Node,(Graph,[],0,[],NumV),_,_,Result).

visit(Node,State,NewState,Min,Result):-
    getId(State,Id),Id2 is Id+1,setId(State,Id2,State2),
    setVal(Node,Id2,State2,State3),
    getStack(State3,Stack),setStack(State3,[Node|Stack],State4),
    getGraph(State4,Graph),findNeighs(Node,Graph,Neighs),
    visitNeighs(Neighs,State4,Id2,State5,Min),
    getVal(Node,State5,Val),
    ifc(Val=Min,
       buildResult(Node,State5,NewState,Result),
       NewState=State5
       ).

visitNeighs([],State,Min,State,Min):-!.

visitNeighs([N|Ns],State,Min,NewState,NewMin):-
    getVal(N,State,Val),
    ifc(Val=0,
       visit(N,State,State2,M,_),
       (M=Val,State2=State)
       ),
    ifc(M<Min,
       Min2=M,
       Min2=Min
       ),
    visitNeighs(Ns,State2,Min2,NewState,NewMin).

buildResult(Node,State,NewState,[Top|R]):-
    getStack(State,[Top|Stack]),setStack(State,Stack,State2),
    getNumV(State2,NumV),V is NumV+1,
    setVal(Top,V,State2,State3),
    ifc(Node=Top,
       (NewState=State3,R=[]),
       buildResult(Node,State3,NewState,R)
       ).

% State
%   = (Graph,Stack,Id,Values,NumV)

getVal(Node,State,Val):-
    getValues(State,Values),
    findValue(Node,Values,Val),!.
getVal(_Node,_State,0).

setVal(Node,Val,State,NewState):-
    getValues(State,Values),
    replaceValue(Node,Val,Values,NewValues),
    setValues(State,NewValues,NewState).

setGraph(OldS,C,NewS):-setcomp(OldS,1,C,NewS).
getGraph(S,C):-getcomp(S,1,C).

setStack(OldS,C,NewS):-setcomp(OldS,2,C,NewS).
getStack(S,C):-getcomp(S,2,C).

setId(OldS,C,NewS):-setcomp(OldS,3,C,NewS).
getId(S,C):-getcomp(S,3,C).

setValues(OldS,C,NewS):-setcomp(OldS,4,C,NewS).
getValues(S,C):-getcomp(S,4,C).

setNumV(OldS,C,NewS):-setcomp(OldS,5,C,NewS).
getNumV(S,C):-getcomp(S,5,C).

setResult(OldS,C,NewS):-setcomp(OldS,6,C,NewS).
getResult(S,C):-getcomp(S,6,C).


getcomp((C,_S),1,C):-!.
getcomp(C,1,C):-!.
getcomp((_C,S),N,C2):-N2 is N-1,getcomp(S,N2,C2).

setcomp((_OldC,S),1,NewC,(NewC,S)):-!.
setcomp(_OldC,1,NewC,NewC):-!.
setcomp((C,OldS),N,NewC,(C,NewS)):-N2 is N-1,setcomp(OldS,N2,NewC,NewS).


test(G):-collectAllRules(R),buildCallGraph(R,G).

% buildCallGraph
% erstellt aus einer funcrules-Liste eine Liste der jeweils aufgerufenen Funktionen
% [(F,CalledFunctions),...]
% ausserdem wird fuer jeden Funktionsaufruf ein Fakt namens arity(F,N) generiert, wobei
% N die groesste Stelligkeit ist, die fuer F-Aufrufe gefunden wurde

buildCallGraph([],[]).
buildCallGraph([(F,FRules)|Rules],[(F,CallSet)|CallLs]):-
    buildCallList(F,FRules,CallList),makeSet(CallList,CallSet),
    buildCallGraph(Rules,CallLs).

buildCallList(_,[],[]).
buildCallList(F,[(_,R)|Rules],CallList):-
    findCalls(F,R,Calls),
    buildCallList(F,Rules,Calls2),
    append(Calls,Calls2,CallList).

findCalls(_,Term,[]):-var(Term),!.          % Variable
findCalls(F,Term,Calls):-		    % Konstruktor
    Term=..[C|Args],isCons(C,_),!,findLCalls(F,Args,Calls).
findCalls(F,Term,Calls):-                   % Selbstaufruf
    Term=..[F|Args],!,findLCalls(F,Args,Calls).
findCalls(F,Term,[Head|Calls]):-	    % Funktionsaufruf
    Term=..[Head|Args],findLCalls(F,Args,Calls).

findLCalls(_F,[],[]).
findLCalls(F,[Arg|Args],Calls):-
    findCalls(F,Arg,ArgCalls),
    findLCalls(F,Args,ArgsCalls),
    append(ArgCalls,ArgsCalls,Calls).

%addarity(F,Args):-
%    length(Args,L),
%    ifc(arity(F,OldL),
%       ifc(OldL<L,(retract(arity(F,OldL)),assertz(arity(F,L))),true),
%       assertz(arity(F,L))
%    ).


% testCallGraph(CallGraph,CallGraph,CheckedCallGraph)
% - testet, ob zu jedem Funktionsaufruf eine Funktionsdefinition existiert
%   wenn nicht, wird nachgeschaut, ob ein Typ fuer diese Funktion existiert (d.h. Funktion
%   aus einer anderen Datei)
%   wenn auch das nicht, wird ein Fehler ausgegeben
% - Aufrufe von den eingebauten Funktionen @, = und == werden rausgeworfen, ebenso die externen Funktionen

testCallGraph(_,[],[]).
testCallGraph(G,[(F,Calls)|CallList],[(F,NewCalls)|NewCallList]):-
    testCalls(G,F,Calls,NewCalls),
    testCallGraph(G,CallList,NewCallList).

testCalls(_G,_C,[],[]).
testCalls(G,C,[(@)|Calls],NewCalls):-!,testCalls(G,C,Calls,NewCalls).
testCalls(G,C,[(==)|Calls],NewCalls):-!,testCalls(G,C,Calls,NewCalls).
testCalls(G,C,[(=)|Calls],NewCalls):-!,testCalls(G,C,Calls,NewCalls).
testCalls(G,C,[F|Calls],[F|NewCalls]):-findCalledF(G,F),!,testCalls(G,C,Calls,NewCalls).
testCalls(G,C,[F|Calls],NewCalls):-function(F,_,_),!,testCalls(G,C,Calls,NewCalls).
testCalls(G,C,[F|Calls],NewCalls):-isExternal(F),!,testCalls(G,C,Calls,NewCalls).
testCalls(_,C,[F|_],_):-
    format("*** Error: function '~w' calls undefined function '~w'~n",[C,F]),fail.

findCalledF(G,F):-findValue(F,G,_).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% collectAllRules
% collect all rules in a list of type [(Functor,Regelliste),...]
%

collectAllRules([(F,FRules)|Rules]):-
    temprule(L,_),!,functor(L,F,_),
    collectRulesofF(F,FRules),
    collectAllRules(Rules).
collectAllRules([]).

collectRulesofF(F,[(L,R)|Rules]):-
    temprule(L,R),functor(L,F,_),!,
%    writeterm(L=R),nl,
    retract(temprule(L,R)),collectRulesofF(F,Rules).
collectRulesofF(_,[]).

%------------------------------------------------------------

% gibt die Zusammenhangskomponente aus, in der sich der Knoten befindet

findSuperNode(N,[Node|_],Node):-member(N,Node),!.
findSuperNode(N,[_|Nodes],Node):-findSuperNode(N,Nodes,Node).

% entfernt alle Knoten einer Liste aus einem Graphen
% (die Knoten selbst und die Kanten, die zu diesen Knoten fuehren)

removeNodes([],G,G).
removeNodes([Node|Nodes],G,NewG):-removeNode(Node,G,G2),removeNodes(Nodes,G2,NewG).

removeNode(_Node,[],[]).
removeNode(Node,[(Node,_)|Nodes],NewNodes):-!,removeNode(Node,Nodes,NewNodes).
removeNode(Node,[(N,Edges)|Nodes],[(N,NewEdges)|NewNodes]):-
    removeEdges(Node,Edges,NewEdges),removeNode(Node,Nodes,NewNodes).

removeEdges(Node,Edges,NewEdges):-removeElem(Node,Edges,NewEdges).

% replaceValue

replaceValue(Node,Value,[],[(Node,Value)]).
replaceValue(Node,Value,[(Node,_OldValue)|Values],[(Node,Value)|Values]):-!.
replaceValue(Node,Value,[V|Values],[V|NewValues]):-replaceValue(Node,Value,Values,NewValues).

% findNeighs

findNeighs(F,G,Neigh):-findValue(F,G,Neigh).

% makeSet
% makes list to set

makeSet(L,S):-makeSet(L,[],S).

makeSet([],S,S).
makeSet([X|Xs],Y,S):-member(X,Y),!,makeSet(Xs,Y,S).
makeSet([X|Xs],Y,S):-makeSet(Xs,[X|Y],S).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Tools
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% sucht in einer Liste der Form [(l,r),...] bei gegebenem l das passende r

findValue(Node,[(Node,Val)|_],Val):-!.
findValue(Node,[_|Values],Val):-findValue(Node,Values,Val).

% entfernt aus einer Liste alle Vorkommen eines bestimmten Elementes

removeElem(_Node,[],[]).
removeElem(Node,[Node|Nodes],NewNodes):-!,removeElem(Node,Nodes,NewNodes).
removeElem(Node,[N|Nodes],[N|NewNodes]):-removeElem(Node,Nodes,NewNodes).
	
%-----------------------------------------------------------------------------------------------------

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Unification of 2 types L and R *with* occur-check
% (detecting infinite types)

unifytype(L,R) :- var(L),var(R),!,
	L=R.
unifytype(L,R) :- var(L),!,
	occursnot(L,R,L,R),L=R.
unifytype(L,R) :- var(R),!,
	occursnot(L,R,R,L),R=L.
unifytype((L1->L2),(R1->R2)) :- !,
	unifytype(L1,R1),unifytype(L2,R2).
unifytype([],[]) :- !.
unifytype([L1|Ls],[R1|Rs]) :- !,
	unifytype(L1,R1),unifytype(Ls,Rs).
unifytype(L,R) :-
	L=..[Head|LArgs],R=..[Head|RArgs],!,unifytype(LArgs,RArgs).


occursnot(L,R,X,Y) :- var(Y),!,
	ifc(X==Y,occurcheckerror(L,R),true).
occursnot(L,R,X,(T1->T2)) :- !,
	occursnot(L,R,X,T1),occursnot(L,R,X,T2).
occursnot(_,_,_,[]) :- !.
occursnot(L,R,X,[T1|Ts]) :- !,
	occursnot(L,R,X,T1),occursnot(L,R,X,Ts).
occursnot(L,R,X,Type) :- Type=..[_|Args],!,
	occursnot(L,R,X,Args).

occurcheckerror(L,R) :-
	writeTypeError("Unification of type ~w with ~w would give infinite type",[L,R]).

%-----------------------------------------------------------------
% Tools

writeTypeError(Format,Args) :-!,
	currline(L),bindVarsInTerm([L|Args],NewArgs),
	append("~n*** Type-Error: ~w~n",Format,NewFormat),
	format(NewFormat,NewArgs),nl,fail.

writeterm(Term):-
	bindVarsInTerm(Term,NewTerm),
	write(NewTerm).

bindVarsInTerm(Term,NewTerm) :-
	assertz(writetermfact(Term)),retract(writetermfact(NewTerm)),numbervars(NewTerm,0,_).



%checkFunc(F,T):-write('############## Function '),write(F),nl,checkType(T).

%checkType(T):-var(T),!,write(T),nl.
%checkType(T):-T=..[H|B],write('Splitting '),write(T),nl,write(H),nl,checkTypeArgs(B).

%checkTypeArgs([]).
%checkTypeArgs([X|Xs]):-checkType(X),checkTypeArgs(Xs).
