----------------------------------------------------------------------
--- Implementation of CurryDoc, a utility for the automatic
--- generation of HTML documentation from Curry programs.
---
--- @author Michael Hanus
----------------------------------------------------------------------

-- * All comments to be put into the HTML documentation must be
--   prefixed by "--- " (also in literate programs!).
--
-- * The comment of a module must occur before the first "module" or
--   "import" line of this module.
--
-- * The comment of a function or datatype must occur before the
--   first definition of this function or datatype.
--
-- * The comments can contain at the end several special comments:
--   @cons id comment   --> a comment for a constructor of a datatype
--   @param id comment  --> comment for function parameter id
--                          (list all parameters in left-to-right order)
--   @return comment    --> comments for the return value of a function
--   @author comment    --> the author of a module (only in module comments)
--   @version comment   --> the version of a module (only in module comments)
--
-- * Current restriction: doesn't properly work for infix operator definitions
--   without a type definition (so it should be always included)

module currydoc where

import Char
import FlatCurry
import HTML
import System
import Time
import List
import Sort(mergeSort,leqStringIgnoreCase)
import Directory
import FileGoodies
import FlexRigid
import AnaOverlapping
import AnaCompleteness
import AnaIndeterminism
import AnaOpComplete
import CategorizedHtmlList
import Distribution

-- Version of currydoc
currydocVersion = "Version 0.3.5 of February 15, 2006"

-- Name of style sheet for documentation files:
currydocCSS = "currydoc.css"

--------------------------------------------------------------------------
-- Check arguments and call main function:
main = do
  args <- getArgs
  case args of
    [modname] -> makeCompleteDoc True ("DOC_"++stripSuffix (head args))
                                 (stripSuffix modname)
    [docdir,modname] -> makeCompleteDoc True docdir (stripSuffix modname)
    ["noindex",docdir,modname] ->
                        makeCompleteDoc False docdir (stripSuffix modname)
    ("onlyindex":docdir:modnames) ->
                        makeIndexPages docdir (map stripSuffix modnames)
    _ -> putStrLn $ "ERROR: Illegal arguments for currydoc: " ++
                    concat (intersperse " " args) ++ "\n" ++
                    "Usage: currydoc <module_name>\n" ++
                    "       currydoc <doc directory> <module_name>\n" ++
                    "       currydoc noindex <doc directory> <module_name>\n" ++
                    "       currydoc onlyindex <doc directory> <module_names>\n"

-- create directory if not existent:
createDir :: String -> IO ()
createDir dir = do
  exdir <- doesDirectoryExist dir
  if exdir then done else system ("mkdir "++dir) >> done

--------------------------------------------------------------------------
--- The main function of the CurryDoc utility.
--- @param withindex - True if the index pages should also be generated
--- @param docdir - the directory name containing all documentation files
--- @param modname - the name of the main module to be documented
makeCompleteDoc :: Bool -> String -> String -> IO ()
makeCompleteDoc withindex docdir modname = do
  putStrLn("CurryDoc ("++currydocVersion++") - the Curry Documentation Tool\n")
  prepareDocDir docdir
  -- parsing source program:
  callFrontendWithParams FCY [] modname
  (alltypes,allfuns,allops) <- readFlatCurryWithImports [modname]
  progname <- findSourceFileInLoadPath modname
  makeDocIfNecessary docdir
                     (genAnaInfo (Prog modname [] alltypes allfuns allops))
                     progname
  time <- getClockTime
  if withindex
   then genMainIndexPage     time docdir [modname] >>
        genFunctionIndexPage time docdir allfuns >>
        genConsIndexPage     time docdir alltypes
   else done
  -- change access rights to readable for everybody:
  system ("chmod -R go+rX "++docdir)
  done

--- Generate only the index pages for a list of (already compiled!) modules:
makeIndexPages :: String -> [String] -> IO ()
makeIndexPages docdir modnames = do
  putStrLn("CurryDoc ("++currydocVersion++") - the Curry Documentation Tool\n")
  prepareDocDir docdir
  (alltypes,allfuns,_) <- readFlatCurryWithImports modnames
  time <- getClockTime
  genMainIndexPage     time docdir modnames
  genFunctionIndexPage time docdir allfuns
  genConsIndexPage     time docdir alltypes
  -- change access rights to readable for everybody:
  system ("chmod -R go+rX "++docdir)
  done

-- create documentation directory (if necessary) with gifs and stylesheets:
prepareDocDir docdir = do
  createDir docdir
  putStrLn ("Copying icons into documentation directory \""++docdir++"\"...")
  pakcshome <- getEnviron "PAKCSHOME"
  -- copying all icons:
  system ("cp "++pakcshome++"/tools/icons/*.gif "++docdir)
  -- copy style sheet:
  system ("cp "++pakcshome++"/tools/currydoc.css "++docdir)


-- generate all analysis infos:
genAnaInfo prog =
  AnaInfo (getFunctionInfo (analyseOverlappings prog))
          (getFunctionInfo (analyseCompleteness prog))
          (getFunctionInfo (analyseIndeterminism prog))
          (getFunctionInfo (analyseOpCompleteness prog))

-- generate documentation for a single module:
makeDoc :: String -> AnaInfo -> String -> IO ()
makeDoc docdir anainfo progname =
 do putStrLn ("Reading comments from file \""++progname++".curry\"...")
    (modcmts,progcmts) <- readComments (progname++".curry")
    makeDocWithComments docdir anainfo progname modcmts progcmts

makeDocWithComments docdir anainfo progname modcmts progcmts = do
  putStrLn ("Reading FlatCurry program \""++progname++".fcy\"...")
  time <- getClockTime
  (imports,hexps) <- generateHtmlDocs time anainfo progname modcmts progcmts
  putStrLn ("Writing documentation to \""++docdir++"/"++
            getLastName progname++".html\"...")
  writeFile (docdir++"/"++getLastName progname++".html")
            (showDocCSS ("Module "++getLastName progname) hexps)
  translateSource2ColoredHtml docdir progname
  mapIO_ (makeDocIfNecessary docdir anainfo) imports

--- Generates the documentation for a module if it is necessary.
--- I.e., the documentation is generated if no previous documentation
--- file exists or if the existing documentation file is older than
--- the FlatCurry file.
makeDocIfNecessary :: String -> AnaInfo -> String -> IO ()
makeDocIfNecessary docdir anainfo modname = do
  progname <- findSourceFileInLoadPath modname
  docexists <- doesFileExist (docdir++"/"++getLastName progname++".html")
  if not docexists
   then copyOrMakeDoc docdir anainfo progname 
   else getModificationTime (progname++".fcy") >>= \ctime ->
        getModificationTime
             (docdir++"/"++getLastName progname++".html") >>= \htime ->
        if ctime>htime
         then copyOrMakeDoc docdir anainfo progname
         else getImports progname >>= \imports ->
              mapIO_ (makeDocIfNecessary docdir anainfo) imports

-- get imports of a program by reading the interface, if possible:
getImports progname = do
  fintexists <- doesFileExist (progname++".fint")
  if fintexists
    then do (Prog _ imports _ _ _) <- readFlatCurryFile (progname++".fint")
            return imports
    else do (Prog _ imports _ _ _) <- readFlatCurryFile (progname++".fcy")
            return imports

copyOrMakeDoc :: String -> AnaInfo -> String -> IO ()
copyOrMakeDoc docdir anainfo progname = do
  hasCopied <- copyDocIfPossible docdir progname
  if hasCopied then done
               else makeDoc docdir anainfo progname

--- Copy the documentation file from standard documentation directoy "CDOC"
--- (used for documentation of system libraries) if possible.
--- Returns true if the copy was possible.
copyDocIfPossible :: String -> String -> IO Bool
copyDocIfPossible docdir progname =
  let docprogname = getDirName progname++"/CDOC/"++getLastName progname in
  do docexists <- doesFileExist (docprogname++".html")
     if not docexists
      then return False
      else
       do ctime <- getModificationTime (progname++".fcy")
          htime <- getModificationTime (docprogname++".html")
          if ctime>htime
           then return False
           else
            do putStrLn ("Copying doc file from "++docprogname++".html")
               system ("cp "++docprogname++".html "++docdir)
               system ("cp "++docprogname++"_curry.html "++docdir)
               return True


generateHtmlDocs time anainfo progname modcmts progcmts = do
  (Prog _ imports types functions ops) <- readFlatCurryFile (progname++".fcy")
  return $
     (imports,
      [h1 [htxt ("Module \""),
           href (getLastName progname++"_curry.html")
                [htxt (getLastName progname++".curry")],
           htxt "\""]] ++
      genHtmlModule modcmts ++
      bigHRule "Exported names:" ++
      genHtmlExportIndex (getExportedTypes types)
                         (getExportedCons types)
                         (getExportedFuns functions) ++
      bigHRule "Summary of exported functions:" ++
      [HtmlStruct "table" [("border","1"),("width","100%")]
       (map (\ht->HtmlStruct "tr" [] [HtmlStruct "td" [] [ht]])
          (concatMap (genHtmlFuncShort progcmts anainfo)
                     functions))] ++
      bigHRule "Imported modules:" ++
      concatMap (\i->[href (getLastName i++".html") [htxt i],
                      breakline]) imports ++
      bigHRule "Exported datatypes:" ++
      concatMap (genHtmlType progcmts) types ++
      bigHRule "Exported functions:" ++
      concatMap (genHtmlFunc progname progcmts anainfo ops)
                functions ++
      curryDocEpilog time)
 where
    deleteFirstHrule [] = []
    deleteFirstHrule ((_:dt1,dd1):dl) = (dt1,dd1):dl

--- generate HTML index for all exported names:
genHtmlExportIndex exptypes expcons expfuns =
  (if htmltypes==[]
   then []
   else [par ([bold [htxt "Datatypes:"], breakline] ++
              intersperse (htxt " | ") htmltypes)] ) ++
  (if htmlcons==[]
   then []
   else [par ([bold [htxt "Constructors:"], breakline] ++
              intersperse (htxt " | ") htmlcons)] ) ++
  (if htmlfuns==[]
   then []
   else [par ([bold [htxt "Functions:"], breakline] ++
              intersperse (htxt " | ") htmlfuns)] )
 where
  htmltypes = map (\n->href ('#':n++"_TYPE") [htxt n])
                  (rmdups (sortStrings exptypes))
  htmlcons  = map (\n->href ('#':n) [htxt n])
                  (rmdups (sortStrings expcons))
  htmlfuns  = map (\n->href ('#':n++"_SHORT") [htxt n])
                  (rmdups (sortStrings expfuns))

  -- remove subsequent duplicates:
  rmdups [] = []
  rmdups [x] = [x]
  rmdups (x:y:xs) = if x==y then rmdups (y:xs)
                            else x : rmdups (y:xs)

-- extract all exported types
getExportedTypes :: [TypeDecl] -> [String]
getExportedTypes types = concatMap getExpType types
 where
   getExpType (Type (_,name) vis _ _) = if vis==Public then [name] else []
   getExpType (TypeSyn (_,name) vis _ _) = if vis==Public then [name] else []

-- extract all exported constructors
getExportedCons :: [TypeDecl] -> [String]
getExportedCons types =
   map (\(Cons (_,name) _ _ _)->name)
       (filter (\(Cons _ _ vis _)->vis==Public) (concatConsDecls types))
 where
   concatConsDecls [] = []
   concatConsDecls (TypeSyn _ _ _ _ : ts) = concatConsDecls ts
   concatConsDecls (Type _ _ _ cdcls : ts) = cdcls ++ concatConsDecls ts

-- extract all exported functions
getExportedFuns :: [FuncDecl] -> [String]
getExportedFuns funs = map (\(Func (_,name) _ _ _ _)->name)
                           (filter (\(Func _ _ vis _ _)->vis==Public) funs)


--- generate HTML documentation for a module:
genHtmlModule modcmts =
  let (maincmt,avcmts) = splitComment modcmts
   in [par [HtmlText maincmt]] ++
      map (\a->par [bold [htxt "Author: "], htxt a])
          (getCommentType "author" avcmts) ++
      map (\a->par [bold [htxt "Version: "], htxt a])
          (getCommentType "version" avcmts)
       
--- generate HTML documentation for a datatype if it is exported:
genHtmlType progcmts (Type (_,tcons) tvis tvars constrs) =
  if tvis==Public
  then
   let (datacmt,conscmts) = splitComment (getDataComment tcons progcmts)
    in [h3 [anchor (tcons++"_TYPE") [htxt tcons]],
        par [HtmlText datacmt],
        par [italic [htxt "Constructors:"], breakline,
             dlist (concatMap
                          (genHtmlCons (getCommentType "cons" conscmts))
                           constrs)],
        hrule]
  else []
 where
  genHtmlCons conscmts (Cons (cmod,cname) _ cvis argtypes) =
    if cvis==Public
    then [([anchor cname [bold [htxt cname]],
            code [HtmlText
             (" :: " ++
              concatMap (\t->" "++showType cmod True t++" -> ") argtypes ++
              tcons ++ concatMap (\i->[' ',chr (97+i)]) tvars)]],
           [par (getConsComment conscmts cname)])]
    else []

  getConsComment [] _ = []
  getConsComment (conscmt:conscmts) cname =
    let (consname,conscomment) = span isIdChar conscmt
     in if consname == cname
        then [code [htxt consname], HtmlText conscomment]
        else getConsComment conscmts cname

genHtmlType progcmts (TypeSyn (tcmod,tcons) tvis tvars texp) =
  if tvis==Public
  then let (typecmt,_) = splitComment (getDataComment tcons progcmts) in
       [h3 [anchor (tcons++"_TYPE") [htxt tcons]],
        par [HtmlText typecmt],
        par [italic [htxt "Type synonym: "],
             if tcons=="String" && tcmod=="Prelude"
             then code [htxt "String = [Char]"]
             else code [HtmlText
                   (tcons ++ concatMap (\i->[' ',chr (97+i)]) tvars ++ " = " ++
                    showType tcmod False texp)]],
        hrule]
  else []

-- get comment for a type name:
getDataComment _ [] = ""
getDataComment n ((DataDef d, cmt):fdcmts) =
   if n == d then cmt
             else getDataComment n fdcmts 
getDataComment n ((FuncDef _,_):fdcmts) = getDataComment n fdcmts


-- get all comments of a particular type (e.g., "param", "cons"):
getCommentType ctype cmts = map snd (filter (\c->fst c==ctype) cmts)


-- generate short HTML documentation for a function if it is exported:
genHtmlFuncShort progcmts anainfo (Func (fmod,fname) _ fvis ftype rule) =
  if fvis==Public
  then [table
         [[[anchor (fname++"_SHORT")
                   [href ('#':fname) [bold [htxt (showId fname)]]],
            code [HtmlText ("&nbsp;&nbsp;::&nbsp;"
                            ++ showType fmod False ftype)],
            HtmlText "&nbsp;&nbsp;"]
            ++ genFuncPropIcons anainfo (fmod,fname) rule],
          [[HtmlText (concat (take 10 (repeat "&nbsp;")) ++
             firstSentence (fst (splitComment (getFuncComment progcmts))))]]]
       ]
  else []
 where
  showId name = if isAlpha (head name) then name
                                       else ('(':name)++")"

  getFuncComment [] = ""
  getFuncComment ((FuncDef f, cmt):fdcmts) = if fname == f
                                             then cmt
                                             else getFuncComment fdcmts
  getFuncComment ((DataDef _,_):fdcmts) = getFuncComment fdcmts


-- generate HTML documentation for a function if it is exported:
genHtmlFunc progname progcmts anainfo ops
            (Func (fmod,fname) _ fvis ftype rule) =
  if fvis==Public
  then let (funcmt,paramcmts) = splitComment (getFuncComment progcmts)
        in [HtmlStruct "font" [("size","+1")]
            [anchor fname
                    [href (getLastName progname++"_curry.html#"++fname)
                          [bold [htxt (showId fname)]]],
              code [HtmlText ("&nbsp;::&nbsp;"++ showType fmod False ftype)]],
            HtmlText "&nbsp;&nbsp;"] ++
           genFuncPropIcons anainfo (fmod,fname) rule ++
           [par [HtmlText funcmt]] ++
           genParamComment paramcmts ++
           -- show further infos for this function, if present:
           (if furtherInfos == []
            then []
            else [dlist [([italic [htxt "Further infos:"]],
                          [ulist furtherInfos])]] ) ++
           [hrule]
  else []
 where
  showId (c:cs) = if isAlpha c then c:cs
                               else ('(':c:cs)++")"

  furtherInfos = genFuncPropComments anainfo (fmod,fname) rule ops

  getFuncComment [] = ""
  getFuncComment ((FuncDef f, cmt):fdcmts) = if fname == f
                                             then cmt
                                             else getFuncComment fdcmts
  getFuncComment ((DataDef _,_):fdcmts) = getFuncComment fdcmts

  genParamComment paramcmts =
    let params = map (span isIdChar) (getCommentType "param" paramcmts)
     in (if params==[]
         then []
         else [par [italic [HtmlText "Example call:&nbsp; "],
                    code [htxt (showCall fname (map fst params))]],
               dlist ([([italic [htxt "Parameters:"]],[])] ++
                      map (\(parid,parcmt)->
                                   ([],[code [htxt parid], HtmlText parcmt]))
                          params)
              ]) ++
         [dlist (map (\rescmt->([italic [htxt "Returns:"]],[HtmlText rescmt]))
                     (getCommentType "return" paramcmts))
         ]

  showCall f params =
    if isAlpha (head f) || length params /= 2
    then "(" ++ showId f ++ concatMap (" "++) params ++ ")"
    else "(" ++ params!!0 ++ " " ++ f ++ " " ++ params!!1 ++ ")"

--------------------------------------------------------------------------
--- Generates icons for particular properties of functions.
genFuncPropIcons anainfo fname rule =
   [detIcon, HtmlText "&nbsp;", flexRigidIcon rule]
 where
   --(non)deterministically defined property:
   detIcon =
    if getOverlappingInfo anainfo fname
    then href "index.html#nondet_explain"
              [addIconParams $ image "nondet.gif" "non-deterministic"]
    else href "index.html#det_explain"
              [addIconParams $ image "det.gif" "deterministic"]

   -- icon for rigid/flexible:
   flexRigidIcon (External _) = htxt ""
   flexRigidIcon (Rule _ rhs) = imageEvalAnnot (getFlexRigid rhs)
    where
      imageEvalAnnot ConflictFR = --bold [htxt "?"] --mixed rigid flexible
          href "index.html#flexrigid_explain"
               [addIconParams $ image "flexrigid.gif" "flexible+rigid"]
      imageEvalAnnot UnknownFR  = htxt ""
      imageEvalAnnot KnownRigid =
          href "index.html#rigid_explain"
               [addIconParams $ image "rigid.gif" "rigid"]
      imageEvalAnnot KnownFlex  =
          href "index.html#flex_explain"
               [addIconParams $ image "flex.gif" "flexible"]

addIconParams hicon = hicon `addAttr` ("align","middle")
                            `addAttr` ("border","0")

--------------------------------------------------------------------------
--- Generates further textual infos about particular properties
--- of a function. The result is a list of HTML expressions to be
--- formatted (if not empty) as some HTML list.
genFuncPropComments anainfo fname rule ops =
   filter (/=[]) [genFixityInfo fname ops,
                  completenessInfo,
                  indeterminismInfo,
                  opcompleteInfo,
                  externalInfo rule]
 where
   -- comment about the definitional completeness of a function:
   completenessInfo =
      let ci = getCompleteInfo anainfo fname
       in if ci==Complete
          then []
          else [htxt (if ci==InComplete
                      then "incompletely defined"
                      else
           "incompletely defined in each disjunction (but might be complete)")]

   -- comment about the indeterminism of a function:
   indeterminismInfo = if getIndetInfo anainfo fname
                       then [htxt "might behave indeterministically"]
                       else []

   -- comment about the indeterminism of a function:
   opcompleteInfo =
      if getOpCompleteInfo anainfo fname
      then [htxt "solution complete, i.e., able to compute all solutions"]
      else []

   -- comment about the external definition of a function:
   externalInfo (External _) = [htxt "externally defined"]
   externalInfo (Rule _ _)   = []


--- Generates a comment about the associativity and precedence
--- if the name is defined as an infix operator.
genFixityInfo fname ops =
    concatMap (\(Op n fix prec)->
                  if n==fname
                  then [htxt ("defined as "++showFixity fix++
                              " infix operator with precedence "++show prec)]
                  else [])
              ops
 where
  showFixity InfixOp  = "non-associative"
  showFixity InfixlOp = "left-associative"
  showFixity InfixrOp = "right-associative"


--------------------------------------------------------------------------
-- Pretty printer for types in Curry syntax:
-- second argument is True iff brackets must be written around complex types
showType :: String -> Bool -> TypeExpr -> String
showType _ _ (TVar i) = [chr (97+i)]
showType mod nested (FuncType t1 t2) =
   brackets nested
    (showType mod (isFunctionType t1) t1 ++ " -&gt; " ++ showType mod False t2)
showType mod nested (TCons tc ts)
 | ts==[]  = showTypeCons mod tc
 | tc==("Prelude","[]") && (head ts == TCons ("Prelude","Char") [])
   = "String"
 | tc==("Prelude","[]")
   = "[" ++ showType mod False (head ts) ++ "]" -- list type
 | take 2 (snd tc) == "(,"                      -- tuple type
   = "(" ++ concat (intersperse "," (map (showType mod False) ts)) ++ ")"
 | otherwise
   = brackets nested
      (showTypeCons mod tc ++ " " ++
       concat (intersperse " " (map (showType mod True) ts)))

isFunctionType (TVar _)       = False
isFunctionType (FuncType _ _) = True
isFunctionType (TCons _ _)    = False

showTypeCons mod (mtc,tc) =
  if mtc == "Prelude"
  then tc --"<A HREF=\"Prelude.html#"++tc++"_TYPE\">"++tc++"</A>"
  else
    if mod == mtc
    then "<A HREF=\"#"++tc++"_TYPE\">"++tc++"</A>"
    else "<A HREF=\""++mtc++".html#"++tc++"_TYPE\">"++tc++"</A>"

-- if first argument is True, put brackets around second argument:
brackets :: Bool -> String -> String
brackets False s = s
brackets True  s = "("++s++")"


-- Split a comment into its main part and parts preceded by "@...":
-- Example: splitComment "aaaa\nbbbb\n@param xxxx\n@return yyyy"
--          = ("aaaa\nbbbb",[("param","xxxx"),("return","yyyy")])

splitComment :: String -> (String,[(String,String)])
splitComment cmt = splitCommentMain (lines cmt)

splitCommentMain [] = ("",[])
splitCommentMain (l:ls) =
  if l=="" || head l /= '@'
  then let (maincmt,rest) = splitCommentMain ls
        in (l++('\n':maincmt),rest)
  else ([],splitCommentParams (takeWhile isAlpha (tail l))
                              (dropWhile isAlpha (tail l)) ls)

splitCommentParams param paramcmt [] = [(param,skipWhiteSpace paramcmt)]
splitCommentParams param paramcmt (l:ls) =
  if l=="" || head l /= '@'
  then splitCommentParams param (paramcmt++('\n':l)) ls
  else ((param,skipWhiteSpace paramcmt)
        : splitCommentParams (takeWhile isAlpha (tail l))
                             (dropWhile isAlpha (tail l)) ls)


--------------------------------------------------------------------------
-- read the comments of a source file to be put in the HTML documentation
readComments filename =
 do prog <- readFile filename
    return (groupLines . filter (/=OtherLine) . map classifyLine . lines
            $ prog)

--- This datatype is used to classify all input lines.
--- @cons Comment   - a comment for CurryDoc
--- @cons FuncDef   - a definition of a function
--- @cons DataDef   - a definition of a datatype
--- @cons ModDef    - a line containing a module definition
--- @cons OtherLine - a line not relevant for CurryDoc
data SourceLine = Comment String  -- a comment for CurryDoc
                | FuncDef String  -- a definition of a function
                | DataDef String  -- a definition of a datatype
                | ModDef          -- a line containing a module definition
                | OtherLine       -- a line not relevant for CurryDoc

-- classify a line of the source program:
-- here we replace blank line comments by a "breakline" tag
classifyLine :: String -> SourceLine
classifyLine line
 | take 3 line == "---" && all isSpace (drop 3 line) = Comment "<BR>"
 | take 4 line == "--- "     = Comment (drop 4 line)
 | take 7 line == "module "  = ModDef
 | take 7 line == "import "  = ModDef
 | otherwise = let id1 = getFirstId line
                in if id1==""
                    then OtherLine
                    else if id1=="data" || id1=="type"
                          then DataDef (getDatatypeName line)
                          else FuncDef id1
 where
   getDatatypeName = takeWhile isIdChar . dropWhile (==' ') . dropWhile isIdChar

-- get the first identifier (name or operator in brackets) in a string:
getFirstId [] = ""
getFirstId (c:cs) | isAlpha c = takeWhile isIdChar (c:cs)
                  | c=='('    = takeWhile (/=')') cs
                  | otherwise = ""

-- is an alphanumeric character, underscore, or apostroph?
isIdChar c = isAlphaNum c || c=='_' || c=='\''


-- group the classified lines into module comment and list of
-- (Func/DataDef,comment) pairs:
groupLines :: [SourceLine] -> (String,[(SourceLine,String)])
groupLines sls =
  let (modcmts,progcmts) = break (==ModDef) sls
   in if progcmts==[]
      then ("", groupProgLines sls)
      else (concatMap getComment modcmts,
            groupProgLines (filter (/=ModDef) (tail progcmts)))
 where
   getComment (Comment cmt) = cmt++"\n"
   getComment (FuncDef _)   = ""  -- this case should usually not occur
   getComment (DataDef _)   = ""  -- this case should usually not occur

groupProgLines :: [SourceLine] -> [(SourceLine,String)]
groupProgLines [] = []
groupProgLines (Comment cmt : sls) = groupComment cmt sls
groupProgLines (FuncDef f : sls) = (FuncDef f, "") : skipFuncDefs f sls
groupProgLines (DataDef d : sls) = (DataDef d, "") : skipDataDefs d sls

groupComment _ [] = []  -- comment not followed by definition -> ignore
groupComment cmt (Comment cmt1 : sls) = groupComment (cmt++"\n"++cmt1) sls
groupComment cmt (FuncDef f : sls) = (FuncDef f, cmt) : skipFuncDefs f sls
groupComment cmt (DataDef d : sls) = (DataDef d, cmt) : skipDataDefs d sls

skipFuncDefs _ [] = []
skipFuncDefs _ (Comment cmt : sls) = groupProgLines (Comment cmt : sls)
skipFuncDefs _ (DataDef d   : sls) = groupProgLines (DataDef d   : sls)
skipFuncDefs f (FuncDef f1  : sls) =
  if f==f1 then skipFuncDefs f sls
           else groupProgLines (FuncDef f1 : sls)

skipDataDefs _ [] = []
skipDataDefs _ (Comment cmt : sls) = groupProgLines (Comment cmt : sls)
skipDataDefs _ (FuncDef f   : sls) = groupProgLines (FuncDef f   : sls)
skipDataDefs d (DataDef d1  : sls) =
  if d==d1 then skipDataDefs d sls
           else groupProgLines (DataDef d1 : sls)


--------------------------------------------------------------------------
-- translate source file into HTML file with syntax coloring
translateSource2ColoredHtml :: String -> String -> IO ()
translateSource2ColoredHtml docdir progname = do
    let output = docdir++"/"++getLastName progname++"_curry.html"         
    putStrLn ("Writing source file as HTML to \""++output++"\"...") 
    callFrontendWithParams HTML [Quiet,OutFile output] progname

-- translate source file into HTML file with anchors for each function:
translateSource2AnchoredHtml :: String -> String -> IO ()
translateSource2AnchoredHtml docdir progname =
 do putStrLn ("Writing source file as HTML to \""++docdir++"/"++getLastName progname++"_curry.html\"...")
    prog <- readFile (progname++".curry")
    writeFile (docdir++"/"++getLastName progname++"_curry.html")
              (showDocCSS (progname++".curry")
                        [HtmlStruct "PRE" []
                              [HtmlText (addFuncAnchors [] (lines prog))]])

-- add the anchors to the classified lines and translate back:
-- first argument: list of already added anchors
-- second argument: list of source lines
addFuncAnchors :: [String] -> [String] -> String
addFuncAnchors _ [] = ""
addFuncAnchors ancs (sl : sls) = let id1 = getFirstId sl in
  if id1=="" ||
     id1 `elem` ["data","type","import","module","infix","infixl","infixr"]
  then htmlQuote (sl++"\n") ++ addFuncAnchors ancs sls
  else if id1 `elem` ancs
       then (sl++"\n") ++ addFuncAnchors ancs sls
       else "<A NAME=\""++id1++"\"></A>"
            ++ htmlQuote (sl++"\n")
            ++ addFuncAnchors (id1:ancs) sls


--------------------------------------------------------------------------
-- generate the index page for the documentation directory:
genMainIndexPage time docdir modnames =
 do putStrLn ("Writing index page to \""++docdir++"/index.html\"...")
    writeFile (docdir++"/index.html")
              (showDocCSS ("Documentation of Curry modules")
                          (htmlIndex modnames ++ curryDocEpilog time))

htmlIndex modnames =
  (if length modnames == 1
   then [h1 [htxt "Documentation of the Curry program ",
            href (head modnames++".html") [htxt (head modnames++".curry")]]]
   else [h1 [htxt "Documentation of the Curry programs:"],
         ulist (map (\m->[href (m++".html") [htxt (m++".curry ")]])
                    (mergeSort leqStringIgnoreCase modnames))]
  ) ++
  [ulist [[href "findex.html" [htxt "All functions"]],
          [href "cindex.html" [htxt "All constructors"]]],
   bold [htxt "Explanations of the icons used in the documentation:"],
   par [anchor "det_explain" [image "det.gif" "deterministic"],
        htxt " Function is deterministically defined, i.e.,",
        htxt " patterns are pairwise exclusive"],
   par [anchor "nondet_explain" [image "nondet.gif" "non-deterministic"],
        htxt " Function is non-deterministically defined, i.e.,",
        htxt " contains overlapping patterns"],
   par [anchor "rigid_explain" [image "rigid.gif" "rigid"],
        htxt " Function is rigid"],
   par [anchor "flex_explain" [image "flex.gif" "flexible"],
        htxt " Function is flexible"],
   par [anchor "flexrigid_explain" [image "flexrigid.gif" "flexible+rigid"],
        htxt " Function is partially flexible and partially rigid"]
   --par [image "impl.gif" "implementation",
   --     htxt " Reference to the implementation of the module or function"]
  ]
   

--------------------------------------------------------------------------
-- generate the function index page for the documentation directory:
genFunctionIndexPage time docdir funs = do
  putStrLn ("Writing function index page to \""++docdir++"/findex.html\"...")
  writeFile (docdir++"/findex.html")
     (showDocCSS "Index to all functions"
         (htmlFuncIndex (sortNames expfuns) ++
          curryDocEpilog time))
 where
   expfuns = map (\(Func name _ _ _ _)->name)
                 (filter (\(Func _ _ vis _ _)->vis==Public) funs)

htmlFuncIndex :: [(String,String)] -> [HtmlExp]
htmlFuncIndex qnames =
   [h1 [htxt "Index to all functions"]] ++
   categorizeByItemKey (map showModNameRef qnames)
   
showModNameRef :: (String,String) -> (String,[HtmlExp])
showModNameRef (modname,name) =
  (name,
   [href (modname++".html#"++name) [htxt name], nbsp, nbsp,
    htxt "(", href (getLastName modname++".html") [htxt modname], htxt ")"]
  )

sortNames names = mergeSort (\(_,n1) (_,n2)->leqStringIgnoreCase n1 n2) names


--------------------------------------------------------------------------
-- generate the constructor index page for the documentation directory:
genConsIndexPage time docdir types = do
  putStrLn ("Writing constructor index page to \""++docdir++"/cindex.html\"...")
  writeFile (docdir++"/cindex.html")
    (showDocCSS "Index to all constructors"
         (htmlConsIndex (sortNames expcons) ++
          curryDocEpilog time))
 where
   expcons = map (\(Cons name _ _ _)->name)
                 (filter (\(Cons _ _ vis _)->vis==Public)
                         (concatMap getCons types))

   getCons (Type _ _ _ cdecls) = cdecls
   getCons (TypeSyn _ _ _ _) = []

htmlConsIndex qnames =
   [h1 [htxt "Index to all constructors"]] ++
   categorizeByItemKey (map showModNameRef qnames)


-----------------------------------------------------------------------
-- Datatype for passing analysis results:

data AnaInfo =
   AnaInfo ((String,String) -> Bool)  -- overlapping?
           ((String,String) -> CompletenessType)  -- completely defined?
           ((String,String) -> Bool)  -- indeterministically defined?
           ((String,String) -> Bool)  -- solution complete?

getOverlappingInfo :: AnaInfo -> (String,String) -> Bool
getOverlappingInfo (AnaInfo oi _ _ _) = oi

getCompleteInfo :: AnaInfo -> (String,String) -> CompletenessType
getCompleteInfo (AnaInfo _ cdi _ _) = cdi

getIndetInfo :: AnaInfo -> (String,String) -> Bool
getIndetInfo (AnaInfo _ _ idi _) = idi

getOpCompleteInfo :: AnaInfo -> (String,String) -> Bool
getOpCompleteInfo (AnaInfo _ _ _ oci) = oci

-- Translate a standard analysis result into functional form:
getFunctionInfo :: [((String,String),a)] -> (String,String) -> a
getFunctionInfo [] n = error ("No analysis result for function "++show n)
getFunctionInfo ((fn,fi):fnis) n = if fn==n then fi
                                            else getFunctionInfo fnis n

-----------------------------------------------------------------------
-- auxiliaries:

-- show HTML doc with standard style sheet:
showDocCSS title hexps = 
  showHtmlPage (page title hexps `addPageParam` pageCSS currydocCSS)

-- generate big hrule containing a text string:
bigHRule s =
 [hrule,
  HtmlStruct "table" [("border","0"),("width","100%"),("cellpadding","3")]
   [HtmlStruct "tr" []
     [HtmlStruct "td" [("bgcolor","#0000ff")]
       [HtmlStruct "font" [("color","#ffffff")]
         [bold [HtmlText "&nbsp;", htxt s]]]]],
  hrule]

-- standard epilog for all generated web pages:
curryDocEpilog time =
  [hrule,
   italic [htxt "Generated by ",
           bold [htxt "CurryDoc"],
           htxt (" ("++currydocVersion++") at "),
           htxt (toDateString time)]]

-- extract last name from a path name:
getLastName = reverse . takeWhile (/='/') . reverse

-- extract directory name from a path name:
getDirName n =
  let revdirname = dropWhile (/='/') (reverse n)
   in if revdirname=="" then "."
                        else reverse (tail revdirname)

-- skip leading blanks or CRs in a string:
skipWhiteSpace = dropWhile isWhiteSpace

isWhiteSpace c = c==' ' || c=='\n'

-- Returns the first sentence in a string:
firstSentence s = let (fs,ls) = break (=='.') s in
  if ls==""
  then fs
  else if tail ls /= "" && isWhiteSpace (head (tail ls))
       then fs ++ "."
       else fs ++ "." ++ firstSentence (tail ls)

-- Sorts a list of strings.
sortStrings :: [String] -> [String]
sortStrings strings = mergeSort leqStringIgnoreCase strings


-- read a list of FlatCurry modules together with all their imported modules
-- and return the lists of type, function, and operator declarations:
readFlatCurryWithImports :: [String] -> IO ([TypeDecl],[FuncDecl],[OpDecl])
readFlatCurryWithImports modules = collectMods modules []
 where
  collectMods [] _ = return ([],[],[])
  collectMods (m:ms) implist =
    if m `elem` implist
    then collectMods ms implist
    else
      do filename <- findFileInLoadPath (m++".fcy")
         (Prog _ imps types funs ops) <- readFlatCurryFile filename
         (ts,fs,os) <- collectMods (ms++imps) (m:implist)
         return (types++ts, funs++fs, ops++os)


-- add a directory name for a Curry source file by looking up the
-- current load path (CURRYPATH):
findSourceFileInLoadPath modname = do
  loadpath <- getLoadPathForFile modname
  mbfname <- findFileInPath (baseName modname) [".lcurry",".curry"] loadpath
  maybe (error ("Curry file for module \""++modname++"\" not found!"))
        (return . stripSuffix)
        mbfname

-----------------------------------------------------------------------
