------------------------------------------------------------------------------
--- This module contains functions to reduce the size of FlatCurry programs
--- and because of this reducing compilation time and shrinking the size of 
--- the final Prolog programs.
---
--- @author Carsten Heine, Michael Hanus
--- @version August 2005
------------------------------------------------------------------------------

module CompactFlat(main,optimizeFlatCurry,defaultRequired,Option(..)) where

import FlatCurry
import SetRBT
import TableRBT
import Maybe
import List
import System
import FileGoodies
import Directory
import Sort(cmpString,leqString)
import XML
import Distribution(getLoadPathForFile)

-- Check arguments and call main function:
main = do
  args <- getArgs
  case args of
    [prog]              -> compactProgAndReplace [] prog
    ["-export",prog]    -> compactProgAndReplace [Exports] prog
    ["-main",func,prog] -> compactProgAndReplace [Main func] prog
    _ -> putStrLn $ "ERROR: Illegal arguments: " ++
                    concat (intersperse " " args) ++ "\n" ++
                    "Usage: compactflat [-export | -main func] <module_name>"

-- replace a FlatCurry program by a compactified version:
compactProgAndReplace options prog = do
   optimizeFlatCurry (Required defaultRequired : options) prog (prog++"_comp")
   system ("mv "++prog++"_comp.fcy "++prog++".fcy")
   putStr ("CompactFlat: compacted program '"++prog++".fcy' written.\n")
   done

------------------------------------------------------------------------------
--- Some options to guide the optimization process:

data Option =
    Verbose     -- for more output
  | Main String -- optimize for one main function supplied here (unqualified!)
  | Exports     -- optimize w.r.t. the exported functions of the module only
  | Required [QName] -- list of functions that should not be deleted

isMainOption o = case o of
                   Main _ -> True
                   _      -> False

getMainFuncFromOptions :: [Option] -> String
getMainFuncFromOptions (o:os) =
   case o of
     Main f -> f
     _      -> getMainFuncFromOptions os

getRequiredFromOptions :: [Option] -> [QName]
getRequiredFromOptions (o:os) =
   case o of
     Required fs -> fs
     _           -> getRequiredFromOptions os

------------------------------------------------------------------------------
--- The  basic functions that are always required in a FlatCurry
--- program (since they might be generated by external functions like
--- "==" or "=:=" on the fly):
defaultRequired =
  [("Prelude","&&"),("Prelude","&"),("Prelude","apply"),
   ("Prelude","ensureNotFree"),
   ("Prelude","letrec"),("Prelude","cond"),
   ("Prelude","ifVar"),("Prelude","=:="),("Prelude","&>"), -- for =:<<=
   ("Prelude","prim_readFileContents"),
   ("Prelude","failure"),
   ("Ports","basicServerLoop"),("Dynamic","isKnownAtTime")]

-------------------------------------------------------------------------------
-- Main Function:
-------------------------------------------------------------------------------

--- Optimizes a FlatCurry program.
--- Merges all imported FlatCurry modules and removes the imported functions,
--- that are not used.
--- @param options - list of options
--- @param sourceFile - name of the .fcy file to optimize. Without suffix .fcy.
--- @param targetFile - where to save. Without suffix .fcy.
--- @return the optimized FlatCurry program saved in targetFile
optimizeFlatCurry :: [Option] -> String -> String -> IO ()
optimizeFlatCurry options source target = 
  if (elem Exports options) && (any isMainOption options)
  then error "CompactFlat: Cannot execute options 'Main' and 'Exports' at the same time!"
  else do
    prog <- readCurrentFlatCurry source
    imports <- readAllImports prog options
    putStrLn ("CompactFlat: Total number of functions: " ++
              show (length (getFuns prog) +
                    foldr (+) 0 (map (length . getFuns) imports)))
    putStr ("CompactFlat: Starting optimization...\n")
    let result = makeOptimizedFlatCurry prog imports options
    putStrLn ("CompactFlat: Number of functions after optimization: " ++
              show (length (getFuns result)))
    writeFCY (target++".fcy") result
    done

--- Create the optimized Prog.
makeOptimizedFlatCurry :: Prog -> [Prog] -> [Option] -> Prog
makeOptimizedFlatCurry prog imports options =
      let
        (newType,funcTable,newOp) = makeGlobalLists (prog:imports)
        newTrans = makeNewTranslations2 prog imports options
        startFuncs = makeStartFuncs newTrans funcTable
        startFSet = getFSetFromFuncs startFuncs
        newFunc = makeNewFuncs startFuncs startFSet funcTable
      in
      Prog (moduleName prog) [] newType newFunc newOp


-------------------------------------------------------------------------------
-- Get all imported Modules:
-------------------------------------------------------------------------------

--- Reads all imported Modules of the main FlatCurry program.
--- @param prog - the main FlatCurry program
--- @param options - options
--- @return a list of all imported FlatCurry programs
readAllImports :: Prog -> [Option] -> IO [Prog]
readAllImports prog options = do
   putStr "CompactFlat: Loading imported modules: "
   imports <- readAllImports' (moduleImports prog) options (emptySetRBT leqString)
   putStr "done.\n"
   return imports


--- Subfunction of readAllImports.
--- @param modnames - a list of the names of the Modules to add
--- @param options - options
--- @param imported - a set of the names of the added Modules
--- @return a list of all imported FlatCurry programs
readAllImports' :: [String] -> [Option] -> SetRBT String -> IO [Prog]
readAllImports' [] _ _ = return []
readAllImports' (f:fs) options imported =
    if elemRBT f imported
      then do
        imports <- readAllImports' fs options imported
        return imports
      else do
        putStr (f ++ "...")
        prog <- readCurrentFlatCurry f
        imports <- readAllImports' (fs++(moduleImports prog)) options
                                   (insertRBT f imported)
        return (prog:imports)


-------------------------------------------------------------------------------
-- Generate the Prog to start with:
-------------------------------------------------------------------------------

makeNewTranslations2 :: Prog -> [Prog] -> [Option] -> [QName]
makeNewTranslations2 p ps options = 
    nub (getRequiredFromOptions options ++ makeNewTranslations p ps options)

-- extract all exported functions
getExportedFuncs :: [FuncDecl] -> [QName]
getExportedFuncs funs =
   map (\(Func name _ _ _ _)->name)
       (filter (\(Func _ _ vis _ _)->vis==Public) funs)


--- Generate the new export list depending on the options.
makeNewTranslations :: Prog -> [Prog] -> [Option] -> [QName]
makeNewTranslations (Prog m i _ fs _) progs options
  | Exports `elem` options = tr
  | any isMainOption options
    = let func = getMainFuncFromOptions options
       in if (m,func) `elem` (map functionName fs)
          then [(m,func)]
          else error ("CompactFlat: Cannot find main function \""++func++"\"!")
  | otherwise        = tr++(getExportedTrans i progs)
 where tr = getExportedFuncs fs


--- Get the Translationtables of a list of progs.
getExportedTrans :: [String] -> [Prog] -> [QName]
getExportedTrans _ [] = []
getExportedTrans ss ((Prog m _ _ fs _):progs) =
 (if m `elem` ss then getExportedFuncs fs else []) ++ getExportedTrans ss progs


--- Generate the FuncDeclarations to start with.
makeStartFuncs :: [QName] -> TableRBT QName FuncDecl -> [FuncDecl]
makeStartFuncs [] _ = []
makeStartFuncs (s2:trs) funcTable =
  let funcDecl = lookupRBT s2 funcTable in
    if isNothing funcDecl
      then makeStartFuncs trs funcTable
      else (fromJust funcDecl):(makeStartFuncs trs funcTable)


getFSetFromFuncs :: [FuncDecl] -> SetRBT QName
getFSetFromFuncs [] = emptySetRBT leqQName
getFSetFromFuncs (f:fs) = insertRBT (functionName f) (getFSetFromFuncs fs)


-------------------------------------------------------------------------------
-- Make the Table with all FuncDeclarations:
-------------------------------------------------------------------------------

--- Generate a TableRBT of all Functions.
makeGlobalLists :: [Prog] -> ([TypeDecl],TableRBT QName FuncDecl,[OpDecl])
makeGlobalLists [] = ([],emptyTableRBT leqQName,[])
makeGlobalLists ((Prog _ _ ts fs ops):progs) =
    updateGlobalLists ts fs ops (makeGlobalLists progs)


--- Add a list of FuncDeclarations to a TableRBT.
updateGlobalLists :: [TypeDecl] -> [FuncDecl] -> [OpDecl] -> 
                     ([TypeDecl],TableRBT QName FuncDecl,[OpDecl]) -> 
                     ([TypeDecl],TableRBT QName FuncDecl,[OpDecl])
updateGlobalLists t1 [] op1 (t2,funcTable,op2) = (t1++t2,funcTable,op1++op2)
updateGlobalLists t1 (f:fs) op1 (t2,funcTable,op2) = 
  updateGlobalLists t1 fs op1 (t2,(updateRBT (functionName f) f funcTable),op2)


-------------------------------------------------------------------------------
-- Add all required Functions to the Prog:
-------------------------------------------------------------------------------

makeNewFuncs :: [FuncDecl] -> SetRBT QName ->
                TableRBT QName FuncDecl -> [FuncDecl]
makeNewFuncs [] _ _ = []
makeNewFuncs (f:fs) fSet funcTable =
    let
      funcCalls = getAllFuncCalls f
      (fs',fSet') = lookupFuncCalls funcCalls fs fSet funcTable
    in
      f:(makeNewFuncs fs' fSet' funcTable)


--- Get the function declaration of a function name.
--- @param functions - the names of the functions to look up.
--- @param funcDecls - the function declarations of the necessary functions.
--- @param funcSet - set of all the names of all necessary functions.
--- @param funcTable - all imported function declarations
--- @return tupel of the necessary function declarations and the set of all 
---         names of the necessary functions
lookupFuncCalls :: [QName] -> [FuncDecl] -> SetRBT QName ->
                   TableRBT QName FuncDecl -> ([FuncDecl],SetRBT QName)
lookupFuncCalls [] fs fSet _ = (fs,fSet)
lookupFuncCalls (t:ts) fs fSet funcTable =
   if elemRBT t fSet 
     then 
       lookupFuncCalls ts fs fSet funcTable
     else
       let funcDecl = lookupRBT t funcTable in
         if isNothing funcDecl
           then lookupFuncCalls ts fs fSet funcTable
           else lookupFuncCalls
                      ts ((fromJust funcDecl):fs) 
                      (insertRBT (functionName (fromJust funcDecl)) fSet)
                      funcTable



-------------------------------------------------------------------------------
-- Functions to get all function calls in a function declaration:
-------------------------------------------------------------------------------

--- Get all function calls in a function declaration and remove duplicates.
--- @param funcDecl - a function declaration in FlatCurry
--- @return a list of all function calls
getAllFuncCalls :: FuncDecl -> [QName]
getAllFuncCalls (Func _ _ _ _ (External _)) = []
getAllFuncCalls (Func _ _ _ _ (Rule _ expr)) = nub (getAllFuncCallsOfExpr expr)


--- Get all function calls in an expression.
--- @param expr - an expression
--- @return a list of all function calls
getAllFuncCallsOfExpr :: Expr -> [QName]
getAllFuncCallsOfExpr (Var _) = []
getAllFuncCallsOfExpr (Lit _) = []
getAllFuncCallsOfExpr (Comb _ funcName exprs) = 
    funcName:(concatMap getAllFuncCallsOfExpr exprs)
getAllFuncCallsOfExpr (Free _ expr) = 
    getAllFuncCallsOfExpr expr
getAllFuncCallsOfExpr (Let bs expr) =
    concatMap (getAllFuncCallsOfExpr . snd) bs ++ getAllFuncCallsOfExpr expr
getAllFuncCallsOfExpr (Or expr1 expr2) = 
    getAllFuncCallsOfExpr expr1 ++ getAllFuncCallsOfExpr expr2
getAllFuncCallsOfExpr (Case _ expr branchExprs) =
    getAllFuncCallsOfExpr expr ++
    concatMap getAllFuncCallsOfBranchExpr branchExprs


--- Get all function calls in a branch expression in case expressions.
--- @param branchExpr - a branch expression
--- @return a list of all function calls
getAllFuncCallsOfBranchExpr :: BranchExpr -> [QName]
getAllFuncCallsOfBranchExpr (Branch _ expr) = getAllFuncCallsOfExpr expr


-------------------------------------------------------------------------------
-- Functions to get direct access to some data inside a datatype:
-------------------------------------------------------------------------------

--- Extracts the Function name of a Function declaration.
functionName :: FuncDecl -> QName
functionName (Func name _ _ _ _) = name


--- Extracts the names of imported Modules of a FlatCurry program.
moduleImports :: Prog -> [String]
moduleImports (Prog _ imports _ _ _) = imports


--- Extracts the name of the Prog.
moduleName :: Prog -> String
moduleName (Prog name _ _ _ _) = name


--- Extracts the functions of the program.
getFuns (Prog _ _ _ funs _) = funs


-------------------------------------------------------------------------------
-- Functions for comparison:
-------------------------------------------------------------------------------

--- Compares two qualified names.
--- Returns True, if the first name is lexicographically smaller than
--- the second name using the leString function to compare String.
leqQName :: QName -> QName -> Bool
leqQName (m1,n1) (m2,n2) = let cm = cmpString m1 m2
                            in cm==LT || (cm==EQ && leqString n1 n2)


-------------------------------------------------------------------------------
-- I/O functions:
-------------------------------------------------------------------------------

-- Read a FlatCurry program (parse only if necessary):
readCurrentFlatCurry :: String -> IO Prog
readCurrentFlatCurry modname = do
  progname <- findSourceFileInLoadPath modname
  fcyexists <- doesFileExist (progname++".fcy")
  if not fcyexists
    then readFlatCurry progname >>= processPrimitives progname
    else do ctime <- getSourceModificationTime progname
            ftime <- getModificationTime (progname++".fcy")
            if ctime>ftime
             then readFlatCurry progname >>= processPrimitives progname
             else readFlatCurryFile (progname++".fcy") >>= processPrimitives progname

getSourceModificationTime progname = do
  lexists <- doesFileExist (progname++".lcurry")
  if lexists then getModificationTime (progname++".lcurry")
             else getModificationTime (progname++".curry")

-- add a directory name for a Curry source file by looking up the
-- current load path (CURRYPATH):
findSourceFileInLoadPath modname = do
  loadpath <- getLoadPathForFile modname
  mbfname <- findFileInPath (baseName modname) [".lcurry",".curry"] loadpath
  maybe (error ("Curry file for module \""++modname++"\" not found!"))
        (return . stripSuffix)
        mbfname

-- read primitive specification and transform FlatCurry program accordingly:
processPrimitives :: String -> Prog -> IO Prog
processPrimitives progname prog = do
  pspecs <- readPrimSpec (moduleName prog) (progname++".prim_c2p")
  return (mergePrimSpecIntoModule pspecs prog)

mergePrimSpecIntoModule trans (Prog name imps types funcs ops) =
  Prog name imps types (concatMap (mergePrimSpecIntoFunc trans) funcs) ops

mergePrimSpecIntoFunc trans (Func name ar vis tp rule) =
 let fname = lookup name trans in
 if fname==Nothing
 then [Func name ar vis tp rule]
 else let Just (lib,entry) = fname
       in if null entry
          then []
          else [Func name ar vis tp (External (lib++' ':entry))]


readPrimSpec :: String -> String -> IO [(QName,QName)]
readPrimSpec mod xmlfilename = do
  existsXml <- doesFileExist xmlfilename
  if existsXml
   then do --putStrLn $ "Reading specification '"++xmlfilename++"'..."
           xmldoc <- readXmlFile xmlfilename
           return (xml2primtrans mod xmldoc)
   else return []

xml2primtrans mod (XElem "primitives" [] primitives) = map xml2prim primitives
 where
   xml2prim (XElem "primitive" (("name",fname):_)
                   [XElem "library" [] xlib, XElem "entry" [] xfun]) =
       ((mod,fname),(textOfXml xlib,textOfXml xfun))
   xml2prim (XElem "ignore" (("name",fname):_) []) = ((mod,fname),("",""))


-------------------------------------------------------------------------------
