---------------------------------------------------------------------------
--- This module provides a general interface for databases
--- (persistent predicates) where each entry consists of a key and an info
--- part. The key is an integer and the info is arbitrary.
--- All functions are parameterized with a dynamic predicate that
--- takes an integer key as a first parameter.
---
--- @author Bernd Brassel, Michael Hanus
--- @version March 2007
---------------------------------------------------------------------------

module KeyDatabase(existsDBKey,allDBKeys,allDBInfos,allDBKeyInfos,
                   getDBInfo,getDBInfos,
                   deleteDBEntry,updateDBEntry,newDBEntry,cleanDB,
                   index,sortByIndex,groupByIndex) where

import Database
import Integer(maxlist)
import Sort
import List

--- Exists an entry with a given key in the database?
--- @param db - the database (a dynamic predicate)
--- @param key - a key (an integer)
existsDBKey :: (Int -> _ -> Dynamic) -> Int -> Query Bool
existsDBKey db key = seq db $ seq key $
  transformQ (/=Nothing) (queryOne (db key))

--- Query that returns all keys of entries in the database.
allDBKeys :: (Int -> _ -> Dynamic) -> Query [Int]
allDBKeys db = seq db $ queryAll (\key -> db key unknown)

--- Query that returns all infos of entries in the database.
allDBInfos :: (Int -> a -> Dynamic) -> Query [a]
allDBInfos db = seq db $ queryAll (db unknown)

--- Query that returns all key/info pairs of the database.
allDBKeyInfos :: (Int -> a -> Dynamic) -> Query [(Int,a)]
allDBKeyInfos db = seq db $ queryAll (\ (key,info) -> db key info)

--- Gets the information about an entry in the database.
--- @param db - the database (a dynamic predicate)
--- @param key - the key of the entry (an integer)
getDBInfo :: (Int -> a -> Dynamic) -> Int -> Query a
getDBInfo db key = seq db $ seq key $
  queryOneWithDefault (error ("getDBInfo: no entry for key '"++show key++"'"))
                      (\info -> db key info)

--- compute the position of an entry in a list
--- fail, if given entry is not an element.
--- @param x - the entry searched for
--- @param xs - the list to search in

index :: a -> [a] -> Int
index x xs = idx 0 xs
  where
    idx n (y:ys) = if x==y then n else idx (n+1) ys


--- Sorts a given list by associated index .
sortByIndex :: [(Int,b)] -> [b]
sortByIndex = map snd . mergeSort (\x y -> fst x < fst y) 

--- Sorts a given list by associated index and group for identical index.
--- Empty lists are added for missing indexes
groupByIndex :: [(Int,b)] -> [[b]]
groupByIndex = addEmptyIdxs 0 . groupBy   (\x y -> fst x == fst y) 
                              . mergeSort (\x y -> fst x <  fst y)
  where
    addEmptyIdxs _ [] = []
    addEmptyIdxs n (((m,x):xs):ys) = 
       if n==m then (x:map snd xs) : addEmptyIdxs (n+1) ys
               else []:addEmptyIdxs (n+1) (((m,x):xs):ys)

--- Gets the information about a list of entries in the database.
--- @param db - the database (a dynamic predicate)
--- @param keys - the list of keys of the entry (integers)
getDBInfos :: (Int -> a -> Dynamic) -> [Int] -> Query [a]
getDBInfos db keys = seq db $ seq (normalForm keys) $
  transformQ sortByIndex
             (queryAll (\ (i,info) -> let key free in
                          db key info |&> (i=:=index key keys)))

--- Deletes an entry with a given key in the database.
--- @param db - the database (a dynamic predicate)
--- @param key - the key of the entry (an integer)
deleteDBEntry :: (Int -> _ -> Dynamic) -> Int -> Transaction ()
deleteDBEntry db key = seq db $ seq key $
  getDB (queryAll (\infos -> db key infos)) |>>= \entries ->
  mapT_ (\infos -> deleteDB (db key infos)) entries

--- Overwrites an existing entry in the database.
--- @param db - the database (a dynamic predicate)
--- @param key - the key of the entry (an integer)
--- @param info - the information to be stored in the updated entry
updateDBEntry :: (Int -> a -> Dynamic) -> Int -> a -> Transaction ()
updateDBEntry db key info = deleteDBEntry db key |>> addDB (db key info)

--- A query that returns a new database key.
newDBKey :: (Int -> _ -> Dynamic) -> Query Int
newDBKey db = 
  transformQ (\ids -> if null ids then 1 else maxlist ids + 1)
             (queryAll (\i -> db i unknown))

--- Stores a new entry in the database and return the key of the new entry.
--- @param db - the database (a dynamic predicate)
--- @param info - the information to be stored in the new entry
newDBEntry :: (Int -> a -> Dynamic) -> a -> Transaction Int
newDBEntry db info =
  getDB (newDBKey db) |>>= \i -> addDB (db i info) |>> returnT i

--- Deletes all entries in the database.
cleanDB :: (Int -> _ -> Dynamic) -> Transaction ()
cleanDB db = getDB (queryAll (\i -> db i unknown)) |>>= mapT_ (deleteDBEntry db)
