% A simple Curry interpreter (without sharing!)

:- prolog_flag(redefine_warnings,A,off).

:-use_module(operators).
:-use_module(maketree).
:-use_module(typecheck).
:-use_module(external).
:-use_module(local).
:-use_module(search).
:-use_module(currytools).
:-use_module(ms3).
:-use_module(library(lists)).
:-use_module(library(system)).   % for datime

:- dynamic currytraceflag/0.
:- dynamic singlestep/0.
:- dynamic optflag/0.
:- dynamic go/0.
:- dynamic io_monad/0.
:- dynamic showexectime/0.
:- dynamic preludepath/1.
:- dynamic modImportsMod/1.
:- dynamic pakcsrc/2. % used in ../curry2prolog/saveprog.pl


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% The Curry-Interpreter
%
% - Hierarchy: 
%
%   * Curryshell reads a term and computes certain keywords:
%     - trace/notrace: switch on or off the trace mode (show each computation step)
%     - single/nosingle: switch on or off single step mode (ask for action after
%       each computation step)
%     - time/notime: switch on or of time measuring for computation
%     - read <file>: reads in file.fl
%     - eval <function>: shows definitional tree for function
%     - type <function>: shows type for function
%     - exit: leave shell
%     - writeflat <file>: save program in flatcurry-notation
%     - writeprelude <file>: save prelude in flatcurry-notation
%     Otherwise interprete the term as goal and call 
%     "compute(goal, list of variables in goal)"
%
%   * compute(Goal,List) transforms the Goal into its internal representation,
%     creates the normal-form and checks correct typing. Then calls "rnstern"
%     for computation of the goal, formats its result for output and outputs it
%     to the screen. Also avoids incorrect use of IO-actions
%
%   * a goal is represented by a tripple: (V,Subst,Term). V the list of variables
%     declared so far (initialised with the list of all free variables of the
%     the initial term), Subst the substitution computed so far, Term the term
%     still to be reduced.
%
%   * rnstern handles a list of disjunctive goals. Calls "rn" for computing one
%     step in a loop, until no goal can reduce anymore. After each step, rnstern
%     shows the result if trace-mode is on and goes into interacive mode if single-
%     step-mode is on.
%
%   * rn handles a list of discuntive goals. Computes one step to the first goal
%     that is reducable by calling cseX(V,Goal,Result). Adds the Result (which is
%     possibly a list of new disjunctive goals) to the list of goals.
%
%   * cseX distinguishes the following forms of a goal:
%     - a variable
%     - a local-declaration
%     - a suspended goal (not reduceable at the moment)
%     - an explicit application (F@X)
%     - a partial application ($F)
%     - a search operator (try)
%     - a searchspace
%     - constraint equality (call equal)
%     - strict equality (call strictequal)
%     - a constructor-term
%     - a function. Call csX then.
%
%   * csX computes the function according to its definitional tree. See Curry-report
%     for explanation




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Shell


% Start TasteCurry Interpreter:
% (arg is absolute name of prelude file without suffix ".fl")
cs(PreludePath):-
  retractall(preludepath(_)),
  assertz(preludepath(PreludePath)),
  cs,
  halt.

cs:-
        % redefine standard if-then-else operators for our syntax:
        op(885,xfy,(;)),
        op(848,xfy,(->)),
        write('TasteCurry Interpreter (Version of 13/04/05)'), nl,
        write('(RWTH Aachen, CAU Kiel, Portland State University)'), nl, nl,
        write('Bug reports: mh@informatik.uni-kiel.de'), nl, nl,
        write('Type "help." for list of available commands'), nl,
        if(
            preludepath(PP), 
            setPreludesPath(PP),
            setPreludesPath(prelude)
          ),
        checkPrelude,
        on_exception(Exc,curryshell,(print_error(Exc),curryshell)).

tryReadTerm(Term,Names) :-
        repeat,
        on_exception(Exc,read_term(Term,[variable_names(Names)]),(print_error(Exc),fail)),
        !.

curryshell:-
%       safe(read_term(Term,[variable_names(N)])),nl,
        op(1150,fx,(eval)),
        tryReadTerm(Term,N), nl,
        op(1150,xfx,(eval)),
        if((nonvar(Term),Term='exit'),fail,true),
        if((nonvar(Term),Term='end_of_file'),fail,true),
        if(
             (
                nonvar(Term),
                member(Term,[help,single,nosingle,trace,notrace,time,notime,opt,noopt,read(_),type(_),eval(_),trees(_),writeflat(_),writeprelude(_),ff(_)])
             ),
             (  %THEN
                if(Term=..[type,X],
                   (symbtab(TrTab),
  %if(
  %     notAVariable(Mod,NotVars),
  %     true,
  %     true
  %  ),
  NotVars=[],

                    root_module(ROOT),
                    name(R, ROOT),
                    renTermDetVars(X, TrTab, NewX, R, R,NotVars,[],_),
                    NewT=type(NewX)
                   ),
                   if(Term=eval(X),
                      (symbtab(TrTab), 
                       root_module(ROOT),
                       name(R, ROOT),
                       renTermDetVars(X, TrTab, NewX, R, R,NotVars,[],_), 
                       NewT=eval(NewX)
                      ),
                      NewT=Term
                     )
                  ),
                  keyWords(NewT,N)
               ),
               (  %ELSE
                  if(
                        symbtab(TrTab),
                        (
                            root_module(ROOT),
                            name(R, ROOT),
                            NotVars=[],
                            renTermDetVars(Term, TrTab, NewTerm, R, R,NotVars,[],SmallVars),
                            write('computing '), write(NewTerm),nl,nl,
                            compute(NewTerm, N, SmallVars)
                        ),
                        compute(Term,N,[])
                    )  
               )
           ),
        nl,
        curryshell.

curryshell:-write('Leaving Curry.'),nl.








checkPrelude:-
  initAll,
  setPreludesPath(prelude),
  if(
       preludepath(PP), 
       true, 
       PP=prelude
    ),
  on_exception(
                 _,
                 (
                    addExtension(PP,'.fl', FN),
                    see(FN)
                 ),
                 (format(user_error, "\n*** Error: prelude's \c
                              implementation does not exist.\n",[]),
                  seen,
                  fail
                 )
              ),
  seen,
  file_property(FN, mod_time(ImplTime)),
  addExtension(PP,'.int', IntfFile),
  on_exception(
                 _,
                 see(IntfFile),
                 GenInt=doIt
              ),
  seen,
  if(
       var(GenInt),
       (
         file_property(IntfFile, mod_time(IntfTime)),
         if(
             ImplTime>IntfTime,      %implem. is newer
             (
                 format(user_output,
                        "prelude's interface is older than implementation ... ignoring it.\n",
                           []
                       ),
                 addExtension(IntfFile,'.bak',Backup),
                 rename_file(IntfFile, Backup),
                 GenInt=doIt
             ),
             GenInt=leaveIt
           )
       ),
       true
    ),
  if(
       GenInt==doIt,
       (
            see(FN),
            findImports([],[(_,_,Exps)],[],prelude,generateInterface),
            seen,
            safe(retractall(currline(_))),
            addOps2Funcs([(prelude,[(prelude,[],Exps,used)])],[],PrelDefs),
            removeOperators(PrelDefs,[],[(_,[(_,_,PDs,_)])]),
            safe(retractall(symbtab(_))),
            asserta(symbtab([(prelude,PDs)])),

            readImplementations([prelude],[(prelude,PDs)],[],[],prelude),
            safe(retractall(currline(_))),
            seen,

            typeCheckAll,
            makeTrees,
            asserta(noInterface(prelude)),
            asserta(root_module(prelude)),
            generateInterfaces(PrelDefs,[(prelude,PDs)]),
            safe(retractall(symbtab(_))),
            safe(retractall(noInterface(_))),
            safe(retractall(root_module(_)))
       ),
        true
    ).



extendedWrite(Term):-
%  addApostrophes(Term,NewTerm),
Term=NewTerm,
  \+ \+ (
            numbervars(NewTerm,0,_),
            writeq(NewTerm)
        ).






%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Tools for curryshell


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% keyWords(Key, VariableNames)
%
% - computes the internal keywords that the user can type in at
%   the shell.
%     - trace/notrace: switch on or off the trace mode (show each computation step)
%     - single/nosingle: switchon or off single step mode (ask for action after
%       each computation step)
%     - time/notime: switch on or of time measuring for computation
%     - opt/noopt: turn on/off optimal definitional tree generation
%     - read <file>: reads in file.fl
%     - eval <function>: shows definitional tree for function
%     - type <function>: shows type for function
%     - writeflat <file>: saves program in flatcurry-notation
%     - writeprelude <file>: saves prelude in flatcurry-notation
%     - ff <file>: saves program in FORMATTED flatcurry notation
%     - exit: leave shell


keyWords(single,_):-
        retractall(singlestep),
        assert(singlestep),
        write('Singlestep reduction on.').
keyWords(nosingle,_):-
        retractall(singlestep),
        write('Singlestep reduction off.').
keyWords(trace,_):-
        retractall(currytraceflag),
        assert(currytraceflag),
        write('Trace on.').
keyWords(notrace,_):-
        retractall(currytraceflag),
        write('Trace off.').
keyWords(time,_):-
        retractall(showexectime),
        assert(showexectime),
        write('Timing on.').
keyWords(notime,_):-
        retractall(showexectime),
        write('Timing off.').
keyWords(opt,_):-
        retractall(optflag),
        assert(optflag),
        write('Optimizing on.').
keyWords(noopt,_):-
        retractall(optflag),
        write('Optimizing off.').
keyWords(read(X),_):-!,safe(readIn(X)).
%keyWords(read(X)):-readProg(X).

keyWords(eval(X),_):-
        nonvar(X),!,
        if(tree(X,T),
           (write('Definitional Tree for '),write(X),write(':'),nl,
            writeNames(T)),
           write('No tree defined.')).
keyWords(eval(_)):-write('Variables not allowed.').

keyWords(type(X),N):-!,%nonvar(X),!,
        if(typeTerm(X,T,_),
           (write('Type of '),
			\+ \+ (makeNames(N,_),write(X),write(': '),nl,writeNames(T))),
           true).

keyWords(help,_) :-
        write('read <name>.        - Load "<name>.cur" (or "<name>.fl")'), nl,
        write('<expression>.       - Evaluate <expression> w.r.t. current program'), nl,
        write('single./nosingle.   - Turn on/off singlestep reduction mode'),nl, 
        write('trace./notrace.     - Show each reduction step'),nl,
        write('time./notime.       - Show execution time after evaluation'),nl,
        write('opt./noopt.         - Generate Definitional Trees with optmatch-strategy'),nl,
        write('type <expression>.  - Show type of <expression>'),nl,
        write('eval <func>.        - Show definitional tree of <func>'),nl,
	write('trees <def-file>.   - Dump definitional trees to file'),nl,
	write('writeflat <file>.   - Create flat curry representation of current file'),nl,
        write('writeprelude <file> - Create flat curry representation of prelude'),nl,
        write('exit.               - Leave the system').

  
%----------------------- BEGIN COMPILER-INTERFACE ------------------------

keyWords(trees(Filename),_):-
    retractall(choicecounter(_)),
	retractall(conv_tree(_,_)),
  	convertTrees, 
	tell(Filename),!,
	ci_writeConstructors,
    ci_writeFunctions,
    ci_writeRootModule,
    ci_writeTransTab,
    ci_writeDefTrees,
	told,write('Tree-file generated').



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% writeflat
% write flatcurry-program to Filename (Extension .fcr is added, if needed)
% 

:- dynamic start/0.


  
keyWords(writeprelude(Filename),_):-
     op(0,xfx,(or)),op(0,xfx,('||')),op(0,xfx,(/\)),op(0,xfx,(&&)),op(0,xfx,(==)),op(0,xfx,(<)),
     op(0,xfx,(>)),op(0,xfx,(<=)),op(0,xfx,(>=)),op(0,xfx,(++)),op(0,xfx,(+)),op(0,xfx,(-)),
     op(0,xfx,(*)),op(0,xfx,(div)),op(0,xfx,(mod)),op(0,xfx,(localVars)),op(0,xfx,(localInPar)),
	 op(0,xfx,(>>=)),op(0,xfx,(>>)),
     addExt(Filename,FilenameExt),
     tell(FilenameExt),!,
     write('prog('),write('prelude'),write(','),
     write('datadecl(['),retractall(start),assert(start),
     flattenConstructors,    
     write(']),'),
     write('funcdecl(['),retractall(start),assert(start),
     flattenFunctions,
     write(']),'),
     write('translation(['),    % write translation table
     symbtab(T),
     writesymbTab(T),
     write('])).'),
     told,write('prelude written to '),write(FilenameExt),nl,
     op(850,xfx,(or)),op(850,xfx,('||')),op(950,xfy,(/\)),op(800,xfy,(&&)),op(700,xfx,(==)),
     op(700,xfx,(<)),op(700,xfx,(>)),op(700,xfx,(<=)),op(700,xfx,(>=)),op(500,xfy,(++)),
     op(500,yfx,(+)),op(500,yfx,(-)),op(400,yfx,(*)),op(400,yfx,(div)),op(400,yfx,(mod)),
	 op(960,xfy,localVars),op(960,xfy,localInPar),op(880,xfy,(>>=)),op(880,xfy,(>>)).

keyWords(writeflat(Filename),_):-
     function(FName,_,_), mainName(FName),
     tree(FName,_), % is there at least one tree in main module?
     !,
     op(0,xfx,(or)),op(0,xfx,('||')),op(0,xfx,(/\)),op(0,xfx,(&&)),op(0,xfx,(==)),op(0,xfx,(<)),
     op(0,xfx,(>)),op(0,xfx,(<=)),op(0,xfx,(>=)),op(0,xfx,(++)),op(0,xfx,(+)),op(0,xfx,(-)),
     op(0,xfx,(*)),op(0,xfx,(div)),op(0,xfx,(mod)),op(0,xfx,(localVars)),op(0,xfx,(localInPar)),
	 op(0,xfx,(>>=)),op(0,xfx,(>>)),
     root_module(RootAsc),name(RootName,RootAsc),
     addExt(Filename,FilenameExt),
     tell(FilenameExt),!,
     write('prog('),write(RootName),write(','),
     write('datadecl(['),retractall(start),assert(start),
     flattenMainConstructors,    
     write(']),'),
     write('funcdecl(['),retractall(start),assert(start),
     flattenMainFunctions,
     write(']),'),
     write('translation(['),    % write translation table
     assert(start), wrMainCons, wrMainFuncs,
     write('])).'),
     told,write('FlatCurry written to '),write(FilenameExt),nl,
     op(850,xfx,(or)),op(850,xfx,('||')),op(950,xfy,(/\)),op(800,xfy,(&&)),op(700,xfx,(==)),
     op(700,xfx,(<)),op(700,xfx,(>)),op(700,xfx,(<=)),op(700,xfx,(>=)),op(500,xfy,(++)),
     op(500,yfx,(+)),op(500,yfx,(-)),op(400,yfx,(*)),op(400,yfx,(div)),op(400,yfx,(mod)),
	 op(960,xfy,localVars),op(960,xfy,localInPar),op(880,xfy,(>>=)),op(880,xfy,(>>)).
keyWords(writeflat(Filename),_):-
	write('*** No defined functions in main program, translation aborted'),
	nl.

addExt(Filename,Filename):-
     name(Filename,FileAsc),name('.flat',Ext),append(_,Ext,FileAsc),!.
addExt(Filename,CompName):-
     name(Filename,FileAsc),name('.flat',Ext),append(FileAsc,Ext,CompAsc),name(CompName,CompAsc).


											
addExt2(Filename,Filename):-
     name(Filename,FileAsc),name('.ff',Ext),append(_,Ext,FileAsc),!.
addExt2(Filename,CompName):-
     name(Filename,FileAsc),name('.ff',Ext),append(FileAsc,Ext,CompAsc),name(CompName,CompAsc).

keyWords(ff(Filename),_):-addExt2(Filename,Out),addExt(Filename,In),
  write('output formatted flat curry'),outputParser(In,Out).
								
outputParser(In,Out):-see(In),read(Term),seen,numbervars(Term,0,_),tell(Out),outFlat(Term,[]),told.

outFlat(prog(Name,D,F,T),Tab):-write('prog '),write(Name),nl,
  outFlat(D,[0|Tab]),
  outFlat(F,[0|Tab]),
  outFlat(T,[0|Tab]).


outFlat(datadecl(D),Tab):-doTab(Tab),write('datadecl'),nl,write('['),nl,outList(D,[0|Tab]),write(']'),nl.
outFlat(funcdecl(D),Tab):-doTab(Tab),write('funcdecl'),nl,write('['),nl,outList(D,[0|Tab]),write(']'),nl.
outFlat(translation(D),Tab):-doTab(Tab),write('translation'),nl,write('['),nl,outList(D,[0|Tab]),write(']'),nl.


outFlat(type(T,Vars,Cons),Tab):-doTab(Tab),write('type '),write(T),nl,doTab(Tab),write('['),outList(Vars,[0|Tab]),doTab(Tab),write(']'),nl,doTab(Tab),write('['),outList(Cons,[0|Tab]),doTab(Tab),write(']'),nl.

outFlat(cons(C,A,TL),Tab):-doTab(Tab),write('cons '),write(C),nl,doTab([0|Tab]),write(A),nl,doTab(Tab),write('['),outList(TL,[0|Tab]),doTab(Tab),write(']'),nl.
  
outFlat(functype(T1,T2),Tab):-doTab(Tab),write('functype '),nl,outFlat(T1,[0|Tab]),nl,outFlat(T2,[0|Tab]).

outFlat(tvar(V),Tab):-doTab(Tab),write('tvar '),write(V),nl.
outFlat(tcons(V,L),Tab):-doTab(Tab),write('tcons '),write(V),nl,doTab(Tab),write('['),outList(L,[0|Tab]),doTab(Tab),write(']'),nl.

outFlat(func(C,A,TL,R),Tab):-doTab(Tab),write('func '),write(C),nl,doTab([0|Tab]),write(A),nl,doTab(Tab),write('['),outList(TL,[0|Tab]),doTab(Tab),write(']'),nl,outFlat(R,[0|Tab]).
							
outFlat(rule(F,E),Tab):-doTab(Tab),write('rule '),nl,doTab([0|Tab]),write(F),nl,outFlat(E,[0|Tab]).
outFlat(choicerule(F,E),Tab):-doTab(Tab),write('choicerule '),nl,doTab([0|Tab]),write(F),nl,outFlat(E,[0|Tab]).
							
outFlat(trans(F,E),Tab):-doTab(Tab),write('trans '),nl,doTab([0|Tab]),write(F),nl,doTab([0|Tab]),write(E),nl.

outFlat(var(X),Tab):-doTab(Tab),write('var '),write(X),nl.
outFlat(int(X),Tab):-doTab(Tab),write('int '),write(X),nl.
outFlat(comb(X,Y),Tab):-doTab(Tab),write('comb '),write(X),nl,doTab(Tab),write('['),outList(Y,[0|Tab]),doTab(Tab),write(']'),nl.
							
outFlat(apply(X,Y),Tab):-doTab(Tab),write('apply '),nl,outFlat(X,[0|Tab]),nl,outFlat(Y,[0|Tab]).

outFlat(or(X,Y),Tab):-doTab(Tab),write('or '),nl,outFlat(X,[0|Tab]),nl,outFlat(Y,[0|Tab]).
							
outFlat(case(X,Y),Tab):-doTab(Tab),write('case '),write(X),nl,doTab(Tab),write('['),outList(Y,[0|Tab]),doTab(Tab),write(']'),nl.
outFlat(fcase(X,Y),Tab):-doTab(Tab),write('fcase '),write(X),nl,doTab(Tab),write('['),outList(Y,[0|Tab]),doTab(Tab),write(']'),nl.
outFlat(guardedexpr(X,Y,Z),Tab):-doTab(Tab),write('guardedexpr '),nl,doTab(Tab),write('['),outList(X,[0|Tab]),doTab(Tab),write(']'),nl,outFlat(Y,[0|Tab]),nl,outFlat(Z,[0|Tab]).
							
outFlat(pattern(X,Y,Z),Tab):-doTab(Tab),write('pattern '),write(X),nl,doTab(Tab),write('['),outList(Y,[0|Tab]),doTab(Tab),write(']'),nl,outFlat(Z,[0|Tab]).
outFlat(vpattern(X,Z),Tab):-doTab(Tab),write('vpattern '),write(X),nl,outFlat(Z,[0|Tab]).
							
outFlat(constr(X,Y),Tab):-doTab(Tab),write('constraint '),nl,doTab(Tab),write('['),outList(X,[0|Tab]),doTab(Tab),write(']'),nl,outFlat(Y,[0|Tab]).

outFlat(X,Tab):-doTab(Tab),write('??? '),write(X),nl.
							
outList([],_).
outList([X|Xs],Tab):-nl,outFlat(X,Tab),outList(Xs,Tab).
  
  
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% convert trees

convertTrees:-tree(X,T),
  name(X,NameS),\+ ci_testFunc(NameS),
  if(choicetree(X,CT),    % if a choicetree exists, convert use it for curry2java
	 convertTheTree(tree(X,CT)),
	 convertTheTree(tree(X,T))),
  fail.
convertTrees.

convertTheTree(tree(Func,Tree)):-
	convertTree(Tree,ConvTree),assertz(conv_tree(Func,ConvTree)).
convertTree(branch(Pattern,Pos,Flag,SubTrees),branch(Pattern,Pos,Flag,ConvSubTrees)):-
	convertTreeList(SubTrees,ConvSubTrees).
convertTree(or(Left,Right),or(ConvLeft,ConvRight)):-
	convertTree(Left,ConvLeft),convertTree(Right,ConvRight).
%NEW CHOICE
convertTree(choice(Tree),choice(ConvTree)):-
  convertTree(Tree,ConvTree).
%convertTree(choice(GlobalVars,Pattern,Choices),choice(Pattern,ConvChoices)):-
%	convertChoices(GlobalVars,Pattern,Choices,ConvChoices).
convertTree(rule(Left,Right),rule(Left,ConvRight)):-
	convertTerm((Left,Right),Right,ConvRight,[1]).

%convertChoices(_,_,[],[]).
%convertChoices(GlobalVars,Pattern,[(Constraint,Right)|Choices],[(ConvCons,ConvRight)|ConvChoices]):-
%  convertTerm((Pattern,(Constraint,Right)),(Constraint,Right),(ConvCons,ConvRight),[1]),
%  convertChoices(GlobalVars,Pattern,Choices,ConvChoices).

renameLocalVars(GlobalVars,Table,Var,Var,Table):-
	var(Var),varmember(Var,GlobalVars),!.
renameLocalVars(_GlobalVars,Table,Var,NewVar,Table):-
	var(Var),checkVar(Var,Table,NewVar),!.
renameLocalVars(_GlobalVars,Table,Var,NewVar,[(Var,NewVar)|Table]):-
	var(Var),!.
renameLocalVars(GlobalVars,Table,Term,NewTerm,NewTable):-
	Term=..[Head|Args],
	renameLocalArgs(GlobalVars,Table,Args,NewArgs,NewTable),
	NewTerm=..[Head|NewArgs].

renameLocalArgs(_GlobalVars,Table,[],[],Table).
renameLocalArgs(GlobalVars,Table,[Arg|Args],[NewArg|NewArgs],NewTable):-
renameLocalVars(GlobalVars,Table,Arg,NewArg,NewTable2),
	renameLocalArgs(GlobalVars,NewTable2,Args,NewArgs,NewTable).

checkVar(Var,[(V,NV)|_Table],NV):-Var==V,!.
checkVar(Var,[_|Table],NV):-checkVar(Var,Table,NV).
		

convertTerm(_Full,X,X,_Pos):-var(X),!.

asd.  
% build a new choice-function that replaces the 'choice'-operator
convertTerm(_Full,committedChoice(Choices),ChoiceFunc,_Pos):-!,
    asd,
	freeVars(choice(Choices),GlobalVars),
	getChoiceName(Name),ChoiceFunc=..[Name|GlobalVars],
	convertTheTree(tree(Name,choice(GlobalVars,ChoiceFunc,Choices))),
	length(GlobalVars,N),
	assertz(function(Name,N,dummy)).

convertTerm(Full,Term,ConvTerm,Pos):-
	Term=..[Head|Args],convertArgs(Full,Args,ConvArgs,Pos,0),ConvTerm=..[Head|ConvArgs].

	getChoiceName(Name):-
		getChoiceCounter(N),name(N,StringN),
		append("internalchoice_",StringN,StringName),name(Name,StringName).

	:- dynamic choicecounter/1.

   getChoiceCounter(X):-
       if(choicecounter(X),
            (retract(choicecounter(X)),X1 is X+1,assertz(choicecounter(X1))),
            (X=1,assertz(choicecounter(2)))
         ).

% extracts the elements of the var-list 'Vars' that occur in the FullTerm too.
localVars(FullTerm,Pos,[Var|Vars],[Var|GlobalVars]):-
	checkVar(FullTerm,Pos,Var),!,localVars(FullTerm,Pos,Vars,GlobalVars).
localVars(FullTerm,Pos,[_Var|Vars],GlobalVars):-
	localVars(FullTerm,Pos,Vars,GlobalVars).
localVars(_,_,[],[]).

checkVar(V,_Pos,Var):-
	var(V),!,V==Var.
checkVar(Term,Pos,Var):-
	Term=..[_Head|Args],
	checkArgs(Args,Pos,Var,0).

checkArgs([Arg|Args],[P|Pos],Var,N):-
	if( if(N==P,
			checkVar(Arg,Pos,Var),
			checkVar(Arg,[P|Pos],Var)
		),
		true,
		(N1 is N+1,checkArgs(Args,[P|Pos],Var,N1))
	).

convertArgs(_Full,[],[],_,_).
convertArgs(Full,[Arg|Args],[ConvArg|ConvArgs],Pos,N):-
	append(Pos,[N],NewPos),
	convertTerm(Full,Arg,ConvArg,NewPos),
	N1 is N+1,
	convertArgs(Full,Args,ConvArgs,Pos,N1).

convertTreeList([],[]).
convertTreeList([T|Ts],[CT|CTs]):-convertTree(T,CT),convertTreeList(Ts,CTs).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
  
ci_writeConstructors:-
    write('constructors'),nl,ci_writeTypes,nl.

ci_writeFunctions:-
    write('functions'),nl,ci_writeFuncs,nl.

ci_writeRootModule:-
    write('rootmodule'),nl,root_module(R),name(N,R),write(N),nl,nl.

ci_writeTransTab:-
    write('transtab'),nl,
% predefined functions/constructors must not be renamed
    write('= ='),nl,
    write('== =='),nl,
    write('@ @'),nl,
    write(': :'),nl,
    write('[] []'),nl,
    symbtab(T),ci_writeTab(T),nl.

ci_writeDefTrees:-
    write('deftrees'),nl,ci_writeTrees,nl.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% write name-transformtation table

ci_writeTab([]):-!.
ci_writeTab([(or,or)|T]) :- !, ci_writeTab([('$or','$or')|T]).
				% since "or" is keyword in def files
ci_writeTab([(N,M)|T]):-
    if(N=='choice',
       true,
       (write(N),write(' '),write(M),nl )
    ),ci_writeTab(T).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% write definitional trees

ci_writeTrees:-conv_tree(X,T),
	name(X,NameS),\+ ci_testFunc(NameS),
    ci_writeTerm(X),write(' '),
	(\+ \+ (numbervars(T,0,_),ci_writeTree(T))),
    nl,fail.
ci_writeTrees:-
	%write('done rule(done,$io(unit))'),nl,
	write('return branch(return(A),{1},rigid,{rule(return(A),$io(A))})').

ci_writeTree(branch(Pattern,Pos,Flag,SubTrees)):-!,
	write('branch('),
	ci_writeTerm(Pattern),write(','),
	ci_writePos(Pos),write(','),
	write(Flag),write(','),
	ci_writeSubTrees(SubTrees),
	write(')').
ci_writeTree(rule(Left,Right)):-!,
	write('rule('),
	ci_writeTerm(Left),write(','),ci_writeTerm(Right),
	write(')').
ci_writeTree(or(Left,Right)):-!,
	write('or('),ci_writeTree(Left),write(','),ci_writeTree(Right),write(')').

%NEW CHOICE  
ci_writeTree(choice(Tree)):-!,
  write('choice('),
  ci_writeTree(Tree),
  write(')').

  
%ci_writeTree(choice(Pattern,Rules)):-				% Rules is not []
%	write('choice('),ci_writeTerm(Pattern),ci_writeChoiceRules(Rules),write(')').

%ci_writeChoiceRules([]):-!.
%ci_writeChoiceRules([(Constraint,Right)|Rules]):-
%	write(',([],'),ci_writeTerm(Constraint),write(','),ci_writeTerm(Right),write(')'),
%	ci_writeChoiceRules(Rules).

  
ci_writeSubTrees(T):-
	write('{'),
	if(T=[X|Xs],
		(ci_writeTree(X),ci_writeRestTrees(Xs)),
		true
	),write('}').
ci_writeRestTrees([]).
ci_writeRestTrees([X|Xs]):-write(','),ci_writeTree(X),ci_writeRestTrees(Xs).

ci_writePos(X):-!,write('{'),ci_writePos2(X),write('}').
ci_writePos2(X#Y):-!,ci_writePos2(X),write(','),ci_writePos2(Y).
ci_writePos2(X):-write(X).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% write functionnames

ci_writeFuncs:-function(Name,Arity,Type),
	name(Name,NameS),\+ ci_testFunc(NameS),
    ci_writeTerm(Name),write(' '),write(Arity),
	if(ci_testIOType(Type),
		write(' isIO'),
		true
	),
	nl,fail.
ci_writeFuncs.

ci_testIOType(X):-var(X),!,fail.
ci_testIOType((_Z->X)):-var(X),!,fail.
ci_testIOType(io(_)).
ci_testIOType((_Z->io(_))).

ci_testFunc("choice"):-!.
ci_testFunc("/\\"):-!.
ci_testFunc(X):-name(Z,X),isExternal(Z).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% write typenames and constructors to file

ci_writeTypes:-typen(Type,Cons),
	Type=..[Name|_],
	if(Name=list,
		true,
		(write(Name),write('{'),ci_writeConsList(Cons),write('}'),nl)
	),fail.
ci_writeTypes.

ci_writeConsList([]).
ci_writeConsList([Cons|ConsList]):-
	constructor(Cons,Arity,_,_),ci_writeTerm(Cons),write(' '),write(Arity),write(' '),
	ci_writeConsList(ConsList).


ci_writeTerm(X):-
	X='$VAR'(_),!,write(X).
ci_writeTerm({}):-!,
	write('valid').
ci_writeTerm(or) :- !,
	ci_writeTerm('$or').	% since "or" is keyword in def files
ci_writeTerm(or(X)) :- !,
	ci_writeTerm('$or'(X)). % since "or" is keyword in def files
ci_writeTerm([]):-!,
	write([]).
ci_writeTerm([X|Xs]):-!,
	write(':('),ci_writeTerm(X),write(','),ci_writeTerm(Xs),write(')').
ci_writeTerm('$'(X)):-!,
	ci_writeTerm(X).
ci_writeTerm(F):-
	F=..['.'|Args],!,write(':'),ci_writeArgs(Args).
ci_writeTerm(F):-
	F=..[','|Args],!,write('pair'),ci_writeArgs(Args).
ci_writeTerm(_Vars localVars Term):-!,
	ci_writeTerm(Term).
ci_writeTerm(_Vars localInPar Term):-!,
	ci_writeTerm(Term).
ci_writeTerm(F):-
	F=..[Head|_],
	member(Head,[show,openPortOnce,openPortSocketOnce,sendPortOnce,
	             readPortOnce,connectPortExt,portName,afterExt,
                 basicStdInLoop,committedChoice,(<--),choice,unsafePerformIO]),
	!,write('notsupported'). %,ci_writeArgs(Args).
ci_writeTerm(F):-
	F=..[Head|Args],!,write(Head),ci_writeArgs(Args).

ci_writeArgs([]):-!.
ci_writeArgs([A|As]):-write('('),ci_writeTerm(A),ci_writeRestArgs(As),write(')').
ci_writeRestArgs([]).
ci_writeRestArgs([A|As]):-write(','),ci_writeTerm(A),ci_writeRestArgs(As).



%%%%%%%%%%%%%%%%%%%%%% FLAT CURRY %%%%%%%%%%%%%%%%%%%%%%%%%

% write a type constructor on standard output:
% if type constructor = constraint, write "success"
writeTC(TC) :-
	TC=constraint -> writeq(success) ; writeq(TC).

% write constructors
%


flattenConstructors:-
     typen(Type,Cons),Type=..[Typname|ArgList],noMainName(Typname),
     if(start,
        (write('type('),retractall(start)),
         write(',type(')
         ),
         writeTC(Typname),
         write(',['),
%         writeList(ArgList),
         wrTypeExpL(ArgList),
         write('],['),
         wrConsList(Type,Cons),
         write('])'),
         fail.
flattenConstructors.

flattenMainConstructors:-
     typen(Type,Cons),Type=..[Typname|ArgList],mainName(Typname),
     if(start,
        (write('type('),retractall(start)),
         write(',type(')
         ),
         writeTC(Typname),
         write(',['),
%         writeList(ArgList),
         wrTypeExpL(ArgList),
         write('],['),
         wrConsList(Type,Cons),
         write('])'),
         fail.
flattenMainConstructors.

mainName(Name):-root_module(R),name(Name,NameAsc),append(R,[95|_],NameAsc).
noMainName(Name):-
      if(mainName(Name),
          fail,
          true).

writeList([]):-!.
writeList([Head]):-!,
     writeq(Head).
writeList([Head|Tail]):-
     writeq(Head),write(','),writeList(Tail).     

%clean('$io','$io'):-!.
% in the following: . must be change into '.', , into ',', # into '#',
% and $ is removed
%clean('.','\'.\''):-!.
%clean(',','\',\''):-!.
%
% remove only the prefix $ that marks partial function calls
%
clean(X,X):-var(X).					 
clean('$'Cons,Cons).
clean(F,F).
%          name(Cons,AsciiCons),
%          if(AsciiCons=[36|AsciiCons2],
%               name(Cons2,AsciiCons2),
%               name(Cons2,AsciiCons)).

% change !! into '!!' , || into '||' and #xxx into '#xxx'
%cleanName([35|Name],Name2):-!,
%     append([39,35|Name],[39],Name2).
%cleanName([124,124],[39,124,124,39]):-!.
%cleanName([33,33],[39,33,33,39]):-!.
%cleanName(Name,Name).    

wrConsList(_Typname,[]):-!.
wrConsList(Typname,[Cons]):-!,
     constructor(Cons,Arity,Typ,Typname),
     write('cons('),
     clean(Cons,Cons2),
     writeq(Cons2),
     write(','),write(Arity),
     write(',['),
     (functor(Typ,(->),2) -> deleteResultType(Typ,CTyp),
                             wrTypeExpList(CTyp)
                           ; true),
     write('])').
wrConsList(Typname,[Cons|Cs]):-
     constructor(Cons,Arity,Typ,Typname),
     write('cons('),
     clean(Cons,Cons2),
     writeq(Cons2),
     write(','),write(Arity),
     write(',['),
     (functor(Typ,(->),2) -> deleteResultType(Typ,CTyp),
                             wrTypeExpList(CTyp)
                           ; true),
     write('])'),
     write(','),
     wrConsList(Typname,Cs).

wrTypList((A->B->C)):- !,writeq(A),write(','),wrTypList((B->C)).
wrTypList((A->_B)):- !,writeq(A). 
wrTypList(_A).

% delete the last element in a non-empty list:
deleteResultType((A->(B->C)),(A->D)) :- !, deleteResultType((B->C),D).
deleteResultType((A->_),A).


%%%%%%%%%%%%%%%%%%%%%
% write functions


%clean2(Cons,Cons2):-
%		  name(Cons,AsciiCons),cleanName(AsciiCons,AsciiCons2),name(Cons2,AsciiCons2).
					 
%clean3(Pattern,Pattern2):-
%		  Pattern=..[Cons|List], clean2(Cons,Cons2),Pattern2=..[Cons2|List].
					 
flattenFunctions:-
     function(Funcname2,Arity,Types),noMainName(Funcname2),clean(Funcname2,Funcname),tree(Funcname,T),
     if(start,
        (write('func('),retractall(start)),
         write(',func(')),
     makePattern(Funcname,Arity,Pattern),
     %clean2(Funcname,Funcname3),
	 writeq(Funcname),
     write(','),
     write(Arity),
     write(',['),
     wrTypeExpList(Types),
     write('],rule('),
	 %clean3(Pattern,Pattern2),				 
	 writeq(Pattern),
	 write(','),
     getRule(Pattern,Pattern,T,Rule),writeq(Rule),
     write('))'),
     fail.
flattenFunctions.

flattenMainFunctions:-
     function(Funcname2,Arity,Types),mainName(Funcname2),clean(Funcname2,Funcname),tree(Funcname,T),
     if(start,
        (write('func('),retractall(start)),
         write(',func(')),
    makePattern(Funcname,Arity,Pattern),
     %clean2(Funcname,Funcname3),
	 writeq(Funcname),
     write(','),
     write(Arity),
     write(',['),
     wrTypeExpList(Types),
     write('],rule('),
	 %clean3(Pattern,Pattern2),				 
	 writeq(Pattern),
	 write(','),
%  nl,nl,write('tree:'),nl,
%     write(T),
%  nl,nl,write('rule:'),nl,
     getRule(Pattern,Pattern,T,Rule),writeq(Rule),
     write('))'),
     fail.
flattenMainFunctions.

wrTypeExpList(T1):-var(T1),!,wrTypeExp(T1).
wrTypeExpList((T1->T2)):-!,wrTypeExp(T1),write(','),wrTypeExpList(T2).
wrTypeExpList(T1):-!,wrTypeExp(T1).

wrTypeExp(T1):-var(T1),!,
    write('tvar('),writeq(T1),write(')').
wrTypeExp((T1->T2)):-!,
    write('functype('),wrTypeExp(T1),write(','),wrTypeExp(T2),write(')').
wrTypeExp(T1):-!,T1=..[Tcons|TexpL],
    write('tcons('),clean(Tcons,Tcons2),writeTC(Tcons2),write(',['),wrTypeExpL(TexpL),write('])').

wrTypeExpL([]):-!.
wrTypeExpL([T]):-!,wrTypeExp(T).
wrTypeExpL([T|Ts]):-wrTypeExp(T),write(','),wrTypeExpL(Ts).

correctVar(P,PL,X,C):-
     P=..[_|Args],PL=..[_|Argsl],selectVar(Args,Argsl,X,C).
selectVar([],[],X,X):-!.
selectVar([A|_],[B|_],X,A):-X==B,!.
selectVar([_|As],[_|Bs],X,C):-selectVar(As,Bs,X,C).

getRule(P,PL,X,var(C)):-var(X),correctVar(P,PL,X,C),!.
getRule(_P,_PL,X,int(X)):-number(X),!.  
getRule(P,PL,X@Y,apply(Exp1,Exp2)):-getRule(P,PL,X,Exp1),getRule(P,PL,Y,Exp2),!.
getRule(P,PL,or(L,R),or(Exp1,Exp2)):-getRule(P,PL,L,Exp1),getRule(P,PL,R,Exp2),!.
getRule(P,PL,rule(L,R),Exp):-
	  unify(L,PL,[],_,SubstList),
	  substitute(R,SubstList,NewR),
	  substitute(L,SubstList,NewL),
	  getRule(P,NewL,NewR,Exp),!.

getRule(P,PL,branch(Pat,Pos,rigid,T),case(Exp,CaseExpList)):-
     getPos(Pat,Pos,Var),
     getRule(P,Pat,Var,Exp),getCaseExpList(P,PL,Pos,T,CaseExpList),!.
getRule(P,PL,branch(Pat,Pos,flex,T),fcase(Exp,CaseExpList)):-
     getPos(Pat,Pos,Var),
     getRule(P,Pat,Var,Exp),getCaseExpList(P,PL,Pos,T,CaseExpList),!.

getRule(P,PL,localVars(X,Y),Constr):-getConstrain(P,PL,localVars(X,Y),Constr),!.
getRule(P,PL,localInPar(X,Y),Constr):-getConstrain(P,PL,localVars(X,Y),Constr),!.
getRule(P,PL,cond(X,Z),guardedexpr([],Constrain,Expr)):-var(X),!,
	  getConstrain(P,PL,X,Constrain),
	  getRule(P,PL,Z,Expr).
getRule(P,PL,cond(localVars(X,Y),Z),guardedexpr(X,Constrain,Expr)):-
     getConstrain(P,PL,Y,Constrain),getRule(P,PL,Z,Expr),!.
getRule(P,PL,cond(Y,Z),guardedexpr([],Constrain,Expr)):-
     getConstrain(P,PL,Y,Constrain),getRule(P,PL,Z,Expr),!.
getRule(P,PL,choice([(localVars(X,Y),Z)|Rest]),choice([GExp|GExpList])):-
     getRule(P,PL,cond(localVars(X,Y),Z),GExp),getRuleList(P,PL,Rest,GExpList),!.

getRule(P,PL,X,comb(Func2,RuleList)):-X=..[Func|Args],function(Func,_,_),
     clean(Func,Func2),
     getRuleList(P,PL,Args,RuleList),!.
getRule(P,PL,X,comb(Cons2,RuleList)):-X=..[Cons|Args],constructor(Cons,_,_,_),
     clean(Cons,Cons2),
     getRuleList(P,PL,Args,RuleList),!.

getRule(P,PL,X,comb(=,RuleList)):-X=..[=|Args],getRuleList(P,PL,Args,RuleList),!.
getRule(P,PL,X,comb(==,RuleList)):-X=..[==|Args],getRuleList(P,PL,Args,RuleList),!.
getRule(P,PL,X,comb(/\,RuleList)):-X=..[/\|Args],getRuleList(P,PL,Args,RuleList),!.
getRule(P,PL,X,comb(<--,RuleList)):-X=..[<--|Args],getRuleList(P,PL,Args,RuleList),!.
getRule(P,PL,$X,Exp):-getRule(P,PL,X,Exp),!.  % Discard leading '$'
getRule(P,PL,X,Constrain):-typeTerm(X,constraint,_),getConstrain(P,PL,X,Constrain),!.
getRule(_P,_PL,X,('### Not Implemented ###')):-nl,writeq(X),nl.

getConstrain(P,PL,X,constr([],C)):-var(X),!,getRule(P,PL,X,C).
getConstrain(P,PL,localVars(X,Y),constr(X,Exp)):-!,
     %getRuleList(P,PL,X,VarList),
	  getRule(P,PL,Y,Exp),!.
getConstrain(P,PL,X,constr([],Exp)):-getRule(P,PL,X,Exp),!.

getCaseExpList(_P,_PL,_LPos,[],[]):-!.
getCaseExpList(P,_PL,Pos,[rule(L,R)],[vpattern(Cons,Expr)]):-
     getPos(L,Pos,Cons),var(Cons),
     getRule(P,L,R,Expr),!.
getCaseExpList(P,_PL,Pos,[branch(Pat,LPos,Flag,T)],[vpattern(Cons,Expr)]):-
     getPos(Pat,Pos,Cons),var(Cons),
     getRule(P,Pat,branch(Pat,LPos,Flag,T),Expr),!.



getCaseExpList(P,PL,Pos,[rule(L,R)|Ts],[pattern(Consname2,VarL,Expr)|CaseExpList]):-
	 unify(L,PL,[],_,SubstList),
	 substitute(R,SubstList,NewR),
	 substitute(L,SubstList,NewL),
     getPos(NewL,Pos,Cons),
     Cons=..[Consname|VarL],%getRuleList([],[],VarL,VarList),
     clean(Consname,Consname2),
	 getRule(P,NewL,NewR,Expr),
     getCaseExpList(P,PL,Pos,Ts,CaseExpList),!.
	  
argl2.
getCaseExpList(P,PL,Pos,[branch(Pat,LPos,Flag,T)|Ts],[pattern(Consname2,VarL,Expr)|CaseExpList]):-
	 unify(Pat,PL,[],_,SubstList),
	 substitute(Pat,SubstList,NewPat),
	 getPos(NewPat,Pos,Cons),
     Cons=..[Consname|VarL],%getRuleList([],[],VarL,VarList),
     clean(Consname,Consname2),
	 getRule(P,NewPat,branch(NewPat,LPos,Flag,T),Expr),
     getCaseExpList(P,PL,Pos,Ts,CaseExpList),!.
getCaseExpList(P,PL,Pos,[or(Exp1,Exp2)|Ts],[pattern(Consname2,VarL,Expr)|CaseExpList]):-
argl2,
	 getArg(Exp1,Pos,Cons,Pat),
	 unify(Pat,PL,[],_,SubstList),
	 substitute(Pat,SubstList,NewPat),
	 getPos(NewPat,Pos,Cons3),
     Cons3=..[Consname|VarL],%getRuleList([],[],VarL,VarList),
     clean(Consname,Consname2), %getRule([],[],Consname2,Consname3),
     getRule(P,NewPat,or(Exp1,Exp2),Expr),
     getCaseExpList(P,PL,Pos,Ts,CaseExpList),!.

getArg(or(A,_),Pos,C,T):-getArg(A,Pos,C,T),!.
getArg(rule(L,_),Pos,C,L):-getPos(L,Pos,C),!.
getArg(branch(Pat,_,_,_),Pos,C,Pat):-getPos(Pat,Pos,C).

getPos(F,Pos#Path,Res):-number(Pos),F=..[_|Args],ith(Args,Pos,X),getPos(X,Path,Res).
getPos(F,Pos,Res):-number(Pos),F=..[_|Args],ith(Args,Pos,Res).

getRuleList(_,_,[],[]).
getRuleList(P,PL,[X|Rest],[Exp|ExpList]):-getRule(P,PL,X,Exp),getRuleList(P,PL,Rest,ExpList).

%%%%%%%%%%%%%%%%%%%%%
% write translations

%flattenTrans:-
%     symbtab(T),
%     writesymbTab(T),
%     wrMainFuncs.

writesymbTab([]):-!.
writesymbTab([(Name,InternalName)|RestTab]):-
     write('trans('),clean(Name,Name2),writeTC(Name2),write(','),
     clean(InternalName,InternalName2),
     writeTC(InternalName2),write(')'),
     if(RestTab=[],
        true,
        (write(','),writesymbTab(RestTab))).

wrMainFuncs:-
       root_module(R),function(Func,_,_),name(Func,Funcname),append(R,[95|RestName],Funcname),name(Func2,RestName),
       if(start,
            (write('trans('),retractall(start)),
            write(',trans(')),
       writeq(Func2),write(','),writeq(Func),write(')'),
       fail.
wrMainFuncs.
	   

wrMainCons:-
	    root_module(R),constructor(Cons,_,_,_),name(Cons,Consname),append(R,[95|RestName],Consname),name(Cons2,RestName),
       if(start,
            (write('trans('),retractall(start)),
            write(',trans(')),
       writeq(Cons2),write(','),writeq(Cons),write(')'),
       fail.
wrMainCons.	   
%----------------------- END COMPILER-INTERFACE ------------------------


readIn(File):-
  if(
       preludepath(PP), 
       true, 
       PP=prelude
    ),
  checkPrelude,
  if(
       readProg(File, PP, SymbTab,ResultGIL,AllKnownNames),
       (
          safe(retractall(symbtab(_))),
          asserta(symbtab(SymbTab)),
          safe(retractall(globalTable(_))),
          asserta(globalTable(ResultGIL)),
          makeTrees,
          !,
          if(
               noInterface(_),
               (
                  generateInterfaces(ResultGIL, AllKnownNames),
                  nl,write('Interfaces generated. Re-reading program ...'),
                  nl,flush_output,
                  readIn(File)       %once again for implicit type importing
               ),
               true
            )
       ),
       (
          seen,told,removeTempFiles,!,
          format(user_error, "*** Reading failed ***\n",[]),fail
       )
    ).



generateInterfaces(ResultGIL, AllKnownNames):-
  % in the ResultGIL we do not have any prefixing
  if(
       noInterface(Module),
       (
          root_module(ROOTMod),
          buildTranslationTable(AllKnownNames, Module, TrTable, ROOTMod),
          append(_,[(_, ModList)|_], ResultGIL),
          append(_, [(Module,Imports,ExportsWithoutRens,_)|_], ModList),
          !,
          addExtension(Module, '.int', FileName),
          datime(datime(DY,DM,DD,DH,DMi,DS)),   %think about it ...

          % we need the data types occurring in function signatures
          addExtension(Module, '.fl', FFS),
          format(user_output, 
                 "re-reading implementation file '~w' ...\n",
                 [FFS]),
          see(FFS),
          getDataDecls([], DatDecl, [], Renamings), !,
          append(Renamings, ExportsWithoutRens, Exports),
          seen,
          format(user_output,
                 "generating interface file '~w' ...\n",
                 [FileName]),
          tell(FileName),
          write('% generated automatically '),
          write(DM),write('/'),write(DD),write('/'),write(DY),
          write(', '),
          write(DH),write(':'),write(DMi),write(':'),
          write(DS),nl,nl,
          write('interface '), write(Module), write('.'),nl,
          modExportsMod(Module, ExportedModules),
          !,
          deleteSomeRenamings(Exports, _NewExports),
          remove_duplicates(_NewExports, NewExports),
          !,
          if(
                ExportedModules==[],     %export no modules
                true,
                if(
                     ExportedModules==[module],  %export all modules
                     (
                        if(
                             Imports==[],
                             true,
                             (
                                  writeRenamings(Imports, NewExports)
%                                 write('  import '),
%                                 write(Imports), 
%                                 write('.'),nl,nl
                             )
                           ),
                        NewExL=Imports
                     ),
                     (
                        checkExportedImpEnts(ExportedModules, [], NewExL),
                        if(
                             NewExL==[],
                             true,
                             (
                                 writeRenamings(NewExL, NewExports) 
 %                               write('  import '),
 %                               write(NewExL), 
 %                               write('.'),nl,nl
                             )
                          )
                     )
                  )
            ),
          if(
               var(NewExL),
               NewExL=[],
               true
            ),
          asserta(modImportsMod(NewExL)),

          generateDataAndFTypes(NewExports, Module, TrTable, DatDecl),
          retractall(modImportsMod(_)),
          !,
          nl,nl,
          told,
          retractall(noInterface(Module)),
          !,
          generateInterfaces(ResultGIL, AllKnownNames)
       ),
       (true,!)
    ).


writeRenamings([],_).
writeRenamings([I|Is], NewExports):-
  write('  import '),write(I),
  I=..[IName|ImportedEntities1],
  ownflatten(ImportedEntities1, ImportedEntities),
  % renamings only for explicitely exported entities (or all if
  % the whoel module is exported, i.e. the signature is empty.
  if(ImportedEntities==[],
      findall((From, To), member(r(From,To,IName), NewExports), _Rens),
      findall((From, To), ( member(r(From,To,IName), NewExports),
                            member(E, ImportedEntities),
                            functor(E, From, _)
                          ),
              _Rens)
    ),
  remove_duplicates(_Rens, Rens),
  writeRenamings(Rens),
  write('.'),
  nl,
  writeRenamings(Is, NewExports).

writeRenamings([]).
writeRenamings([(From, To)|Rest]):-
  nl,
  write('         '),
  write('renaming '),
  writeq(From),
  write(' to '),
  writeq(To),
  writeRenamings(Rest).



checkExportedImpEnts([], L, L).
checkExportedImpEnts([M|Ms], OldML, NewML):-
  functor(M,MName,_),
  addExtension(MName, '.fl', ImF),
  addExtension(MName, '.int', InF),     
  if(
       (file_exists(ImF);file_exists(InF)),
       NewM=[M|OldML],
       (
          format(user_error,
                  "+++ Warning: exported module '~w' does not exist \c
                   ... ignoring.\n",
                   [MName]),
          NewM=OldML
       )
    ),
  checkExportedImpEnts(Ms, NewM, NewML).



getDataDecls(Accu1, Data, Accu2, Renamings):-
  read(Line),
  if(
       Line==end_of_file, 
       (
           Data=Accu1,
           Renamings=Accu2
       ),
       if(
            Line=(data A=B),
            % od for "originally declared"
            getDataDecls([od(A=B)|Accu1], Data, Accu2, Renamings),
            if(
                  Line=(import F),
                  (
                     parseImport((import F), _, IRs),
                     ownappend(IRs, Accu2, Accu3),
                     getDataDecls(Accu1, Data, Accu3, Renamings)
                  ),          
                  getDataDecls(Accu1, Data, Accu2, Renamings)
              )
         )
    ).



writeInterfaceDecls([], L, L).

writeInterfaceDecls([id(Name=Consts)|Rest], Data, NewData):-
  !,
  write('  data '),
           extendedWrite(Name=Consts),write('.'),nl,
  functor(Name, Func, _),
  % w for written
  writeInterfaceDecls(Rest, [w(Func)|Data], NewData).

writeInterfaceDecls([id(Name)|Rest], Data, NewData):-
  !,
  write('  data '),
  extendedWrite(Name),write('.'),nl,
  % w for written; Name is nullary
  writeInterfaceDecls(Rest, [w(Name)|Data], NewData).

writeInterfaceDecls([_|Rest], Data, NewData):-
  writeInterfaceDecls(Rest, Data, NewData).


       
writeFuncSigTypes([], _, _).

%write only once!
writeFuncSigTypes([fd(Name)|Rest], Data, N):-
  member(w(Name), Data),
  !,
  writeFuncSigTypes(Rest, Data, N).

writeFuncSigTypes([fd(Name)|Rest], Data, N):-
  member(od(L=_), Data),   %=R for export with all constructors
  functor(L, Name, _),
  !,
  if(
       N==0,
       (write('  % implicitely exported:'), nl, M is N+1),
       M=N
    ),
  functor(L, LFunc, _),                 % omit param.
  write('  data '),
  extendedWrite(LFunc),write('.'),nl,
%  \+ \+ (
%           numbervars(L=R, 0, _),         %=R with all Consts.
%           extendedWrite(L=R),write('.'),nl   %L=R with all Consts.
%        ),
  writeFuncSigTypes(Rest, [w(Name)|Data], M).

writeFuncSigTypes([_|Rest], Data, N):-
  writeFuncSigTypes(Rest, Data, N).


% end of recursion
generateDataAndFTypes([], _, _, Data):-
  remove_duplicates(Data, CleanData),
  if(
       member(id(_), CleanData),
       (
          !,
          write('  % explicitely exported:'),nl,
          writeInterfaceDecls(CleanData, CleanData, NewData)
       ),
       NewData=Data
    ),
  !,
  % still missing: subterms in function signatures declared in this
  % module and *not* already written. (implicit exports)
  writeFuncSigTypes(NewData, NewData, 0).

generateDataAndFTypes([c(_,_,_)|Rest], Mod, TT, DD):-
  generateDataAndFTypes(Rest, Mod, TT, DD).

generateDataAndFTypes([f(Name, dontcare, Prio, Kind, Mod, _)|Rest], Mod, TrTable, DD):-
  !,
  root_module(ROOT),
  NotVars=[],
  renTermDetVars(Name, TrTable, NewName, Mod, ROOT,NotVars,[],_),
  typeTerm(NewName, Type, _),   %getType
  unrename3(Type, NewType, TrTable),
  getSubterms(NewType, Subterms),
  if(
       Kind==ixl,
       (
          write('  '),write('infixl('),
          write(Prio), write(',('),
          extendedWrite(Name),write(')).'),nl,
          Par=yes
       ),
       if(
            Kind==ixr,
            (
               write('  '),write('infixr('),
               extendedWrite(Prio), write(',('),
               extendedWrite(Name),write(')).'),nl,
               Par=yes
            ),
            if(
                 Kind==ix,
                 (
                    write('  '),write('infix('),
                    write(Prio), write(',('),
                    extendedWrite(Name),write(')).'),nl,
                    Par=yes
                 ),
                 Par=no
              )
         )
     ),
       (
          %  numbervars(NewType, 0, _),
           write('  '), 
           if(
                true,  %Par==yes,  (e.g. >> without opdecl)
                (write('('),extendedWrite(Name),write(')')),
                extendedWrite(Name)
             ),
           write('::'),
           extendedWrite(NewType),write('.'),nl
        ),

  % no nondeterminism
  generateDataOfFuncSigs(Subterms,DD, NewDD),
  generateDataAndFTypes(Rest, Mod, TrTable, NewDD).

% fd for 'function declaration'
generateDataOfFuncSigs([], L, L).
generateDataOfFuncSigs([ST|STs], DataList, NewDataList):-
  generateDataOfFuncSigs(STs, [fd(ST)|DataList], NewDataList).




% we do not need functions that are declared elsewhere
generateDataAndFTypes([f(_, dontcare, _, _, _, _)|Rest], Mod, TT, DD):-
  !,
  generateDataAndFTypes(Rest, Mod, TT, DD).

% explicit type signature
generateDataAndFTypes([f(Name, Type, Prio, Kind, Mod, _)|Rest], Mod, TT, DD):-
  !,
  if(
       Kind==ixl,
       (
          write('  '),write('infixl('),
          write(Prio), write(',('),
          extendedWrite(Name),write(')).'),nl,
          Par=yes
       ),
       if(
            Kind==ixr,
            (
               write('  '),write('infixr('),
               write(Prio), write(',('),
               extendedWrite(Name),write(')).'),nl,
               Par=yes
            ),
            if(
                 Kind==ix,
                 (
                    write('  '),write('infix('),
                    write(Prio), write(',('),
                    extendedWrite(Name),write(')).'),nl,
                    Par=yes
                 ),
                 Par=no
              )
         )
     ),
      (
          % numbervars(Type, 0, _),
           write('  '), 
           if(
                true,    %Par==yes, (e.g. >> without opdecl)
                (write('('),extendedWrite(Name),write(')')),
                extendedWrite(Name)
             ),
           write('::'),
           extendedWrite(Type),write('.'),nl
        ),
  generateDataAndFTypes(Rest, Mod, TT, DD).

% other module:
generateDataAndFTypes([f(_, _, _, _, _, _)|Rest], Mod, TT, DD):-
  !,
  generateDataAndFTypes(Rest, Mod, TT, DD).
  
generateDataAndFTypes([r(_From, _To, _In)|Rest], Mod, TT, DD):-
  generateDataAndFTypes(Rest, Mod, TT, DD).

generateDataAndFTypes2([r(From, To, In)|Rest], Mod, TT, DD):-
format(user_output, "YES:  ~w ~w ~w\n\n", [From, In, To]),
   % renamings only for modules in the header (or all)
  if(
       (
          modImportsMod(ImportedModules),
          member(M, ImportedModules),
          M=..[In|ImportedEntities1],
          ownflatten(ImportedEntities1, ImportedEntities)
       ),
       if(
            (
               (ImportedEntities==[]);
               (member(E, ImportedEntities), functor(E,From,_))
            ),
            (
               !,
               write('  renaming '), extendedWrite(From),write(' in '),
               extendedWrite(In), write(' to '), extendedWrite(To),write('.'),nl
            ),
            !
         ),
       !
    ),
  generateDataAndFTypes(Rest, Mod, TT, DD).

generateDataAndFTypes([r(_, _, _)|Rest], Mod, TT, DD):-
  !,
  generateDataAndFTypes(Rest, Mod, TT, DD).

% partial or full constructor export:
% id for 'interface declaration'
generateDataAndFTypes([d((Name=Consts),Mod,_)|Rest], Mod, TT, DD):-
  !,
  generateDataAndFTypes(Rest, Mod, TT, [id(Name=Consts)|DD]).

% another module:
generateDataAndFTypes([d(_=_,_,_)|Rest], Mod, TT, DD):-
  !,
  generateDataAndFTypes(Rest, Mod, TT, DD).

% no constructor export:
% id for 'interface declaration'
generateDataAndFTypes([d(Name,Mod, _)|Rest], Mod, TT, DD):-
  !,
  generateDataAndFTypes(Rest, Mod, TT, [id(Name)|DD]).

% other modules:
generateDataAndFTypes([d(_,_,_)|Rest], Mod, TT, DD):-
  !,
  generateDataAndFTypes(Rest, Mod, TT, DD).

% deletes renamings of entities that are not eported
%deleteSomeRenamings(E,E).
deleteSomeRenamings(E, NewE):-
  findall(r(From,To,In), member(r(From,To,In), E), Renamings),
  deleteSomeRenamings(Renamings, E, NewE).

deleteSomeRenamings([], L, L).
deleteSomeRenamings([r(From,To,In)|Rs], Entities, Result):-
  \+ ((member(d(D1=_,_,_), Entities), functor(D1, To, _))
      ;(member(d(D2,_,_),Entities), functor(D2, To, _))
      ;(member(c(C,_,_), Entities), functor(C,To,_))
      ;(member(f(F,_,_,_,_,_), Entities), functor(F,To,_))
     ),
  !,
  delete(Entities, r(From,To,In), NewE),
  deleteSomeRenamings(Rs, NewE, Result).
deleteSomeRenamings([_|Rs], Entities, Result):-
  deleteSomeRenamings(Rs, Entities, Result).

% $ or # at the beginning of a name !!!       
addApostrophes(OldTerm,OldTerm):-var(OldTerm),!.
addApostrophes(OldTerm,NewTerm):-
  OldTerm=..[Functor|Args],
  if(
        % if one decides to write ($io) instead of '$io':
       ((Functor=='$');(Functor=='#')),
       (
          [Args2]=Args,
          functor(Args2,FirstArg,_),
          Args2=..[_|RestArg],
          FirstArg=..[FirstArgFunctor|_],
          name(FirstArgFunctor,FirstArgFunctorAsc),
          append([39,36|FirstArgFunctorAsc],[39],FAFAWA),
          name(WithApo, FAFAWA),
          addAposArgs(RestArg, [], NewFirstArg),
          NewFunctor=..[WithApo|NewFirstArg],
          ArgList=[],nl
       ),
       (
          name(Functor,AsciiFunctor),
          if(
               ((AsciiFunctor=[36|_]);   %$ or #
                (AsciiFunctor=[35|_])),
               (
                   append([39|AsciiFunctor],[39],NewFunctorAscii),
                   name(NewFunctor,NewFunctorAscii)
               ),
               NewFunctor=Functor
            ),
          ArgList=Args
       )
    ),
  addAposArgs(ArgList,[],NewArgs),
  if(
       NewArgs==[],
       NewTerm=NewFunctor,
       NewTerm=..[NewFunctor|NewArgs]
    ).


addAposArgs([],A,A).
addAposArgs([A1|Rest],Accu,Result):-
  addApostrophes(A1,Res1),
  append(Accu,[Res1],NewAccu),
  addAposArgs(Rest,NewAccu,Result).


safe(X):-if(X,true,true).
%damit auch weitergeht, falls X fail liefert. z.B. bei X=readIn kann das
%readIn durch einen Syntaxfehler ein fail liefern, das Programm muss aber troztdem
%true liefern.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% compute(Goal,OldNames)
%
% - computes the goal, the user has specified.
% - Oldnames = list of all variables that appear in goal. Needed for output

compute(InputGoal,OldNames,SmallVars):- 
        retractall(io_monad),
        retractall(go),assert(go),
        retractall(currline(_)),assertz(currline(InputGoal)),
        ifc((transformExpr(InputGoal,Goal1),                      % readin-Transforming
             transformLambda(Goal1,Goal1a),                 % transform lambda-expressions
             cTrans(Goal1a,Goal1a,(_Vars,_Locals,Goal2)),!, % create normal-form
            typeTerm(Goal2,Type,NewGoal1),                        % Typcheck
            retransformLocal(NewGoal1,NewGoal),                   % retransform local-operators
            (isIO(Type) -> assert(io_monad) ; true),!,
            (containsIO(Type)
               -> write('Error: I/O actions in data structures not allowed at top-level!'),
                  nl,!,fail
                ; true),
            statistics(runtime,[StartTime,_]),!,
            rnstern([([],[],NewGoal)],OldNames,NewNames,Result,0), % compute the goal
            statistics(runtime,[StopTime,_])),
           (addSmallVars(SmallVars, [], Small2), % add small vars' names
            ownappend(Small2,OldNames,OldNames2),
            ownappend(Small2,NewNames,NewNames2),
            clearResult(Result,OldNames2,Result1),
%kill all substitutions of variables that have not been in the original goal
            collectNewVars(Result1,NewNames2,0,NewNames1),
%if trace-mode was not activated or interrupted, then there might be unnamed variables
%in the goal, e.g. as A=s(_123). Those must be named by collectNewVars.
%If trace-mode was activated, there are no such variables, so collectNewVars does nothing.

            makeNames(NewNames1,_Names),nl,  
            safe(symbtab(NTab)),
            showResult(Result1, NTab),                % formatierte Ausgabe
            ifc(showexectime,
                (nl, nl, write('Execution time: '),
                 Time is (StopTime-StartTime)/1000,
                 write(Time), write(' sec.')),
                true),
               clearSearchLambdas),

            (nl,write('Error occured.'))).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Tools for compute


addSmallVars([],L,L).
addSmallVars([(Name,Var)|Rest], Accu, Result):-
  addSmallVars(Rest, [(Name=Var)|Accu], Result).

% is the type an I/O type?
isIO(Type):-nonvar(Type),Type=io(_).

% does the type contain an I/O type?
containsIO(Type) :- var(Type), !, fail.
containsIO(Type) :-
        Type =.. [_|TArgs],
        haveIO(TArgs).

haveIO([T|_]) :- hasIO(T).
haveIO([_|Ts]) :- haveIO(Ts).

hasIO(T) :- var(T), !, fail.
hasIO(io(_)) :- !.
hasIO(T) :- containsIO(T).

targetTypeOfArgNotIO(Term,Func,Arg):-nonvar(Arg),
	Arg=..[Head|_Args],
	function(Head,_,_),
	ergType(Head,Type),
	isIO(Type),
	write('*** Error: '),
	writeterm(Term),nl,
	write('Function of type IO not allowed as argument of "'),
	write(Func),write('"!'),nl,!,fail.
targetTypeOfArgNotIO(_,_,_).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% transformLambda(Term, NewTerm)
%
% - transforms lambda-abstractions from the user's input into
%   $searchLambda-expressions. Currently, only abstractions with
%   a body of type "constraint" are allowed.
% - NOT MUCH TESTED YET, JUST A QUICK HACK
% - NESTED LAMBDA-ABSTRACTIONS ARE NOT ALLOWED


transformLambda(Term,NewTerm):-
        retractall(currline(_)),assertz(currline(Term)),
%        transformExpr(Term,Term1),
        transformLambda2(Term,NewTerm).


transformLambda2(Var,Var):-var(Var),!.

transformLambda2(\Var,_):-var(Var),!,writeTypeError("Wrong definition of lambda-expression",[]).

transformLambda2(\ (Var->_Term),_):- \+ var(Var),!,writeTypeError("Variable after \\ expected",[]).

transformLambda2(\ (Var->Term),NewTerm):-!,
        transformLambda3(Var,Term,NewTerm).
transformLambda2(_,\_Term,_,_):-!,writeTypeError("Wrong definition of lambda-expression",[]).


transformLambda2(Term,NewTerm):-
        Term=..[H,Arg|Args],!,transformLambda2(Arg,NewArg),
        Term2=..[H|Args],transformLambda2(Term2,Term3),
        Term3=..[H|NewArgs],
        NewTerm=..[H,NewArg|NewArgs].

transformLambda2(Term,Term).



transformLambda3(X,Body,LambdaCall):-
        typeTerm(Body,Type,_NewBody),!,
        ifc(Type==constraint,
            (freeVars(Body,GlobalVars),
             removeVarFromSet(X,GlobalVars,VarList),
             makeLambda(X,[],VarList,Body,$LambdaCall)),
            writeTypeError("Wrong definition of lambda-expression, only constraints allowed as bottom at command line",[])).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% rnstern(Input,OldNames,NewNames,OutPut,Steps)
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln
%
%rnstern: Mehrschrittreduktion bis zum Ende
% OldNames = Liste der Variablennamen im Goal. Fuer Trace-Ausgabe.
% Bei Trace wird evtl. Liste upgedatet, deshalb NewNames

rnstern([],_,_,[],_).
rnstern(Eingabe,OldNames,NewNames,Ausgabe,S):-
        rn(Eingabe,A2),
        if(Eingabe==A2,
           (Ausgabe=A2,NewNames=OldNames
%,nl,write(S),nl
            ),
           (showTrace(A2,OldNames,NewNames1),
            if(go,              
               (
%               S1 is S+1, % only for time out
%               (S<2000 ->  % only for time out
                             (!,rnstern(A2,NewNames1,NewNames,Ausgabe,S1))
%                         ; write('Time out: more than 2000 reduction steps (probably infinite computation)'), nl % only for time out
%               ) % only for time out
               ),
               (Ausgabe=A2,NewNames=OldNames))
           )
          )
         ,!.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Tools for rnstern


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% compose(Term,Result,Subst,NewResult)
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln
%
%compose verknuepft in einer Liste von Goals jede der Substitutionen
%mit der uebergebenen Substitution
%Falls nur ein suspendiertes Goal ex. wird der vorige Term unveraendert
%zurueckgeliefert

compose(Term,[(V,Subst,X)],Subst,[(V,Subst,Term)]):-nonvar(X),  
        X=suspend(_),!.
compose(_,[],_,[]).
compose(_,[(V_i,Sigma_i,Goal_i)|Xs],Sigma,[(V_i,NewSigma_i,Goal_i)|Rest]):-
        appendSubst(Sigma,Sigma_i,NewSigma_i),
        compose(_,Xs,Sigma,Rest).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% showTrace(Result,OldNames,NewNames)
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln

showTrace(Result,OldNames,NewNames):-
        if((currytraceflag;singlestep),
            showGoals(Result,OldNames,NewNames),
            NewNames=OldNames),   
% Debug-Option an: Zwischenergebnis ausgeben
                if(singlestep,
%Singlestep-Option an, dann Auswahlmenue
                   optmenue,
                   true).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% optmenue
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln

optmenue:-
        write('g: go, t: trace, n: notrace, a: abort, any other key: single step  >'),
        get0(C),nl,
        debug_option(C,_),        % Eingabe pruefen
        if(member(C,[110,116]),optmenue,true).
% letzte Zeile bewirkt, dass bei Eingabe von trace/notrace 
% das Ergebnis und die Auswahlzeile nochmal angezeigt werden




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% showGoals(Result,OldNames,NewNames)
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln

showGoals(Result,OldNames,NewNames):-
        collectNewVars(Result,OldNames,0,NewNames),
        \+ \+ (makeNames(NewNames,_),showResult(Result,[])),nl.
%makeNames gibt den Varialen aus der Eingabe die richtigen Namen



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% debug_option(..)
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln

debug_option(103,false):-get0(_),   % bei g singlestep ausschalten (durchlaufen)
        retractall(singlestep).
%das get0 faengt das Return-Zeichen ab
debug_option(116,true):-get0(_),    % bei t trace-Modus einschalten
        keyWords(trace,_),nl.
debug_option(110,true):-get0(_),    % bei n trace-Modus abschalten
        keyWords(notrace,_),nl.
debug_option(97,false):-get0(_),    % bei a abbrechen
        retractall(go),nl.
debug_option(_,false).                 
% alles andere tut gar nichts (insbesondere bleibt man im Singlestep-Modus)







%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% rn 
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln

% RN: Arbeitet auf Liste von Goals, greift das am weitesten links stehende
% Goal raus, das nicht suspendiert ist, und wertet es einen Schritt weit aus
% ein suspend wird ueberlaufen,
% ein zurueckgeliefertes suspend wird eingereiht,

%rn([(V,Subst,X)|Xs],[(V,Subst,X)|Xs]):-var(X),!.
rn([],[]).
rn([X|Xs],[X|Result1]):-suspended([X]),
        rn(Xs,Result1).


rn([(V,Subst,Goal)|Xs],Result):-!,
        cseX(V,Goal,false,Result1),!,
        ifc(suspended(Result1),
           (rn(Xs,NewXs),Result=[(V,Subst,suspend(Goal))|NewXs]),
           (compose(Goal,Result1,Subst,NewResult),
            append(NewResult,Xs,Result),!,
            (io_monad -> onlyOneWorld(Result,Goal) ; true)
            )).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Tools for rn



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% onlyOneWorld
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln
%

onlyOneWorld([_,_|_],InitGoal) :- !,
        nl, write('*** Error: Cannot duplicate the world!'), nl,
        write('Goal: '),
        write(InitGoal), nl,
        write('produces a disjunction of I/O actions!'), nl, nl,
        fail.
onlyOneWorld(_,_).






%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% cseX(V,Term,LocalFlag,Result)
%
% - the cases are ordered that the special cases come first.
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln

% cseX trifft die Vorauswahl: Gleichheit, Variable, Konstruktorterm, Funktionsapplikation oder Funktion.


cseX(V,X,_Local,[(V,[],suspend(X))]):-var(X),!.       % Variablen -> suspend


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% local-Deklarationen

cseX(V,_List localVars T,false,[(V,[],T)]):-!. % local-flag=false  => do not collect vars in V

cseX(V,List localVars T,true,[(NewV,[],T)]):-!,  % local-flag=true => collect vars in V
        append(V,List,NewV).

cseX(V,List localInPar T,false,[(V,[],NewT)]):-!,   % local-flag=false => do not collect vars in V 
        makeSubstForLocalVars(List,Subst,_NewVars), % create new vars and rename the declared   
        substitute(T,Subst,NewT).                   % ones by the newly created

cseX(V,List localInPar T,true,[(NewV,[],NewT)]):-!, % local-flag=true => collect vars in V
        makeSubstForLocalVars(List,Subst,NewVars),  % create new vars,
        append(V,NewVars,NewV),                     % collect them in V, and
        substitute(T,Subst,NewT).                   % rename the declared by the new ones


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% stuff for local searchSpaces
%
% - according to S47ff, S63ff
%
% - If we meet searchLambda (not as parameter of try), we must transform
%   it into a lambda-abstraction, because the parameter of searchLambda
%   is a searchSpace that was given back by cseXSearch. We should have
%   transformed it via return at the time cseXSearch gave it back, but we
%   delayed it, because if it is again passed as parameter to "try" we
%   can restart computing immediately. Otherwise, we had packed the
%   result into a lambda-abstraction and would now have to unpack it
%   already again. Very inefficient.
% - a variable as parameter for try suspends.
% - if try has a searchLambda-parameter, this is a result of 
%   a former cseXSearch, that returned several soulutions.
%   The parameter of searchLambda is a searchSpace, so just call
%   cseXSearch as in the usual case.
% - Otherwise, the try-Operator is transformed into a local searchSpace.

% - to compute a searchSpace, we try to evaluate the Term
%   via cseX with the local-flag=true. This will compute Term
%   according to the special evaluation mechanisms that we have
%   defined for local spaces. Call cseXlocal to evaluate the
%   result. cseXlocal can return a new searchSpace or an instable
%   searchSpace or it can (auflsen) a searchSpace and return its
%   result(s) in a list of lambda-abstractions.
% - if a searchSpace became instable, it contained global variables
%   and so could not create several copies of a searchSpace before
%   binding these variables, that we kept in the GlobalV-parameter.
%   nobound checks, if GlobalV is a list of variables. If so, no
%   one of the global variables was bound up to now, and so the
%   searchSpace can not evaluate further. So suspend it.
%   If nobound fails, at least one of the global variables was bound
%   and so we can again try to evaluate the searchSpace. Perhaps the
%   binding of the global variable(s) enables it to reduce now.
%   NOTICE: The instable case works for works for S=searchSpace(..) 
%   AND S=choiceSpaces(..) !
%
% - notice that the apply-predicate had to be changed to match the
%   case "searchLambda(..)@X" and "unboundVars(..)@X".


cseX(V,searchLambda(X,VLocal,Sigma,Term),_Local,[(V,[],LambdaCall)]):-
        tl(X,VLocal,Sigma,Term,LambdaCall).


cseX(V,try(searchLambda(X,VLocal,Sigma,Term)),_Local,[(V,[],Result)]):-nonvar(VLocal),
%nonvar(VLocal) makes sure that the parameter of try is not a variable
        cseX(VLocal,Term,true,TempResult),
        cseXSearch(searchSpace(X,VLocal,Sigma,Term),TempResult,Result).

cseX(V,try(F),_Local,[(V,[],LSpace)]):-nonvar(F),!,createSearchSpace(F,LSpace). 

cseX(V,try(F),_Local,[(V,[],suspend(try(F)))]).


cseX(V,try(searchSpace(X,VLocal,Sigma,Term)),_Local,[(V,[],Result)]):-
        cseX(VLocal,Term,true,TempResult),
        cseXSearch(searchSpace(X,VLocal,Sigma,Term),TempResult,Result).


cseX(V,try(F),_Local,[(V,[],LSpace)]):-createSearchSpace(F,LSpace). 


cseX(V,searchSpace(X,VLocal,Sigma,Term),_Local,[(V,[],Result)]):-
        cseX(VLocal,Term,true,TempResult),
        cseXSearch(searchSpace(X,VLocal,Sigma,Term),TempResult,Result).

% works for S=searchSpace(..) AND S=choiceSpaces(..)
cseX(V,unboundVars(GlobalV,S),_Local,
              [(V,[],suspend(unboundVars(GlobalV,S)))]):-
        nobound(GlobalV).

cseX(V,unboundVars(_GlobalV,S),_Local,Result):-
        cseX(V,S,_Local,Result).


%cseX(V,browse(G),_Local,[(V,[],suspend(browse(G)))]):-var(G),!.
%cseX(V,browse($G),_Local,Result):-
%	G=..[GHead|GArgs],
%	tree(GHead,rule(L,R)),
%	L=..[GHead|Args],
%	last(Args,SearchVar),
%% test, if the body of the lambda-abstraction is solved.
%% If yes, only the right side of the body, where possibly locally declared
%% variables have been replaced by new ones, is returned. In case of an empty
%% body, a fresh variable is returned.
%	testIfRisSolved(SearchVar,Args,R,BodyWithoutLocals),!,
%% Test, if all global variables that are passed to the body are bound to
%% ground terms. % If yes, pass the ground terms to the body
%	if(groundTermList(GArgs),
%           (append(GArgs,[_Fresh],NewGArgs),
%	    FunctionCall=..[GHead|NewGArgs],
%	    leftUnify(L,FunctionCall,Subst),     
%            substitute(BodyWithoutLocals,Subst,NewBody),
%% and show the result on the screen.
%	    Result=[(V,[],printToScreen(NewBody))]),
%%If no, then suspend, because no variables other than
%% abstracted ones can be displayed on the screen!
%	   Result=[(V,[],suspend(browse('$'(G))))]).
%
%
%% Here we arrive if the body of the lambda-abstraction is not solved.
%cseX(_V,browse($G),_Local,_Result):-
%	write('*** Error: '),
%	writeterm(browse('$'(G))),nl,
%	write('Argument of browse is not a solved search goal!'),!,fail.
%
%% If the parameter of browse is not a partial function, it is a meta-construct,
%% i.e. try. Just compute the parameter then.
%cseX(V,browse(G),Local,Result):-
%	cseX(V,G,Local,ResultG),
%	replace(browse(G),1,ResultG,Result).

	
% split a list in it's last element and the rest.
splitForLast([X],[],Y):-Y=X.
splitForLast([X,Y|Xs],[X|GlobalVars],SearchVar):-
	splitForLast([Y|Xs],GlobalVars,SearchVar).

% test, if R is solved, i.e. it has the form {} or
% SearchVar=t with t a constructor term containing only the search
% variable and the global variables that are passed to the body, or
% [X1..Xn] localVars X=t. As the local variables have been replaced
% with fresh ones when getting the definitional tree, just drop them
% and proceed as in the second case, adding the local variables to
% the set of variables that can appear in the body.
%
% Return a new body: a fresh variable, if the body was {},
% the right side of the equation otherwise.

testIfRisSolved(_SearchVar,_Vars,{},_FreshVar).
testIfRisSolved(SearchVar,Vars,X=Term,Term):-X==SearchVar,!,
	constructorTermWithVars(Vars,Term).
testIfRisSolved(SearchVar,Vars,L localVars X=Term,Term):-X==SearchVar,!,
	append(L,Vars,NewVars),
	constructorTermWithVars(NewVars,Term).

constructorTermWithVars(Vars,X):-var(X),!,varmember(X,Vars).
constructorTermWithVars(Vars,T):-
        T=..[Head|Args],
        isCons(Head,_),!,
        constructorTermWithVarsList(Vars,Args).

constructorTermWithVarsList(_Vars,[]).
constructorTermWithVarsList(Vars,[X|Xs]):-
	constructorTermWithVars(Vars,X),!,
	constructorTermWithVarsList(Vars,Xs).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% stuff for choiceSpaces
%
% - the choice-operator is transformed into a list of choicespaces
% - compute the choiceSpaces through cseXChoice. This always 
%   gives back only ONE goal. If choice succeded, we will get back
%   a result of the form [(ChoiceV,[],E)], where E is the body
%   of the condition which was reduced to "{}". ChoiceV is the set
%   of variables that appears in E. If the local flag is true, we
%   have computed the choice-call in another choice or in a searchSpace.
%   So we must compute the variable set, and therefore give back
%   V united with ChoiceV. ChoiceV can contain variables that were
%   declared local in the C that reduced to {} (only then ChoiceV
%   can be not empty) and still appear in Sigma(E) that we give back.
%   This variables must be collected as they are new variables that
%   come back from reducing choice. So if we computed choice for
%   example in a searchSpace, these new variables are now local ones
%   to the searchSpace!
% - choiceResult is needed because cseXChoice might give back [] if
%   there was only one space left which failed then.
% - if all choiceSpaces became instable, test if at least one of the
%   contained global variables was bound. If so, retry to compute
%   the choiceSpaces. THIS IS DONE ABOVE at cseX(V,unboundVars(GlobalV,S)..)
%   which works for S=searchSpace(..) AND S=choiceSpaces(..) !
%
% - notice that the apply-predicate had to be changed to match the
%   case "choiceSpaces(..)@X" and "unboundVars(..)@X".


cseX(V,committedChoice(Args),_Local,[(V,[],choiceSpaces(Spaces))]):-!,createChoiceSpaces(Args,Spaces). 

cseX(V,choiceSpaces(Spaces),false,Result):-
        cseXChoice(Spaces,CResult),
        choiceResult(CResult,V,false,Result).

cseX(V,choiceSpaces(Spaces),true,Result):-
        cseXChoice(Spaces,CResult),
        choiceResult(CResult,V,true,Result).

choiceResult([],_V,_Local,[]).
choiceResult([(_ChoiceV,[],Result)],V,false,[(V,[],Result)]).
choiceResult([(ChoiceV,[],Result)],V,true,[(NewV,[],Result)]):-
        append(V,ChoiceV,NewV).


cseX(V,unboundVars(GlobalV,choiceSpaces(Spaces)),_Local,
               [(V,[],suspend(unboundVars(GlobalV,choiceSpaces(Spaces))))]):-
        nobound(GlobalV).

cseX(V,unboundVars(_GlobalV,choiceSpaces(Spaces)),Local,Result):-
        cseX(V,choiceSpaces(Spaces),Local,Result).


cseX(V,unboundVars(GlobalV,searchSpace(X,SV,Sigma,Term)),_Local,
              [(V,[],suspend(unboundVars(GlobalV,searchSpace(X,SV,Sigma,Term))))]):-
        nobound(GlobalV).

cseX(V,unboundVars(_GlobalV,searchSpace(X,SV,Sigma,Term)),_Local,Result):-
        cseX(V,searchSpace(X,SV,Sigma,Term),_Local,Result).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%


cseX(V,T1/\T2,Local,Result):- !,evalInterleavedAnd(V,T1,T2,Local,Result).

cseX(V,F@X,Local,Result):-!,apply(V,F,X,Local,Result).     % Funktionsapplikation

cseX(V,$X,_Local,[(V,[],suspend('$'(X)))]).
%if(nonvar(X),             % partielle Funktionsapplikation -> suspend
%                      Result=[(V,[],suspend('$'(X)))],
%                      Result=[]).

cseX(V,X=Y,Local,Result):- !,equalX(V,X,Y,Local,Result).          % =-Funktion (constraint). 
cseX(V,X<--Y,Local,Result):- !,matchX(V,X,Y,Local,Result).         % <---Funktion (constraint). 
 
cseX(V,X==Y,Local,Result):- !,striktequalX(V,X,Y,Local,Result).    % ==-Funktion (Bool).

cseX(V,X,_Local,[(V,[],X)]):-suspended([(_,_,X)]),!.    % suspended bleibt
                      
cseX(V,X,_Local,[(V,[],suspend(X))]):- integer(X),!.  % integer constant -> suspend

cseX(V,X,Local,Result):- X=..[Head|Args],isCons(Head,_),!,  % Konstruktor
                     findnotsuspended(V,Args,Local,Resulti,I,1),
                     if(Resulti==nothing,
                         Result=[(V,[],suspend(X))],       % wenn alle Argumente suspendiert -> suspend
                         replace(X,I,Resulti,Result)     % sonst: das ausgewertete Argument einsetzen
                     ).

% show function: display a term on the screen:
% Variables lead to a suspension, partial functions are displayed
% as "<function>", partial constructors as "<constructor>",
% io_functions as "<IO action>", lists via showList, all other
% terms via showContructors which enforces reduction to head normal form.
cseX(V,show(X),_Local,[(V,[],suspend(show(X)))]):-var(X).
cseX(V,show(I),_Local,[(V,[],showConstructor(I))]):-integer(I).
cseX(V,show($F),_Local,[(V,[],String)]):-
	F=..[Head|_Args],
	if(function(Head,_,_),
	   String="<function>",
	   String="<constructor>").
cseX(V,show(X),Local,Result):-
	X=..[Head|_Args],
%	function(Head,_,_),
	ergType(Head,Type),
	switchOnType(V,X,Type,Local,Result).

cseX(V,Term,Local,Result):-                         % sonst: eine Funktion
        tree_of_root(Term,Tree),
        csX(V,Term,Tree,Local,Result).


cseX(_V,Term,_Local,_Result):-
        write('***Runtime error: Dont know how to compute '),writeterm(Term),nl,!,fail.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Tools for cseX

% see explanation above, at cseX(show)
switchOnType(V,X,Type,_Local,[(V,[],showConstructor(X))]):-var(Type).
switchOnType(V,_X,io(_),_Local,[(V,[],"<IO action>")]).
switchOnType(V,X,list(_),_Local,[(V,[],showList(X))]).
switchOnType(V,C,_Type,_Local,[(V,[],showConstructor(C))]).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% evalInterleavedAnd(V,T1,T2,Local,Result)
%
% - works after the following semantics:
% 
%   {} /\ T = T
%   T /\ {} = T
%   T1/\T2 = [] if cseX(T1)=[] or cseX(T2)=[]
%          = suspend(T1/\T2) if cseX(T1)=suspend and cseX(T2)=suspend
%          = replace(T1/\T2,1,cseX(T1))   
%                    if cseX(T2)=suspend or |cseX(T1)| =< |cseX(T2)|
%          = replace(T1/\T2,2,cseX(T2))   
%                    if cseX(T1)=suspend or |cseX(T1)| > |cseX(T2)|


evalInterleavedAnd(V,T1,T2,_Local,[(V,[],T2)]):-
    nonvar(T1),T1=={}.

evalInterleavedAnd(V,T1,T2,_Local,[(V,[],T1)]):-
    nonvar(T2),T2=={}.


% reduce T1 and T2 to R1 and R2 and compare the results
evalInterleavedAnd(V,T1,T2,Local,Result):-
        cseX(V,T1,Local,Result1),
        cseX(V,T2,Local,Result2),
        avoidND(V,Result1,Result2,T1/\T2,Result).

% if one failes, fail at all
avoidND(_V,[],_R2,_Term,[]).
avoidND(_V,_R1,[],_Term,[]).

% if T1 suspended, test result of T2. 
avoidND(V,R1,R2,Term,Result):-suspended(R1),
        testR2(V,R2,Term,Result).

% if T1 did not suspend, but T2 did, replace T1 by R1.
% Otherwise T1 and T2 reduced and gave back one ore more solutions.
% To avoid as much ND as possible, prefer the reduction that returned
% less solutions (less ND).
avoidND(_V,R1,R2,Term,Result):-suspended(R2),replace(Term,1,R1,Result).
avoidND(_V,R1,R2,Term,Result):-
        length(R1,L1),
        length(R2,L2),
        takeShorter(L1,L2,R1,R2,Term,Result).

takeShorter(L1,L2,R1,_R2,Term,Result):-L1=<L2,replace(Term,1,R1,Result).
takeShorter(_L1,_L2,_R1,R2,Term,Result):-replace(Term,2,R2,Result).

% if T2 suspended, too, suspend at all
% otherwise replace T2 by R2.
testR2(V,R2,Term,[(V,[],suspend(Term))]):-suspended(R2).
testR2(_V,R2,Term,Result):-replace(Term,2,R2,Result).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% findnotsuspended
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln
%

% findnotsuspended: sucht ersten Term in einer Liste, der nicht suspended ist und wertet ihn aus
% falls nicht gefunden -> Term=nothing

findnotsuspended(_,[],_Local,nothing,_,_).
findnotsuspended(V,[X|Xs],Local,Result,Pos,N) :-
         cseX(V,X,Local,R),
         if(suspended(R),
             (N1 is N+1,findnotsuspended(V,Xs,Local,Result,Pos,N1)),
             (Result=R,Pos=N)
         ).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% tree_of_root(Function,Tree)
%
% - returns the definitional tree for Function

tree_of_root(Function, Tree) :- Function=..[Root|_],tree(Root,Tree).





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% apply(V,F,X,Local,Result)
%
% - tries to attach X as parameter to F, i.e. return F(X).
% - If F is a variable, suspend
% - If F is an application itself, i.e. F=G@Y, first evaluate
%   G@Y, and replace F in F@X by the result of G@Y.
% - If F is a partial application, attach X to F and compare
%   the new number of parameters with the arity of F. If there
%   are still parameters missing, mark F as partial function again.
%
% - The last case is needed for the search- and choice-stuff. The try
%   operator returns lambda-abstractions which are packed in the 
%   "searchLambda(..)"-format and only transformed in real lambda-
%   abstractions if we access them otherwise than by the try-operator.
%   Therefore a "searchLambda(..)@X"-call can appear. "searchLambda(..)"
%   cannot be marked as partial function, because then we could
%   reach "X/$searchLambda(..)" substitutions which is not allowed
%   as searchLambda is not visibile function and not a reduced term.
%   So in this case we have to evaluate "searchLambda(..)" before we
%   can apply X.
%   The last case of "apply" meets that possibility: If a Term is
%   to be applied to a function without a '$', first evaluate this
%   function.
%   The same arises with "choiceSpaces(..)@X", if the result of
%   the former choice-call is a partial application as for example in
%   choice {X=1} -> (\X->{X=1}), and therefore also with
%   "unboundVars(V,choiceSpaces(..))@X".

apply(V,F,X,_Local,[(V,[],suspend(F@X))]):-var(F),!. 

apply(V,F,X,Local,Result):-F=G@Y,!,
                   apply(V,G,Y,Local,Result1),
                   replace(F@X,1,Result1,Result).


apply(V,$F,X,_Local,[(V,[],Appl)]):-!,
        F=..[Head|Args],
        append(Args,[X],NewArgs),
        Appl1=..[Head|NewArgs],
        length(NewArgs,Length),
        arity(Head,Arity),
        if(Arity>Length,
           Appl='$'Appl1,
           Appl=Appl1).



% The following is a hack because of an error in the typechecker.
% Not all partial function calls are typed correctly and marked
% with a $. So if any other construct than unboundVars, searchLambda
% od choiceSpaces appears here, test if it has enough parameters.
% If yes, evaluate it. Otherwise mark it with $ and recall apply.
%
% without the hack, the rule is as follows:
% apply(V,F,Y,Local,Result):-
%       cseX(V,F,Local,Result1),
%       replace(F@Y,1,Result1,Result).

apply(V,F,Y,Local,Result):-
        F=..[Head|Args],
        if(member(Head,[unboundVars,searchLambda,choiceSpaces]),
           (cseX(V,F,Local,Result1),
            replace(F@Y,1,Result1,Result)),
           (length(Args,Length),
            arity(Head,Arity),
            if(Arity=Length,
               (cseX(V,F,Local,Result1),
                replace(F@Y,1,Result1,Result)),
               apply(V,'$'F,Y,Local,Result)))).




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% equalX(V,X,Y,Local,Result)
%
% - tries to unify X and Y with the following rules:
%   1) X = t, X Var, t groundterm  => {X/t} {}
%   2) X = s(t1..tn), X Var, s constructor => {X/s(X1..Xn)} X1=t1/\../\Xn=tn
%   3) X = f(t1..tn), X Var, f function => X=e if cseX(f(t1..tn))=e.
%   4) X = Y, X, Y Var => {X/Y} {}
%   if X is not a variable but Y is one, the first three cases parallel, with Y=...
%   5) f(t1..tn) = t, t not Var, => e=t, if cseX(f(t1..tn))=e
%   6) s(t1..tn) = s(e1..en) => t1=e1/\../\en=tn
%   7) s(t1..tn) = c(e1..ek), s=\= c or n=\=k => fail
%   8) t = f(t1..tn), t not Var, => t=e, if cseX(f(t1..tn))=e
%
% - these are the normal rules. But we also have to take care of the Local-flag.
%   If it is set to true, only variables that appear in V can be bound. So change the
%   rules as follows:
%    a) If X and Y are variables check the flag. 
%          If it is true and X in V => X/Y
%             If X not in V but Y in V => Y/X. 
%             If neither X nor Y in V => suspend
%         If the flag is false => X/Y
%     b) If only X is a variable, call bind(X,Y) which will perform 1)-3). In case
%        2) it will suspend it X is not in V and the flag is true.
%     c) If only Y is a variable, call bind(X,Y)...
%   So with a) we try to find one way of binding X to Y or Y to X. In b) and c) we
%   know that the other term is not a variable, so it might be a constructorterm
%   or a function. Only one direction of binding is possible then
%   5)-8) as usual


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% a) and b)

equalX(V,X,Y,Local,Result):-var(X),!,
        ifc(var(Y),
           bindVariables(V,X,Y,Local,Result),
           bind(V,X,Y,Local,Result)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%c)
equalX(V,X,Y,Local,Result):-var(Y),!,bind(V,Y,X,Local,Result).
                   

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 5)-8)
% a little bit imperativ, but who wants to pass all these parameters to help-predicates...
equalX(V,X,Y,Local,Result):-
        X=..[C1|ArgsX],!,
        ifc(isCons(C1,_),                            % if X is constructorterm,
           (Y=..[C2|ArgsY],!,                            % then check Y
            ifc(isCons(C2,_),                           % If Y is also a constructorterm
               ifc(C1==C2,                                % with the same constructor
                  equalArgs(V,ArgsX,ArgsY,Result),       % try to unify the arguments
                  Result =[]),                           % otherwise fail 
	       (targetTypeOfArgNotIO((X=Y),(=),Y),     % make sure, Y is not of Type ... -> io
	        cseX(V,Y,Local,Result1),               % otherwise evaluate function Y
                replace(X=Y,2,Result1,Result)))),      % and replace y by its result
           (targetTypeOfArgNotIO((X=Y),(=),X),         % make sure, X is not of Type ... -> io
	    cseX(V,X,Local,Result1),                   % otherwise evaluate function X
            replace(X=Y,1,Result1,Result))).           % and replace X by its result




%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% equalArgs(V,ArgsX,ArgsY,Result)
%
% - create t1=s1/\../\tn=sn for ArgsX=[t1..tn],ArgsY=[s1..sn]
% - we know that they have the same arity, because constructors with
%   the same name but a different arity are not allowed in curry

equalArgs(V,X,Y,[(V,[],Result)]):-
        equalArgsX(X,Y,Result).

equalArgsX([],[],{}).                  % n=0
equalArgsX([X],[Y],X1=Y1):-!,X=X1,Y=Y1.   % n=1. match this to avoid a ..sn=tn/\{}
equalArgsX([X|Xs],[Y|Ys],X1=Y1/\(Result)):-X=X1,Y=Y1,  % n>1
        equalArgsX(Xs,Ys,Result).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% bindVariables(V,X,Y,Local,Result)
%
% - if local-flag is false, bind X/Y
% - if it is true and X bindable, bind X/Y. If X not bindable but Y bindable,
%   bind Y/X. Remove the variable, that is bound, from V.
% - suspend otherwise

bindVariables(V,X,Y,false,[(V,[(X/Y)],{})]).

bindVariables(V,X,Y,true,[(NewV,[(X/Y)],{})]):-
        varmember(X,V),
        removeVarFromSet(X,V,NewV).

bindVariables(V,X,Y,true,[(NewV,[(Y/X)],{})]):-
        varmember(Y,V),
        removeVarFromSet(Y,V,NewV).

bindVariables(V,X,Y,true,[(V,[],suspend(X=Y))]).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% bind(V,X,Y,Local,Result)
%
% - X is a variable, Y cannot be one. Distiguish if Y is a partial
%   function, a constructor term or a function call
% - In the first two cases, suspend if X is a variable that cannot
%   be bound (Local=true, X not in V). The cases Local=true and Local=false
%   are distinguished by pattern matching for performance reasons.


%%%%%%%%%%%%%%%%%%%%
% - If Y is a partial function $f(t1..tn), result in 
%   {Y/$f(X1..Xn)} X1=t1/\../\Xn=tn if Local=false. If local=true,
%   first check if X can be bound. Suspend if not.

bind(V,X,$Y,false,NewResult):-
        Y=..[Head|Args],!,
	bindList(V,X,Head,Args,false,Result),
	transformResult(Result,NewResult).


bind(V,X,$Y,true,NewResult):-
        if(varmember(X,V),
           (Y=..[Head|Args],
            bindList(V,X,Head,Args,true,Result),
            transformResult(Result,NewResult)),
           NewResult=[(V,[],suspend(X=($Y)))]).

transformResult([],[]).
transformResult([(V,[X/Term],Result)],[(V,[X/($Term)],Result)]).


%%%%%%%%%%%%%%%%%%%%
% distinguish constructor-term and function
% If we have a function, just evaluate it and replace Y by the result.

bind(V,X,Y,Local,Result):-
        Y=..[Head|Args],!,
        ifc(isCons(Head,_),
           bindConstructor(V,X,Y,Head,Args,Local,Result),
          (targetTypeOfArgNotIO((X=Y),(=),Y),
	    cseX(V,Y,Local,Result1),
            replace(X=Y,2,Result1,Result))).


%%%%%%%%%%%%%%%%%%%%
% If local is false, just call bind list. Otherwise check if X can 
% be bound and suspend if not.

bindConstructor(V,X,_Y,Head,Args,false,Result):-
        bindList(V,X,Head,Args,false,Result).

bindConstructor(V,X,_Y,Head,Args,true,Result):-
        varmember(X,V),
        bindList(V,X,Head,Args,true,Result).
bindConstructor(V,X,Y,_Head,_Args,true,[(V,[],suspend(X=Y))]).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% bindList(V,X,Head,Args,Local,Result)
%
% - this is case 2). X=s(t1..tn) (or X=$f(t1..tn)). Result in
%   {X/s(X1..Xn)} X1=t1/\../\ Xn=tn. We know that X is bindable,
%   if we have come to this point. Local is checked only to
%   see if we have to adjust V or not.
% - first perform an occurCheck. X shall not appear in the ti!.
% - to avoid unnecessary computations, we distinguish between
%   those ti which can be reduced further, i.e. they contain
%   a function symbol, and those who cannot. The irreducible
%   ti will not be replaced by Xi. E.g. X=s(f(Y),Z,[1]) will
%   result in {X/s(A,Z,[1])} A=f(Y)
%   We need to scan the ti, but in general we will yet have
%   a computational advantage. And we keep the set of new variables
%   small. E.g. X=[Y] will not result in  {X/[A|B]} A=Y/\B=[] but
%   in {X/[Y]} {}! 

      
bindList(V,X,Head,Args,false,[(V,[X/Result],ArgsGoal)]):-  % if local=false, leave V unchanged
        occurCheck(X,Args),
        createBindList(Args,NewArgs,_VarList,ArgsGoal),
        Result=..[Head|NewArgs].

bindList(V,X,Head,Args,true,[(NewV,[X/Result],ArgsGoal)]):-
        occurCheck(X,Args),
        createBindList(Args,NewArgs,VarList,ArgsGoal),
        Result=..[Head|NewArgs],
        removeVarFromSet(X,V,NewV1),  % remove X from V, because X dissapears from the goal
        append(VarList,NewV1,NewV).   % the new variables must be appended to V.

bindList(_V,_X,_Head,_Args,_Local,[]).   % the occurCheck reported that X appears in the ti

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% createbindList(Args,NewArgs,VarList,Result)
%
% - NewArgs arises from Args in replacing all elements that can reduce further
%   by new variables. These new variables are collectet in VarList.
% - Result will be the goal in form X1=t1/\../\Xk=tk for all ti from
%   Args, that are reducible.

createBindList([],[],[],{}). 
createBindList([Arg|Args],[Arg|RestList],VarList,BindArgs):-reducedTerm(Arg),!,
        createBindList(Args,RestList,VarList,BindArgs).
createBindList([Arg|Args],[_New|RestList],[_New|VarList],Result):-
        createBindList(Args,RestList,VarList,BindArgs),
        if(BindArgs=={},            % {} signals that we are at the end,
           Result=(_New=Arg),          % so don't create a /\-conjunction
           Result=(_New=Arg/\BindArgs)).
       

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% occurCheck(V,Term)
%
% - fails, if V appears in Term.

occurCheck(X,Y):-var(Y),!,
        if(X==Y,false,true).
occurCheck(X,Y):-Y=..[_|Args],!,occurCheckList(X,Args).

occurCheckList(_,[]).
occurCheckList(X,[Y|Ys]):-
        occurCheck(X,Y),occurCheckList(X,Ys).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% matchX(V,Pattern,Term,Local,Result)
%
% - tries to match Term onto Pattern 
%
% This matching function is only called inside a committed choice guard and therefore
% all variables in the pattern are local (due to a transformation in which the matching call
% is created).
% Thus, the rules are simple:
%   1) X <-- t, X Var, => {X/t} {}   No occur check is needed because X comes from function
%                                   declaration, t is a parameter to a function call
%      NOTE: t may be a function call. The binding X/t is ok anyway, because this is the same
%            as is done in pattern matching when calling a function.
%            X will never be visible outside.
%   2) s(t1..tn) <-- x => suspend
%   3) s(t1..tn) <-- s(e1..en) => t1=e1/\../\en=tn
%   4) s(t1..tn) <-- c(e1..ek), s=\= c or n=\=k => fail
%   5) s(t1..tn) <-- f(t1..tn), => s(t1..tn)<--e, if cseX(f(t1..tn))=e

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
matchX(V,Pattern,Term,_Local,[(NewV,[Pattern/Term],{})]):-  % if Pattern is a variable and term not a 
  var(Pattern),!,                                          % function call, then bind
  removeVarFromSet(Pattern,V,NewV).

matchX(V,Pattern,Term,_Local,[(V,[],suspend(Pattern<--Term))]):-  % else pattern is a constructor term
  var(Term).                                                    % if Term is a variable, suspend
  
matchX(V,Pattern,Term,Local,Result):-% pattern is a constructor term, term constructor term or func call
  Pattern=..[PHead|PArgs],
  isCons(PHead,_),!,             % only for error detection. Must always be true
  Term=..[THead|TArgs],
  ifc(isCons(THead,_),                             % term is constructor term
      ifc(PHead==THead,
          (matchArgs(V,PArgs,TArgs,Result1),       % match the arguments
           Result=[(V,[],Result1)]),
           Result=[]),
     (cseX(V,Term,Local,Result1),                  % term is a function call
      replace(Pattern<--Term,2,Result1,Result))).


matchArgs(_V,[],[],{}).                  % for 0-ary constructors
matchArgs(_V,[X],[Y],X<--Y).
matchArgs(_V,[X|Xs],[Y|Ys],X<--Y/\(Result)):-
  matchArgs(_V,Xs,Ys,Result).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% strictequal
%
% muss noch auf Englisch kommentiert werden, Namen in Englisch umwandeln
%

%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% 2) strikte Gleichheit: Beide Seiten werden strikt ausgewertet,
%% X=..  oder ..=X suspendiert.
%% Es wird ein sequentielles and verwendet und kein nebenlaufiges, damit
%% eben bei X=.. direkt suspendiert und nicht weitergerechnet wird.

striktequalX(V,X,Y,Local,Result):-nonvar(X),nonvar(Y),!,
        X=..[C1|ArgsX],
        if(isCons(C1,_),
           (Y=..[C2|ArgsY],
            if(isCons(C2,_),
               if(C1==C2,
                  striktequalArgs(V,ArgsX,ArgsY,Result),
                  Result =[(V,[],false)]),         
               (targetTypeOfArgNotIO((X==Y),(==),Y),
	       cseX(V,Y,Local,Result1),
               replace(X==Y,2,Result1,Result)))),  
           (targetTypeOfArgNotIO((X==Y),(==),X),
	   cseX(V,X,Local,Result1),
           replace(X==Y,1,Result1,Result))).
                    
striktequalX(V,X,Y,_Local,[(V,[],suspend(X==Y))]). %X oder/und Y sind Variablen

striktequalArgs(V,X,Y,[(V,[],Result)]):-
        striktequalArgsX(X,Y,Result).

striktequalArgsX([],[],true). % Konstruktoren nullstellig --> fertig
striktequalArgsX([X],[Y],X1==Y1):-!,X=X1,Y=Y1.
striktequalArgsX([X|Xs],[Y|Ys],X1==Y1&&(Result)):-X=X1,Y=Y1,
        striktequalArgsX(Xs,Ys,Result).
%sonst AND-Ausdruck mit den GLeichungen der Argumente ausgeben





%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% csX(V,Term,DefTree,Local,Result)
%
% - cse is defined in the Curry-report. csX is the extension by the VarList
%   parameter (which is named as X in my diploma thesis).
% - evaluate Term according to its Definitional Tree. See Curry-Report
%   for explanations.
% - V containts the list of all variables declared to the
%   actual computation space. Only these variables may be bound. This
%   becomes important when we work with local spaces (in Search and Choice)
% - Local is a boolean flag and signalizes if we are in a local space.
%   Only then we use VarList because in the global space every variable
%   is declared and can be bound. Only in local spaces we need a distinction.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% external function call
% 
% - the local flag is not needed because all external functions can only
%   be evaluated if all arguments are ground terms. So no variables exist
%   in the arguments anymore and so no variables can be bound at all.

csX(V,Term,rule(_,_),_Local,[(V,[],Result)]) :-  % evaluate external functions
        functor(Term,F,_),
        isExternal(F), !,
%       Term=..[_Head|Args],
%       groundTest(Term,Args),!,         
        ifc(external_call(Term,Result),
            true,
            (write('*** Unknown error with external function '),write(F),
             write('. Please report.'),nl,!,fail)).
        
%groundTest(_Term,Args):-groundTermList(Args).
%groundTest(Term,_Args):-write('Unallowed call of external function: '),
%       numbervars(Term,0,_),write(Term),nl,
%       write('Arguments must be ground terms!'),!,fail.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% rule-Tree
%
% - apply rule to term
% - V remains unchanged because the rules are in a normalform where
%   all extra variables of Right are declared as "local List in R"
%   and so are appended to V in the next computation step (if any
%   extra variables exist at all)
% - the local flag is not needed, because we search a substitution sigma,
%   so that sigma(Left)=Term. So no variables of Term will be bound. 
%   V also remains unchanged, because all variables in Left are replaced
%   by subterms of T. So no new variables will appear but all that 
%   appear in T will now appear in Subst as terms that are substituted
%   for the variables of Left, and so they will be passed to Right and
%   have therefore to remain in V.

csX(V,Term,rule(Left,Right),_Local,[(V,[],Result)]):-
        leftUnify(Left,Term,Subst),     
        substitute(Right,Subst,Result).
csX(_,_,rule(_,_),_,[]).     % if a function is not defined totally and 
                        % and called with a undefined parameter



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
% 4.8.97 and-trees disabled. /\ is now trated by cseX.
%
% and-Tree
%
% - try to avoid nondeterminism if possible
% - first try left branch. If it suspends, return result of right branch
% - if length(R1)=<1, the left branch either failed or executed a 
%   deterministic step, so return R1.
% - Otherwise evaluate right branch, too, and compare R1 and R2. At this
%   point we already know that R1 did not suspend, fail or reduced deterministic
%   So the strategy is now as follows:
%   * If the right branch suspended, return R1. 
%   * If lengt(R2)=<1, the right branch either failed or reduced deterministic
%     and so we return R2 to prefer deterministic steps.


% original and-treatment
%csX(V,Term,and(Tree1,Tree2),Local,Result):-
%        csX(V,Term,Tree1,Local,R1),!,
%        if(suspended(R1),
%           csX(V,Term,Tree2,Local,Result),
%           Result=R1).


% avoid-ND-treatment
%csX(V,Term,and(Tree1,Tree2),Local,Result):-
%        csX(V,Term,Tree1,Local,R1),!,
%        evaluateLeftResult(V,Term,R1,Tree2,Local,Result).
%
%
%evaluateLeftResult(V,Term,R1,Tree2,Local,Result):-
%       suspended(R1),!,
%       csX(V,Term,Tree2,Local,Result).
%       
%evaluateLeftResult(V,Term,R1,Tree2,Local,Result):-
%       length(R1,L1),
%       ifc(L1=<1,
%            Result=R1,
%            (csX(V,Term,Tree2,Local,R2),!,
%            compareLeftAndRight(R1,R2,Result))).
%
%compareLeftAndRight(R1,R2,R1):-suspended(R2).
%compareLeftAndRight(R1,R2,Result):-
%       length(R2,L2),
%       ifc(L2=<1,
%           Result=R2,
%           Result=R1).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% or-Tree
%
% - try left branch first, if it suspend, suspend at all.
%   If not, try second one. If it suspends, suspend at all.
% - V is given to both branches because their computations
%   are independent, they are two computations of the same goal.


csX(V,Term,or(Tree1,Tree2),Local,Result):-
      csX(V,Term,Tree1,Local,R1),!,
      if(suspended(R1),
         Result=[(V,[],suspend(Term))],
         (csX(V,Term,Tree2,Local,R2),
         if(suspended(R2),
            Result=[(V,[],suspend(Term))],
            append(R1,R2,Result)))).



%csX(V,Term,or(Tree1,Tree2),Local,Result):-
%      csX(V,Term,Tree1,Local,Result1),!,
%      csXOrResult(V,Term,Tree2,Local,Result1,Result).
%
%csXOrResult(V,Term,_Tree2,_Local,Result1,[(V,[],suspend(Term))]):-
%       suspended(Result1).
%
%csXOrResult(V,Term,Tree2,Local,Result1,Result):-
%        csX(V,Term,Tree2,Local,Result2),
%        csXOrResult2(V,Term,Local,Result1,Result2,Result).
%
%csXOrResult2(V,Term,_Local,_Result1,Result2,[(V,[],suspend(Term))]):-
%       suspended(Result2).
%
%csXOrResult2(_V,_Term,false,Result1,Result2,Result):-
%       append(Result1,Result2,Result).
%
%csXOrResult2(_V,_Term,true,[],Result2,Result2).
%
%csXOrResult2(_V,_Term,true,Result1,[],Result1).
%
%csXOrResult2(V,Term,true,_Result1,_Result2,[(V,[],suspend(Term))]).

        
          


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% branch-Tree
%
% - branch(pattern,p,r,trees)
% - distinguish following cases:
%   * Term|p is a constructor-term and the pattern of a tree can
%     be unified with the term. Apply this tree then.
%   * Term|p is a constructor-term but no tree can be applied.
%     Return empty list.
%   * Term|p is a variable and r=rigid. Suspend then.
%   * Term|p is a variable, r=flex, local=false. Then the variable can be unified
%     with the pattern of each tree. Do this and apply this substitution
%     to the term and give it back.
%   * Term|p is a variable, r=flex, local=true. Test then if the variable is in V.
%     If no, suspend because the variable cannot be bound.
%     If yes, work as if local=false.
%   * Term|p is a function call. Compute the function and replace the
%     call with its result (done by the replace function).
%   * Term|p can be a "List localVars T". "localVars" will be reconized as
%     function call and computed via cseX.


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% distinguish if Term|p is a variable or not 

csX(V,Term,branch(_,P,Mode,Trees),Local,Result):-
       teilterm(Term,P,Teilterm),
       if(nonvar(Teilterm),
          constr_or_func(V,Term,Teilterm,P,Trees,Local,Result),
          variable(V,Term,Teilterm,P,Mode,Trees,Local,Result)),!.
csX(_,_,branch(_,_,_,_),_,[]).   % if a function is not defined totally and 
                              % and called with a undefined parameter


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Term|p is a variable
% variable(VarList,Term,Variable,p,r,Trees,Result).
% 
% - r=rigid, then result
% - r=flex and local-flag=true. Check if X in in V, suspend if it is not.
% - If X is in V and local=true, suspend if there is more than one tree.
%   Otherwise X is in V and only one tree exists. Than we can apply the
%   tree. Because in this case all variable(...,true,...) have failed,
%   we apply the last clause which is the general clause for local=false,
%   which is applicable now because we have checked that all conditions
%   for applying with local=true hold. 
% - The general case:
%   Get the pattern from each tree, and bind the variable to it. 
%   If the local-flag is true, we remove X from V, collect the variables
%   from the pattern and append them to V because the variables
%   can now appear in the term after applying the subtitution X/Pattern.
%   Then apply the substitution.
% - recall variable with old V because each tree forms a independent result


variable(V,Term,_,_,rigid,_,_,[(V,[],suspend(Term))]).
variable(_,_,_,_,flex,[],_,[]).   % no trees left
variable(V,Term,X,_P,flex,_Trees,true,[(V,[],suspend(Term))]):-
  \+ strictmember(X,V).
%variable(V,Term,_X,_P,flex,[_Tree1_,_Tree2|_MoreTrees],true,[(V,[],suspend(Term))]).
variable(V,Term,X,P,flex,[Ti|Resttrees],Local,Result):-
        pat(Ti,PatTi),
        teilterm(PatTi,P,PatTi0),
        Sigma_i=[(X/PatTi0)],
        changeVifLocal(V,X,PatTi0,Local,NewV),
        substitute(Term,Sigma_i,NewTerm),
        variable(V,Term,X,P,flex,Resttrees,Local,ResultRest),
        Result=[(NewV,Sigma_i,NewTerm)|ResultRest].


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% changeVifLocal(V,X,PatTi0,Local,NewV)
%
% - If the local-flag is false, do not change V.
%   If it is true, remode V from X and append the free 
%   variables that appear in PatTi0

changeVifLocal(V,_X,_PatTi0,false,V).
changeVifLocal(V,X,PatTi0,true,NewV):-
        removeVarFromSet(X,V,NewV1),
        freeVars(PatTi0,VarsPatTi0),
        append(VarsPatTi0,NewV1,NewV).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Term|p is a constructor-term or a function
%
% - if it is a constructor, call constr to check each tree to see, if
%   one is unifiable with the constructor-term
% - if it is a function, compute the function and replace its appearance
%   with the computation result
% - the local-flag ist just passed to constr or to cseX.

constr_or_func(V,Term,Teilterm,P,Trees,Local,Result):-
        functor(Teilterm,Head,Argnumber),
        if(isCons(Head,_),
           constr(V,Term,Head,Argnumber,P,Trees,Local,Result),
           (cseX(V,Teilterm,Local,FuncResult),replace(Term,P,FuncResult,Result))).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% constr(V,Term,Constructor,ArgumentNumber,p,Trees,Local,Result)
%
% - constr checks if one tree-pattern is unifiable with the constructor-term
%   This is possible, if the constructor-term and the tree-pattern have the
%   same constructor with the same arity on the top.
%   Compute the term with the according tree then.
% - if the tree-pattern is a variable, we have an external function. The tree
%   is applicable then.
% - the local-flag is just passed to csX and not of interest elsewhere, because 
%   no variable can be bound.

constr(_,_,_,_,_,[],_,[]).  % no trees left
constr(V,Term,Constructor,ArgNumber,P,[Ti|Resttrees],Local,Result):-
       pat(Ti,PatTi),
       teilterm(PatTi,P,PatTi0),
       applicable(V,Term,Constructor,ArgNumber,P,PatTi0,[Ti|Resttrees],Local,Result).


applicable(V,Term,Constructor,ArgNumber,_P,PatTi0,[Ti|_Resttrees],Local,Result):-
        nonvar(PatTi0),
        functor(PatTi0,PatHead,PatArgNumber),
        Constructor==PatHead,
        ArgNumber==PatArgNumber,!,
        csX(V,Term,Ti,Local,Result).
applicable(V,Term,_Constructor,_ArgNumber,_P,PatTi0,[Ti|_Resttrees],Local,Result):-
        var(PatTi0),!,
        csX(V,Term,Ti,Local,Result).
applicable(V,Term,Constructor,ArgNumber,P,_PatTi0,[_Ti|Resttrees],Local,Result):-
           constr(V,Term,Constructor,ArgNumber,P,Resttrees,Local,Result).

%       ifc(onlyleftUnify(Teilterm,PatTi0,Subst),
%          (csX(Term,Ti,Result1),compose2(Term,Result1,Subst,Result)),
%          constr(Term,Teilterm,P,Resttrees,Result)).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Tools for csX


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% pat(DefTree,Patterm)
% 
% - returns the pattern from a definitional tree
%

pat(rule(L,_),L).

pat(branch(Pi,_,_,_),Pi1):-Pi=Pi1.

pat(or(T1,_),T):-pat(T1,T).

%pat(and(T1,_),T):-pat(T1,T).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Tools used by more than one function and other modules


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%replace (Term,p,List of results,Result)
%
% - replaces Term|p with the result of the computation of Term|p. 
%   This computation can result in a list, then return a new term
%   for each element of the list. If the computation suspended,
%   suspend the Term 
% - used by csX, equal, strictequal

replace(Term,_P,ResultList,[(V,S,suspend(Term))]):-
        suspended(ResultList),!,ResultList=[(V,S,_)].
replace(_,_,[],[]).  % no more results
replace(Term,P,[(V,Subst,Goal)|Xs],[(V,Subst,NewGoal)|Es]):-
        substitute(Term,Subst,NewTerm),
        substTerm(NewTerm,P,Goal,NewGoal),
        replace(Term,P,Xs,Es).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% suspended(Result)
% 
% - checks if the Result is suspended
% - used by several functions

suspended([(_,_,X)]):-nonvar(X),X=suspend(_).



