%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHANGES since July 98:
%
%%%%%
% 8.12.99 - Frank Steiner
% - for the transformation of rules with several guards:
%   - rule for createNF added
%   - transformLocal predicate added
%
%%%%%
% 8.1.98 - Frank Steiner
%
% for enabling the new choice annotation:
% cTrans(Subst,choice...) removed, transChoiceArgs removed

l:-[local].


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% cTrans(Term,Rule,(VarList,LocalList,NewTerm))
%
% - just adds the Subst parameter for the following cTrans


cTrans(Term,Rule,(VarList,LocalList,NewTerm)):-cTrans([],Term,Rule,(VarList,LocalList,NewTerm)).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% cTrans(Subst,Term,Rule,(VarList,LocalList,NewTerm))
%
% - collects free variables of Term in VarList
% - replaces locally declared variables with fresh ones
% - Subst contains the locally declared variables and the
%   fresh variables that replace the declared
%   Note: an automatic replacement by a X=X_fresh -Statement
%   is not possible, because X might be used more than once
%   as local variable,e.g. "local X in (f(X)/\local X in t/\ g(X))"
%   and these two X must be distinguished.
% - Rule containts the rule for that we actually transform the Term.
%   That is because we want to display the whole rule when we report
%   a warning if local variables are declared more than once in a list.
% - LocalList gives back the names of all fresh variables that are
%   now in NewTerm.
% - VarList gives back all free variables that were found in Term.


%%%%%%%%%%%%%%%%%%%%%%%
% Variable
%
% if Term is a variable, it can be a local one. Then the variable
% will be element of the substitution. Replace it with the
% new variable that is found in the substitution.
% if the variable is not in substitution, then it has not been
% declared local. So collect it as free variable.

cTrans(Subst,Term,_Rule,Result):- 
	var(Term),!,
  	if(substMember(Term,Subst,SubstValue),
	   Result=([],[],SubstValue),
	   Result=([Term],[],Term)).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% {}-brackets
%
% If one has specified a constraint {{X=1}} it will be translated into
% {X=1} during the readin. So we still have to remove the inner {}-bracktes
% because they would be recognized as function otherwise.

cTrans(Subst,{Term},Rule,Result):- 
	cTrans(Subst,Term,Rule,Result).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% local List in Term
%
% - first check if the list contains variables declared more than
%   once. Replace it by a single appearance and report an warning.
%   It might be that the user wanted to specify two different names.
% - then create the substitution X/X_fresh for each X 
% - also update Subst:
%   Given the case "local X in {local X in t} /\ g", all X in t must be
%   replaced by the second declared X, all in g by the first one.
%   To avoid forming a set for Subst, we just work as follows:
%   - Scanning the first local-declaration leads to Subst1=[(X/X_1)] an
%     cTrans(Subst1,{local X in t} /\ g).
%   - Scanning the second local-declaration leads to Subst2=[(X/X_2),(X/X_1)]
%     and cTrans(Subst2,t) /\ cTrans(Subst1,g). 
%   - So Subst contains both appearances of X. But the recent declared X
%   comes first, so in t all appearances of X will be replaced by X_2.
%   But as g is called with Subst1, there all X will be replaced by X_1.
%   So Subst works as described in the op.sem. although it is not a set.
% - then call cTrans for Term and pass the newly created substitution to it.
% - cTrans might create further local variables. Give them back, too.

cTrans(Subst,List localVars Term,Rule,(Vars,Locals,NewTerm)):-!,
	checkForSet(List,List localVars Term,Rule,Set),
	appendSubstForLocalVars(Set,Subst,Locals1,NewSubst),
	cTrans(NewSubst,Term,Rule,(Vars,Locals2,NewTerm)),
	append(Locals1,Locals2,Locals).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Constraint /\ Constraint
%
% - we can have conjunctions of constraints by /\. This
%   is a special case, becaus in both parameters of this
%   function, local can appear.In arguments of any other
%   function no local can appear,because constraints cannot be
%   uses as parameters of functions.
% - Both parts work with the same Subst, because local-Declaration in
%   one branch of /\ are not visible in the other branch. But in both
%   branches the variables locally declared before are visible (as long
%   as they are not declared local again), so pass Subst to both
%   branches. See example in "local.. in.." above.
% - The two lists of local variables can easily be appended because 
%   no doubles can appear (each local creates its own new variables).
%   In contrast, C1 and C2 can containt the same free variables, so
%   the varlists must be united to a SET.
% - Notice, that  "local X in T1 /\ T2" is interpreted as 
%   "local X in (T1 /\ T2)". Local variables that shell only be used
%   in the branches must be specified as "{{local X in T1} /\ {local X in T2)}}"


cTrans(Subst,C1/\C2,Rule,(Vars,Locals,NewC1/\NewC2)):-
	cTrans(Subst,C1,Rule,(Vars1,Locals1,NewC1)),
	cTrans(Subst,C2,Rule,(Vars2,Locals2,NewC2)),
	appendSet(Vars1,Vars2,Vars),
	append(Locals1,Locals2,Locals).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% - constraints are of the form "t1=t2". The ti shell not be constraints
%   again, so there are no local variables given back.
% - Free variables must be united as set again (see above)

cTrans(Subst,T1=T2,Rule,(Vars,[],NewT1=NewT2)):-
	cTrans(Subst,T1,Rule,(Vars1,Locals1,NewT1)),!,
        checkNoLocals(Rule,T1=T2,Locals1),
	cTrans(Subst,T2,Rule,(Vars2,Locals2,NewT2)),!,
        checkNoLocals(Rule,T1=T2,Locals2),
        appendSet(Vars1,Vars2,Vars).

checkNoLocals(_Rule,_Term,[]):-!.
checkNoLocals(Rule,Term,_List):-error(Rule,"Left and right hand side of '=' in '~w' mustn be~nconstraints and contain local-declarations!",[Term]).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% choice([(c1,e1),..,(cn,en)])
%
% - must be treated explicitely
% - a local-operator in front on a ci always ranges over ei, even if
%   the variables bound in ci appear already outside of
%   choice. 
% - If there ist a local-operators in front of a ci, append their
%   substitution with new variables to Subst. Then transform ci and ei.
% - declare the local variables of ci (together with those that were
%   declared in front of ci) in ci, the local variables of ei in ei.
%
%cTrans(Subst,choice(Args),Rule,(Vars,[],choice(NewArgs))):-
%	transChoiceArgs(Subst,Args,Rule,Vars,NewArgs).
%
%transChoiceArgs(_Subst,[],_Rule,[],[]).
%
%transChoiceArgs(Subst,[(C,E)|Args],Rule,Vars,[(CNew,ENew)|NewArgs]):-
%	cTrans(Subst,C,Rule,(FreeVarsC,LocalsC,C1)),
%	cTrans(Subst,E,Rule,(FreeVarsE,LocalsE,E1)),
%	declareVarsInTerm(LocalsC,C1,CNew),
%	declareVarsInTerm(LocalsE,E1,ENew),
%	appendSet(FreeVarsC,FreeVarsE,FreeVars),
%	transChoiceArgs(Subst,Args,Rule,VarsArgs,NewArgs),
%	appendSet(FreeVars,VarsArgs,Vars).
%
%
%transChoiceArgs(Subst,[List localVars (C,E)|Args],Rule,Vars,[(CNew,ENew)|NewArgs]):-
%	checkForSet(List,List localVars (C,E),Rule,Set),
%	appendSubstForLocalVars(Set,Subst,OuterLocals,NewSubst),
%	cTrans(NewSubst,C,Rule,(FreeVarsC,LocalsC,C1)),
%	cTrans(NewSubst,E,Rule,(FreeVarsE,LocalsE,E1)),
%	append(LocalsC,OuterLocals,AllLocals),
%	declareVarsInTerm(AllLocals,C1,CNew),
%	declareVarsInTerm(LocalsE,E1,ENew),
%	appendSet(FreeVarsC,FreeVarsE,FreeVars),
%	transChoiceArgs(Subst,Args,Rule,VarsArgs,NewArgs),
%	appendSet(FreeVars,VarsArgs,Vars).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% f(t1..tn)
%
% - last case: Term is any function.Then apply  cTrans to
%   each argument. Arguments can be constraints again, so there
%   might be local variables declared. These variables are not of
%   interest outside the argument, so declare declare them locally.
%   Free variables in the arguments are collected and given back.
% - as local-declarations in parameters can result in the same variable
%   names appear more than once in a goal while computing it (see
%   chapter about renaming variables and applying substitutions), these
%   declarations are marked with the keyword "localInPar" instead of
%   "localVars". "localInPar" will rename the variables while computing.


cTrans(Subst,Term,Rule,(Vars,[],NewTerm)):-
	Term=..[Head|Args],
	cTransArgs(Subst,Args,Rule,(Vars,NewArgs)),
	NewTerm=..[Head|NewArgs].



cTransArgs(_Subst,[],_Rule,([],[])).
cTransArgs(Subst,[Term|Rest],Rule,(Vars,[NewTerm|NewRest])):-
	cTrans(Subst,Term,Rule,(Vars1,Locals,Term1)),
	declareParVarsInTerm(Locals,Term1,NewTerm),
	cTransArgs(Subst,Rest,Rule,(RestVars,NewRest)),
	appendSet(Vars1,RestVars,Vars).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% createNF(Rule,NewR)
%
% - collect all local-declarations to the top of a constraint
% - rename all locally declared variables
% - collect all free variables and declare then in the condition
%   of the rule
% - Rule is of the form L=R (R can be cond(C,R) for a conditional rule)
% - NewR gives back the transformed right side of the rule.



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% L = X, X variable
%
% - if X appears in L, leave the rule unchanged. Otherwise change to
%   L if {local X in {}}   = X.
% - this case must be matched explicitely here to enable the
%   pattern matching in the next cases.

createNF(L=R,NewR):-var(R),!,
	freeVars(L,FreeLVars),
	ifc(strictmember(R,FreeLVars),
	    NewR=R,
            NewR=cond(R localVars {},R)).

%        \+ \+ (numbervars((L,R),0,_),
%	write(L),write(' = '),write(R),write('  ->  '),nl,
%	write(L),write(' = '),write(NewR),nl,nl).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% L if C = R
%
% - If right side is a conditional rule, transform C and
%   and R into their NF:
% - apply cTrans to C and to R. That gives back the local und free
%   Vars of C and R.
% - Ccollect the free variables from C and from R and remove 
%   the variables that appear in the  head of L. Add the locally 
%   declared variables from C and  declare all extravariables 
%   local to C.
% - If R is a constraint, it might have locally declared variables.
%   These are not of interest outside of R ans so are declared
%   local to R.
% - if there ist a local declaration in front of C, it ranges over
%   C and R. Create a substitution for the local variables and
%   pass it to cTrans. So the variables are renamed the same
%   way in C and R


createNF(L=cond(C,R),cond(NewCond,NewRight)):-
	cTrans(C,L=cond(C,R),(CFree,CLocals,NewC)),
	cTrans(R,L=cond(C,R),(RFree,RLocals,NewR)),
	appendSet(CFree,RFree,CRFree),
 	freeVars(L,FreeLVars),
	removeVars(FreeLVars,CRFree,AllFree),
	appendSet(AllFree,CLocals,DeclareVars),
	declareVarsInTerm(DeclareVars,NewC,NewCond),
	declareVarsInTerm(RLocals,NewR,NewRight).

%       \+ \+ (numbervars((L,C,R,NewCond,NewRight),0,_),
%       write(L),write(' if {'),
%       write(C),write('} = '),write(R),write(' -> '),nl,
%       write(L),write(' if {'),write(NewCond),write('} = '),
%       write(NewRight),nl,nl).

createNF(L=(List localVars cond(C,R)),cond(NewCond,NewRight)):-
	checkForSet(List,List localVars cond(C,R),L=List localVars cond(C,R),Set),
	appendSubstForLocalVars(Set,[],OuterLocals,StartSubst),	
	cTrans(StartSubst,C,L=cond(C,R),(CFree,CLocals,NewC)),
	cTrans(StartSubst,R,L=cond(C,R),(RFree,RLocals,NewR)),
	appendSet(CFree,RFree,CRFree),
 	freeVars(L,FreeLVars),
	removeVars(FreeLVars,CRFree,AllFree),
	appendSet(AllFree,CLocals,DeclareVars1),
	appendSet(DeclareVars1,OuterLocals,DeclareVars),
	declareVarsInTerm(DeclareVars,NewC,NewCond),
	declareVarsInTerm(RLocals,NewR,NewRight).


%       \+ \+ (numbervars((L,C,R,NewCond,NewRight),0,_),
%       write(L),write(' if '),write(List),write(' localIn {'),
%       write(C),write('} = '),write(R),write(' -> '),nl,
%       write(L),write(' if {'),write(NewCond),write('} = '),
%       write(NewRight),nl,nl).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% L=R
%
% - collect the free and the local variables in R. 
% - declare local variables of R in R.
% - remove the variables that appear in L from the set of free variables
%   of R. Declare the free variables as local in a
%   "dummy" condition {}:
%   "L if {[..] localVars {}} = R"
%   This is the only formally correct method to
%   declare the extra variables of R as local variables if
%   R is not a constraint.
%
% New: special case is L=condList(list of guards, list of rhs).
% We have a list (condList) of cond(C,R) expressions which we
% create the NF for in the usual way. But then we split the
% guards and the rhs into two lists and return condList(GuardList, CondList).
% This format is expected by the typechecker and easier to handle for
% the future operations. But it arrives in this format here, because
% that makes it easy to pass the cond()-parts to createNF.
  
createNF(_L=condList([]),condList([],[])).
createNF(L=condList([Cond|RestList]),condList([NewC|NewCList],[NewR|NewRList])):-
  createNF(L=Cond,cond(NewC,NewR)),
  createNF(L=condList(RestList),condList(NewCList,NewRList)).


createNF(L=R,NewRight):-
	freeVars(L,FreeLVars),
	cTrans(R,L=R,(RVars,RLocals,NewR1)),
    declareVarsInTerm(RLocals,NewR1,NewR),
  	removeVars(FreeLVars,RVars,FreeRightVars),
           if(FreeRightVars==[],
			  NewRight=NewR,
              (declareVarsInTerm(FreeRightVars,{},Cond),
               NewRight=cond(Cond,NewR))).

%      \+ \+ (numbervars((L,R,NewRight),0,_),
%	write(L),write('  = '),write(R),write('  ->  '),nl,
%	write(L),write('  = '),write(NewRight),nl,nl).






%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% transformLocal(R,NewR)
%
% - transforms all "[X1..Xn] localVars T " into
%   "X1 localVars X2 localVars ... Xn localVars T"
% - applied before typechecking (when called from maketree in retransformGuards)
% - This format is neccessary for the Typechecker, because
%   localVars is declared as function that can have only ONE
%   variable as parameter. This must be because otherwise [A,B] localVars c
%   would fail to typecheck if A and B are variables with different types.
  
transformLocal(R,R):-var(R).
transformLocal(X localVars T,X localVars T):-var(X).
  
transformLocal([] localVars T,T).
  
transformLocal([X|Xs] localVars T,X localVars NewT):-
  transformLocal(Xs localVars T,NewT).

transformLocal([X|Xs] localInPar T,X localVars NewT):-
  transformLocal(Xs localInPar T,NewT).
  
transformLocal(Term,NewTerm):-
	Term=..[Head|Args],
	transformLocalArgs(Args,NewArgs),
	NewTerm=..[Head|NewArgs].

transformLocalArgs([],[]).
transformLocalArgs([Arg|Args],[NewArg|NewArgs]):-
	transformLocal(Arg,NewArg),
	transformLocalArgs(Args,NewArgs).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% retransformLocal(R,NewR)
%
% - transforms all "X1 localVars X2 localVars ... Xn localVars T"
%   into "[X1..Xn] localVars T".
% - applied after the typechecking, to retransform the
%   local-Declarations into the faster computable form with
%   all variables in a single list. The typechecker cannot
%   process this format, but cseX does.


retransformLocal(R,R):-var(R).
retransformLocal(X localVars T,[X|Vars] localVars NewT):-
	collectLocals(T,Vars,NewT1),
	retransformLocal(NewT1,NewT).

retransformLocal(X localInPar T,[X|Vars] localInPar NewT):-
	collectLocals(T,Vars,NewT1),
	retransformLocal(NewT1,NewT).

retransformLocal(Term,NewTerm):-
	Term=..[Head|Args],
	retransformLocalArgs(Args,NewArgs),
	NewTerm=..[Head|NewArgs].

retransformLocalArgs([],[]).
retransformLocalArgs([Arg|Args],[NewArg|NewArgs]):-
	retransformLocal(Arg,NewArg),
	retransformLocalArgs(Args,NewArgs).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% collectLocals(Term,VarList,NewTerm)
%
% - removes all "Xi localIn{Par}.." declarations from the front of 
%   Term and collects the Xi in VarList.
% - stops as soon as it finds anything else than a "Xi localVars.."

collectLocals(T,[],T):-var(T).
collectLocals(X localVars T,[X|Vars],NewT):-
	collectLocals(T,Vars,NewT).
collectLocals(X localInPar T,[X|Vars],NewT):-
	collectLocals(T,Vars,NewT).
collectLocals(T,[],T).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% tools
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%%%%%%%%%%%%%%
% declareVarsInTerm (Varlist, Term, NewTerm)
% 
% - if Varlist=[X1..Xn], NewTerm is set to 
%   "X1 localVars (X2 localVars ...Xn localVars Term)
% - This format is neccessary for the Typechecker, because
%   localVars is declared as function that can have only ONE
%   variable as parameter.


declareVarsInTerm([],Term,Term):-!.
declareVarsInTerm([X|Xs],Term,(X localVars NewTerm)):-
	declareVarsInTerm(Xs,Term,NewTerm).

%%%%%%%%%%%%%%%%%%%%%%%%%%%
% declareParVarsInTerm (Varlist, Term, NewTerm)
% 
% - if Varlist=[X1..Xn], NewTerm is set to 
%   "X1 localInPar (X2 localInPar ...Xn localInPar Term)
% - This format is neccessary for the Typechecker, because
%   localInPar is declared as function that can have only ONE
%   variable as parameter.


declareParVarsInTerm([],Term,Term):-!.
declareParVarsInTerm([X|Xs],Term,(X localInPar NewTerm)):-
	declareParVarsInTerm(Xs,Term,NewTerm).


%%%%%%%%%%%%%%%%%%%%%%%%%%%
% declareVarsListInTerm (Varlist, Term, NewTerm)
% 
% - if Varlist is not empty, NewTerm is set to "Varlist localVars Term"

declareVarsListInTerm([],Term,Term):-!.
declareVarsListInTerm(List,Term,(List localVars Term)).


%%%%%%%%%%%%%%%%%%%%%
%substMember(Variable, Substitution, Value)
%
% checks if the variable is part of the substitution. If yes,
% returns the value to be substituted for the variable

substMember(X,[(X1/Y1)|_],Y):-X==X1,!,Y=Y1.
substMember(X,[_|Xs],Y):-substMember(X,Xs,Y).


%%%%%%%%%%%%%%%%%%%%
% checkForSet(List,Term,Rule,Set)
% 
% - checks if the list is a set. If it is not, a set is created
%   and a warning is reportet that a variable appears more then
%   once.
% - Term and Rule are needed for the error report.

checkForSet([],_,_,[]):-!.
checkForSet([X|Xs],Term,Rule,Set):-
	if(strictmember(X,Xs),
	   (warning(Rule,"Local variable ~w declared more than once in ~w~n",[X,Term]),
            Set=Set1),
           Set=[X|Set1]),
	checkForSet(Xs,Term,Rule,Set1).
 

%%%%%%%%%%%%%%%%%%%%
% appendSubstForLocalVars(VarList,Subst,Locals,NewSubst)
%
% - for every variable X of the list append a (X/X_Fresh)
%   to the beginning substitution, with X_Fresh a fresh variable
% - collect the fresh variables in Locals

appendSubstForLocalVars([],Subst,[],Subst).
appendSubstForLocalVars([X|Xs],Subst,[X_fresh|Locals1],[(X/X_fresh)|Subst1]):-
	appendSubstForLocalVars(Xs,Subst,Locals1,Subst1).


%%%%%%%%%%%%%%%%%%%
% appendSet(VarList1,VarList2,NewVarList)
%
% adds the elements of the first list, that are not in the
% second list, to the beginning of the second
% Both lists must be a set already!!!

appendSet([],X,X).
appendSet(X,[],X).
appendSet([X|Xs],Ys,NewList):-
	strictmember(X,Ys),!,
	appendSet(Xs,Ys,NewList).
appendSet([X|Xs],Ys,[X|NewList]):-
	appendSet(Xs,Ys,NewList).


%%%%%%%%%%%%%%%%%%%
% removeVarsFromSubst(VarList,Subst,NewSubst)
%
% - removes the substitutions for all variables in VarList

removeVarsFromSubst([],Subst,Subst).
removeVarsFromSubst([X|Xs],Subst,NewSubst):-
	removeVarFromSubst(X,Subst,NewSubst1),
	removeVarsFromSubst(Xs,NewSubst1,NewSubst).

removeVarFromSubst(_,[],[]).
removeVarFromSubst(X,[(Y/_)|Xs],Xs):-X==Y,!.
removeVarFromSubst(X,[(Y/YSubst)|Xs],[(Y/YSubst)|NewXs]):-
	removeVarFromSubst(X,Xs,NewXs).


%%%%%%%%%%%%%%%%%%%%%%%%%%
% removeDomSubstFromVarSet(Subst,VarSet,NewVarSet)
%
% - removes all variables of dom(Subst) from the set of variables, VarSet.

removeDomSubstFromVarSet([],V,V).
removeDomSubstFromVarSet([(X/_T)|Subst],V,NewV):-
	removeVar(X,V,NewV1),
	removeDomSubstFromVarSet(Subst,NewV1,NewV).



%%%%%%%%%%%%%%%%%%%%%%%%%%
% warning reports a warning

warning(Rule,Format,Args) :-
	append("~nWarning: ~w~n",Format,NewFormat),
	\+ 
        \+ 
        (numbervars((Rule,Args),0,_),
	       format(NewFormat,[Rule|Args])),nl.

%%%%%%%%%%%%%%%%%%%%%%%%%%
% error reports an error

error(Rule,Format,Args) :-
	append("~n*** Type-Error: ~w~n",Format,NewFormat),
	\+ 
        \+ 
        (numbervars((Rule,Args),0,_),
	       format(NewFormat,[Rule|Args])),nl,!,fail.


