%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% CHANGES since July 98:
%
%%%%%
% 8.12.98 - Frank Steiner
% - for the transformation of rules with several guards:
%   - getFuncRule changed to do the check for "l = condList(..)"
%   - transformGuards added
%   - decideOnGuardType added
%   - creatNewGuardRules added
%   - cascade addded
%   - makeAllTrees changed into a fail-redo-loop, because the "function"
%     predicate must stay in memory for all functions all the time due
%     to the typecheck that might be performed.

%%%%%
% 8.1.99 - Frank Steiner
%
% General: 
%    addPosition(N,[X|Xs],[N'#'X|Xs1]):-addPosition(N,Xs,Xs1). (the hash was without ' ' before)
%
%  * makeATree, errorInTree added,
%    createTheTree changed
%
% for enabling the new choice annotation:
%  * makePatternList  added
%  * makeEvalTree(Func,choice):-!,... rule added.
%  * createChoiceArgs,
%    createMatchingConstraints,
%    createChoiceRule,
%    combineCandMC predicates added
  

%%%%%
% 2.8.99 - Frank Steiner
%						   
% additionally create a "choice(normal tree)" for "f eval choice" that is used
% for writing out in the def file. curry2java uses this for the new choice
%
% affected: makeEvalTree(choice...),
% createChoiceArgs
						   
  
listTrees:-
  tree(X,Y),write(X),write(': '),write(Y),nl,
  retract(tree(X,Y)),
  listTrees,
  assertz(tree(X,Y)).
listTrees.


mr:-[maketree].

%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%% Tools fuer Regelsuche


%versucht, eine Regel zu einer Funktion zu suchen und zu retracten.
%beim Retract werden die Var gebunden, so dass Pat dann die Regel
%enthaelt.
% transformGuards nimmt nun die Rcktransformation von Regeln mit
% mehreren Guards zurck, die beim Einlesen in f = condList(...)
% umgewandelt wurden.
getFuncRule(Func,rule(Pat,R)):-
   fArity(Func,N),
   makePattern(Func,N,Pat),
   retract(rule(Pat,R1)),
   transformGuards(Pat,R1,R).
   
% transformGuards(L,R,NewR).
% prft, ob die rechte Regelseite die Form condList(Guards,Rhs))
% hat. Wenn dies der Fall ist, typechecke den ersten Guard.
% a) dabei kommt Typ constraint heraus: Erstelle genauso viele
%    Regeln fr L wie es Guards/Rhs gibt. Also fr werden zu
%    condList([x=1,x=2],[True,False]) zwei Regeln
%    L |x=1 = True
%    L |x=2 = False
%    erstellt. Bzw, in real, als NewR wird "cond(x=1,True)"
%    zurckgeliefert, und es wird eine neue Regel "L=cond(x=2,False)"
%    asserted. Die erste Regel wird also einfach als Austausch
%    fr die L=condList(...) Regel weiterverarbeitet.
%
% b) Es kommt Typ bool heraus. In dem Fall erstelle eine if-then-else
%    cascade, wobei als letzter else-Fall "else failed" genommen wird,
%    failed ist eine Funktion die immer fehlschlgt (vordefiniert).
   
transformGuards(Pat,condList([C|CList],[R|RList]),NewR):-
  nonvar(CList), %if R1 was a variable, fail here
  %nl,nl,write('***Found a rule with several guards: '),write(Pat),
  %write(' = '),write(condList([C|CList],[R|RList])),nl,
  transformLocal([C,R],[C1,R1]),
  fixvars([Pat,C1,R1],[FixedPat,FixedC,FixedR],_FixList),!,
  typeTerm([],(FixedPat,(FixedC,FixedR)),pair(_PatType,pair(CType,_RType)),_NewTerm,_),
  decideOnGuardType(Pat,CType,condList([C|CList],[R|RList]),NewR).

transformGuards(_Pat,R,R).
  
% return (last paramater) the rhs for the first rule. For the other guards/rhs
% create completely new rules
decideOnGuardType(Pat,constraint,condList([C|CList],[R|RList]),cond(C,R)):-!,
  %write('type of guard is constraint. creating rules'),nl,
  createNewGuardRules(Pat,CList,RList).

  createNewGuardRules(_L,[],[]).
  createNewGuardRules(L,[C|CList],[R|RList]):-
    assert(rule(L,cond(C,R))),
    %write('created rule '),write(L),write(' = '),write(cond(C,R)),nl,
    createNewGuardRules(L,CList,RList).

% create a if-then-else cascade and return as rhs
% if there is only one guard, then do not make a cascade but
% change the guard into a constraint by adding '=true'.
decideOnGuardType(_Pat,bool,condList([C|CList],[R|RList]),NewR):-!,
  %write('guard type is bool'),nl,
  if(CList==[],
	 NewR=cond(C=true,R),
	 cascade([C|CList],[R|RList],NewR))
  %write('created cascade '),write(R)
  .

  cascade([C],[R],'$ite'(C,R,failed)).
  cascade([C|CList],[R|RList],'$ite'(C,R,E)):-
    cascade(CList,RList,E).




  
% collect all rules for a function
getAllFuncRules(Func,Rules):-
   ifc(getFuncRule(Func,Rule),
	   (getAllFuncRules(Func,Rules1),Rules=[Rule|Rules1]),
	   Rules=[]).
   
%checkAnyConsRule(Num,Rules) prueft auf eine Regel mit irgendeinem
%Konstruktor an der Stelle Num, ohne die Regel zu entfernen
checkAnyConsRule(Num,[rule(L,_R)|Rules]):-!,
	ifc((teilterm(L,Num,Arg),nonvar(Arg)),
	   true,
	   checkAnyConsRule(Num,Rules)).

%getAnyConsRule(Num,Rules,Constructor,Rule) sucht auf eine Regel mit irgendeinem
%Konstruktor an der Stelle Num und liefert Konstruktor und Regel zurueck
getAnyConsRule(Num,[rule(L,R)|Rules],Cons,Rule):-!,
	ifc((teilterm(L,Num,Arg),nonvar(Arg)),
	   (Rule=rule(L,R),Arg=..[Cons|_]),
	   getAnyConsRule(Num,Rules,Cons,Rule)).

%checkVarRule(Num,Rules) prueft auf eine Regel mit einer
%Variable an der Stelle Num, ohne die Regel zu entfernen
checkVarRule(Num,[rule(L,_R)|Rules]):-
  teilterm(L,Num,Arg),
  ifc(var(Arg),
	  true,
	  checkVarRule(Num,Rules)).
  

%getVarRule(Num,Rules) sucht auf eine Regel mit einer
%Variable an der Stelle Num und liefert sie zurueck
getVarRule(Num,[rule(L,R)|Rules],Rule):-
	teilterm(L,Num,Arg),
        ifc(var(Arg),
	   Rule=rule(L,R),
	   getVarRule(Num,Rules,Rule)).

%separateRules(Num,RuleList,ListOfConsRules,ListOfVarRules)
%separates the list of rules into two lists of rules:
%one with all rules with a constructor at position num,
%one with all rules with a variable at this position

separateRules(_Num,[],[],[]).
separateRules(Num,[rule(L,R)|Rs],Cs,Vs):-
  separateRules(Num,Rs,Cs1,Vs1),
  teilterm(L,Num,Arg),
  ifc(var(Arg),
	  (Vs=[rule(L,R)|Vs1],Cs=Cs1),
	  (Cs=[rule(L,R)|Cs1],Vs=Vs1)).


%getConsRule(Num,Cons,Rules,ConsRule,OtherRules).
%finds and removes one rule with the specified constructor at position Num
%ConsRule will be a list with one or no element, if not such rule exists
getConsRule(_,_,[],[],[]).
getConsRule(Num,Cons,[rule(L,R)|Rs],ConsRules,OtherRules):-
     teilterm(L,Num,Arg),
     ifc((nonvar(Arg),Arg=..[Cons|_]),   %Test auf gewuenschten Konstruktor
	(ConsRules=[rule(L,R)],
	 OtherRules=Rs),
        (getConsRule(Num,Cons,Rs,ConsRules,OtherRules1),
	 OtherRules=[rule(L,R)|OtherRules1])).


%getAllConsRules(Num,Cons,Rules,ConsRules,OtherRules).
%getAllConsRules such aus der Regelliste alle Regeln raus, die an Num
%den gewuenschten Konstruktor haben

getAllConsRules(_Num,_,[],[],[]).
getAllConsRules(Num,Cons,[rule(L,R)|Rs],ConsRules,OtherRules):-
     getAllConsRules(Num,Cons,Rs,ConsRules1,OtherRules1),
     teilterm(L,Num,Arg),
     ifc((nonvar(Arg),Arg=..[Cons|_]),   %Test auf gewuenschten Konstruktor
	(ConsRules=[rule(L,R)|ConsRules1],
	 OtherRules=OtherRules1),
        (ConsRules=ConsRules1,
	 OtherRules=[rule(L,R)|OtherRules1])).


%getConsRules(Num,Cons,Rules,ConsRules,OtherRules).
%getConsRules sucht aus der Regelliste alle Regeln raus, die
%an Num denselben Konstruktor haben wie die erste Regel, die gefunden wird.
%liefert den Konstruktor auch zurueck
getConsRules(_Num,_,[],[],[]).  %z.B. wenn nur Rules mit Variablen
getConsRules(Num,Cons,[rule(L,R)|Rs],ConsRules,OtherRules):-
        teilterm(L,Num,Arg),
        ifc((nonvar(Arg),Arg=..[Cons|_]),   %Test auf irgendeinen Konstruktor
	(getAllConsRules(Num,Cons,Rs,ConsRules1,OtherRules),
	 ConsRules=[rule(L,R)|ConsRules1]),
	(getConsRules(Num,Cons,Rs,ConsRules,OtherRules1), %kein Konstruktor,dann
	 OtherRules=[rule(L,R)|OtherRules1])).   %rekursiv aufrufen


%%% getRuleForEachCons(Num,Rules,[C|ConstrList],ErgRules,OtherRules)
%%% tries to find a rule for each constructor
getRuleForEachCons(_,Rules,[],[],Rules).    %no more constructors in list
getRuleForEachCons(_,[],_,[],[]).	    %no more rules
getRuleForEachCons(Num,Rules,[C|ConstrList],ErgRules,OtherRules):-
	getConsRule(Num,C,Rules,ConsRule,RestRules),
	getRuleForEachCons(Num,RestRules,ConstrList,ErgRules1,OtherRules),
	append(ConsRule,ErgRules1,ErgRules).



%noMoreRules, prueft nur, ob die Regelliste leer ist
noMoreRules([]).


%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%% MakeTree


%makeTrees sucht alle Baeume aus dem Speicher und erstellt die zugehoerigen
% Funktionsbaeume

makeTrees:-
	write('Trees'),flush_output,  
	makeAllTrees,
	nl,
        checkForMoreTrees.

makeTrees:-nl,write(' generating aborted.').

checkForMoreTrees:-
        ifc(retract(evaltree(Func,Tree)),
	    (write('****Warning: Eval annotation found without matching function definition. '),
	     nl,write('Function name: '),write(Func),
	     nl,write('Annotation: '),write(Tree),nl,flush_output,
	     checkForMoreTrees),
	    true).


makeAllTrees:-function(Func,_,_),
  createTheTree(Func),fail.

createTheTree(Func):-!,
  ifc(evaltree(Func,Tree),    % Baum zu Funktion angegeben?
	  (makeATree(Func,Tree),
	   retract(evaltree(Func,Tree))),     %erst hier retracten. Wird bei Erzeugung
		                                  %benoetigt, um die komplette Annotation 
		                                  %bei Fehlern auszugeben,
	  makeATree(Func)),
  write('.'),flush_output,!.
	  
makeAllTrees:-write('generated'),nl,flush_output.
  
makeATree(Func,Tree):-if(makeEvalTree(Func,Tree),true,errorInTree(Func)).
makeATree(Func):-if(makeDefaultTree(Func),true,errorInTree(Func)).

errorInTree(Func):-
  nl,write('****Error: Cannot create tree for function '),write(Func),
  write(' for some unknown reason!'),nl,!,fail.

  
%%%%%%%%%%% makeDefaultTree
% erstellt einen Baum zu einer Funktion nach einer Standardstrategie,
% falls kein expliziter Baum angegegen wurde

%makeDP. Berechnet fuer einen Term alle Positionen, an denen eine Variable steht.
%Wird codiert in der Form 1#2#1 statt 1.2.1 weg. Punkt in Prolog vordef.

makeDP(Pat,DP):-
	Pat=..[_Head|Args],
	checkPositions(Args,1,DP).

%checkPositions prueft jedes Argument der Liste ob es eine Variable ist.
%Wenn ja, Position in Liste der Variablen-Positionen des Pattern hinzufuegen.
%Wenn nein, dann fuer das Argument rekursiv alle Variablenstellen berechnen
%und die Position des Arguments davorhaengen. also z.B. bei f(s(X)) hat
%s(X) Position 1, ist keine Variable, rekursiv wird wieder 1 geliefert
%also als Ergebnis 1.1 berechnet

checkPositions([],_,[]).
checkPositions([X|Xs],N,DP):-
	ifc(var(X),
	   (N1 is N+1,
	    checkPositions(Xs,N1,DP1),
 	    DP=[N|DP1]),
	   (makeDP(X,RekDP),
	    addPosition(N,RekDP,DPList),
	    N1 is N+1,
	    checkPositions(Xs,N1,DP1),
	    append(DPList,DP1,DP))).

addPosition(_N,[],[]).
addPosition(N,[X|Xs],[N'#'X|Xs1]):-addPosition(N,Xs,Xs1).

%checkDP berechnet die eigtl. DP Menge, also die Menge der Positionen, fuer die
%mindestens eine Regel existiert, die an dieser Position einen Konstruktor hat

checkDP([],_Rules,[]).
checkDP([X|Xs],Rules,NewDP):-
	 ifc(checkAnyConsRule(X,Rules),   % existiert Regel mit einem Konstruktor?
 (NewDP=[X|NewDP1],
	     checkDP(Xs,Rules,NewDP1)),
	    checkDP(Xs,Rules,NewDP)).

%-------------------------
%-------------------------
%makeDefault
%Tree erstellen, checken ob alle Regeln verbraucht
%und Tree hinzufuegen.

makeDefaultTree(Func) :-             %externe Funktionen gesondert behandeln
	isExternal(Func),!,
        makeExternalDefaultTree(Func,Tree),
	assert(tree(Func,Tree)).


makeDefaultTree(Func):-
	fArity(Func,N),
	makePattern(Func,N,Pat),
	getAllFuncRules(Func,Rules),!,
        ifc(Rules==[],
            (nl,write('****Error: No rules defined for function '),write(Func),
	     write('.'),nl,write('Cannot create tree!'),!,fail),
	    true),
	getDefaultTree(Func,Pat,Rules,ErgTree),
	assert(tree(Func,ErgTree)).


%getDefaultTree
%checken, ob DP leer. In dem Fall einfach einen Regelbaum erstellen.
getDefaultTree(Func,Pat,Rules,ErgTree):-
	makeDP(Pat,DP),checkDP(DP,Rules,NewDP),!,
	ifc(NewDP=[],getDefaultRuleTree(Func,Rules,ErgTree),
	   defaultTreeDPNotEmpty(Func,Pat,NewDP,Rules,ErgTree)).

%DP ist also nicht leer. 2 Flle: Entweder es existiert eine Regel (oder mehr)
% mit einer Variable an der leftmost-Position aus DP (erstes Element).
% Dann erstelle einen Or-Baum, ein Teil mit den Variablenregeln,
% ein Teil mit den Konstruktor-Regeln.
% Im anderen Fall wird ein Branch-Baum erstellt.
defaultTreeDPNotEmpty(Func,Pat,[Pos|DP],Rules,ErgTree):-
	ifc(checkVarRule(Pos,Rules),
	    if(optflag,                                  
% optimized definitional Tree 
                avoidOr(Func,Pat,[Pos],DP,Rules,ErgTree),          
% standard definitional Tree
	        defaultOrTree(Func,Pat,Pos,Rules,ErgTree)),        
	   defaultBranchTree(Func,Pat,[Pos|DP],Rules,ErgTree)).


% Search DP-Set for first Position with only Constructor-rules and create 
% a branch tree at this position.
% If no such Position exists, create an or-tree as usual.
avoidOr(Func,Pat,[Pos|_],[],Rules,ErgTree):-
         defaultOrTree(Func,Pat,Pos,Rules,ErgTree).
avoidOr(Func,Pat,VarPosList,[Pos|DP],Rules,ErgTree):-
         separateRules(Pos,Rules,_ConsRules,VarRules),
         ifc(VarRules=[],
             (append(VarPosList,DP,NewVarPosList),
              defaultBranchTree(Func,Pat,[Pos|NewVarPosList],Rules,ErgTree)),
             (append(VarPosList,[Pos],NewVarPosList),
              avoidOr(Func,Pat,NewVarPosList,DP,Rules,ErgTree)) 
             ).

%getDefaultRuleTree
%The DP-Set was empty, so there are only rules left that are identical
%in their left sides. If the right sides differ, then generate an
%or-tree, otherwise take only one rule and generate a warning message

%if only one rule exists, create a normal rule-Tree
getDefaultRuleTree(_,[R],R).
getDefaultRuleTree(Func,[rule(L1,R1),rule(L2,R2)|Rs],ErgTree):-  %more than one rule
           ifc(variants((L1=R1),(L2=R2)),
	      (nl,write('**** Warning: Rule for function '),
	       write(Func),write(' is redundant, will be dropped: '),nl,
	       writeNames((L1=R1)),nl,nl,
%continue but drop first rule
	       getDefaultRuleTree(Func,[rule(L2,R2)|Rs],ErgTree)), 
%different rules, then continue and create an or-tree
	      (getDefaultRuleTree(Func,[rule(L2,R2)|Rs],ErgTree1),
	       ErgTree=or(rule(L1,R1),ErgTree1))).
	   
        
	   
%separate list of rules in two lists: one with all rules with constructors 
%at Num and one with all rules with variables at Num. Create own tree
%for each 
defaultOrTree(Func,Pat,Num,Rules,or(ErgTree1,ErgTree2)):-
	separateRules(Num,Rules,ConsRules,VarRules),
	getDefaultTree(Func,Pat,ConsRules,ErgTree1),
	getDefaultTree(Func,Pat,VarRules,ErgTree2).

	
%defaultBranchTree
%Sammelt fuer jeden Konstruktor alle Regeln mit diesem Konstruktor (bzgl.
%der leftmost-Position aus DP), und erstellt einen Branch-Baum wobei
%fuer jeden Zweig der Baum eingesetzt wird, der rekursiv fuer
%jeden Konstruktor berechnet wurde. (siehe Hanus-Paper)


defaultBranchTree(Func,Pat,[Num|DP],Rules,branch(Pat,Num,Flag,ErgTrees)):-
	getFlag(Func,Flag),
	defaultBranchHelpFunc(Func,Pat,[Num|DP],Rules,ErgTrees).

defaultBranchHelpFunc(Func,Pat,[Num|DP],Rules,ErgTrees):-
	getConsRules(Num,Cons,Rules,ConsRules,OtherRules),!,   %Regeln zu 1 Konstr. suchen
	ifc(ConsRules=[],
           ErgTrees=[],
           (changePattern(Pat,Num,Cons,NewPat),  %Pattern updaten
	    getDefaultTree(Func,NewPat,ConsRules,ErgTree1), %und tree erstellen
	    defaultBranchHelpFunc(Func,Pat,[Num|DP],OtherRules,ErgTrees1),
%rekursiver Aufruf fuer die anderen Konstruktoren
	   ErgTrees=[ErgTree1|ErgTrees1])).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	   
% makeExternalDefaultTree(Func,Tree) -> makeExternalDefaultTree(Func,N,Counter,Tree).
% creates a standard tree of type 1:rigid(_=>2:rigid(_=>...))
% Counter is neccessary because the LAST part of the tree is different from 
% the others, i.e. it contains a rule.
% Keep in mind that no matching on constructors is allowed! Only variables!

makeExternalDefaultTree(Func,Tree):-
	fArity(Func,N),
	makePattern(Func,N,Pat),
	makeExternalDefaultTree(Pat,N,1,Tree).

makeExternalDefaultTree(Pat,0,_N,rule(Pat,_)).                %0-ary Function
makeExternalDefaultTree(Pat,N,N,branch(Pat,N,rigid,[rule(Pat,_)])). % last part of tree


%!!! Achtung! Evtl. muss in rule neues Pattern rein und nicht das alte Pat. Wird
% so jedenfalls bei allen anderen Bumen gemacht.

makeExternalDefaultTree(Pat,N,Counter,branch(Pat,Counter,rigid,[Tree])):-
	Counter1 is Counter+1,
	makeExternalDefaultTree(Pat,N,Counter1,Tree).



%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%	   
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% generate definitional tree according to eval specification


%%%makeEvalTree(Func,Tree) creates a definitional tree for Func according
%%%to the specified eval-annotation Tree. No return-parameter!

makeEvalTree(Func,Tree) :-             %externe Funktionen gesondert behandeln
	isExternal(Func),!,
	decideOnExternalTree(Func,Tree,ErgTree),
	assert(tree(Func,ErgTree)).

% If an external function has a "eval f rigid" tree, just create
% the standard tree.
decideOnExternalTree(Func,rigid,ErgTree):-makeExternalDefaultTree(Func,ErgTree).
decideOnExternalTree(Func,Tree,ErgTree):-
        checkExternalEvalAnnot(Func,Tree),
					  makeExternalEvalTree(Func,Tree,ErgTree).

					  
% If the tree is "f eval choice" sth. special must be done. Collect all rules
% and transform them into ONE rule with a committed choice function on the rhs.
% The scheme is (as example we have one rule with and one without guards):
% f(t1,..,tn) | L localVars c = r1
% f(s1,..,sn)                 = r2
%
% That is translated into:
% f(x1,..,xn) = committedChoice   L u M1 localVars t1<--x1/\../\tn<--xn/\c -> r1
%                                     M2 localVars s1<--x1/\../\sn<--xn    -> r2
%
% where x1,..,xn are fresh variables, M1 are the variables occuring in the ti,
% M2 those occuring in the si.
% <-- is the matching function which performs exactly the same as would do the
% pattern matching when calling the original rules for f.
%
% By that transformation, we translate the new annotation "f eval choice" into
% the old form using a choice-function.
%
% NEW: Additionally create a tree "choice(normal tree)" for the new choice in
% curry2java


  
makeEvalTree(Func,choice):-!,
  changeFlagInto(Func,flex),% the new function must be flexible because no patterns will occur in the lhs
                            % and all arguments (even variables) must be passed to the choice function
                            % suspension of variable arguments will take place there if neccessary
  fArity(Func,N),
  makePatternList(Func,N,Pattern,VarList), % create the new lhs and return also the new variables
  createChoiceArgs(Func,VarList,RuleList,Args),             % create the list of (guard,rhs) pairs
  assertz(tree(Func,rule(Pattern,committedChoice(Args)))),  % and create the tree
  changeFlagInto(Func,residuate),
  getDefaultTree(Func,Pattern,RuleList,ErgTree),                % additionally get normal tree
  assert(choicetree(Func,choice(ErgTree))).                 % and make it a choice tree
  

  %%%%%%%%%%%%%%%%%%%%%%%%
  % create a pair (guard,rhs) where guard is combined from an optional guard that
  % comes with the rule and the new matching constraint
  % Also return the list of rules found for the actual function
  
  createChoiceArgs(Func,VarList,[rule(L,R)|RuleList],[ChoiceRule|Rules]):-
    getFuncRule(Func,rule(L,R)),!,      % take a rule for that function
    L=..[Func|PatternList],           % and get the pattern of the lhs
    createMatchingConstraint(PatternList,VarList,MatchingConstraint,PatternVars),
                                      % create the t1<--x1/\../\tn<--xn constraint
    createChoiceRule(R,MatchingConstraint,PatternVars,ChoiceRule),!, % combine it with the guard and the rhs
    createChoiceArgs(Func,VarList,RuleList,Rules).
  
  createChoiceArgs(_Func,_VarList,[],[]):-!.

  %%%%%%%%%%%%%%%%%%%%%%%%%
  % creates the t1<--x1/\../\tn<--xn constraint. First arg: [t1,..,tn], second: [x1,..,xn]
  % returns: constraint and the variables found in the ti
  createMatchingConstraint([],[],{},[]).  % for 0-ary functions
  createMatchingConstraint([P],[X],P<--X,PatternVars):-freeVars(P,PatternVars).
  createMatchingConstraint([P|Ps],[V|Vs],P<--V/\RestConstr,PatternVars):-
    freeVars(P,PatVars1),
    createMatchingConstraint(Ps,Vs,RestConstr,PatVars2),
    append(PatVars1,PatVars2,PatternVars).

  % combine the t1<--x1/\../\tn<--xn constraint and an optional rule guard into one guard
  createChoiceRule(R,MConstr,PatVars,(NewC,R)):-var(R),!,
    declareVarsListInTerm(PatVars,MConstr,NewC).
  createChoiceRule(cond(C,R),MConstr,PatVars,(NewC,R)):-!,
    combineCandMC(C,MConstr,PatVars,NewC).
  createChoiceRule(R,MConstr,PatVars,(NewC,R)):-!,
    declareVarsListInTerm(PatVars,MConstr,NewC).
  
      combineCandMC(C,MConstr,PatVars,NewC):-var(C),!,
        declareVarsListInTerm(PatVars,MConstr/\C,NewC).
      combineCandMC(Vars localVars C,MConstr,PatVars,NewC):-
        append(PatVars,Vars,AllVars),
        declareVarsListInTerm(AllVars,MConstr/\C,NewC).
      combineCandMC(C,MConstr,PatVars,NewC):-
        declareVarsListInTerm(PatVars,MConstr/\C,NewC).

% If the tree is "f eval flex" or "f eval rigid", just change the tree-flag for
% the function and create the standard tree then, nothing else is meant by such
% an annotation.
makeEvalTree(Func,flex):-!,
	changeFlagInto(Func,narrow),
	makeDefaultTree(Func).

makeEvalTree(Func,rigid):-!,
	changeFlagInto(Func,residuate),
	makeDefaultTree(Func).

makeEvalTree(Func,Tree):-
	fArity(Func,N),
	makePattern(Func,N,Pat),
	getAllFuncRules(Func,Rules),!,
	getEvalTree(Func,Tree,Pat,Rules,RestRules,ErgTree),!,
	ifc(noMoreRules(RestRules),         %all rules must have been used
	    assert(tree(Func,ErgTree)),
	    (nl,write('****Error: Too many rules specified for function according to eval annotation:'),
	     nl,write('Function: '),write(Func),nl,write('Annotation: '),write(Tree),nl,
	     write('List of rules left: '),writeRuleList(RestRules),nl,!,fail)).

%-------------------------

%checkExternalEvalAnnot
%checks if the tree defined for a external function has correct format.


checkExternalEvalAnnot(_Func,rule).
checkExternalEvalAnnot(Func,(Pos:rigid)):-!,checkNumber(Func,Pos).
checkExternalEvalAnnot(Func,(Pos:rigid(X=>Tree))):-
	ifc(checkNumber(Func,Pos),
	    (var(X),
	     checkExternalEvalAnnot(Func,Tree)),
	    true).

checkExternalEvalAnnot(Func,Tree):-nl,
        write('****Error: Wrong eval annotation for external function.'),nl,
	write('Function: '),write(Func),nl,write('Annotation: '),write(Tree),nl,
	write('Correct eval annotation format:'),nl,
	write('annotation ::= rule | number:rigid [Variable=>annotation]'),nl,!,fail.


checkNumber(Func,Pos):-
	number(Pos),
	fArity(Func,N),
	Pos<(N+1),!.
checkNumber(Func,Pos):-nl,
	write('****Error: Wrong position in eval annotation for external function.'),nl,
	write('Function: '),write(Func),nl,write('Position: '),write(Pos),nl,!,fail.


% makeExternalEvalTree(Func,Tree,ErgTree)
% creates a tree for the external function from the eval annotation


makeExternalEvalTree(Func,Tree,ErgTree):-
	fArity(Func,N),
	makePattern(Func,N,Pat),
	makeExternalEvalTree2(Pat,Tree,ErgTree).

%!!! Achtung! Evtl. muss in rule neues Pattern rein und nicht das alte Pat. Wird
% so jedenfalls bei allen anderen Bumen gemacht.

makeExternalEvalTree2(Pat,rule,rule(Pat,_)).               
makeExternalEvalTree2(Pat,N:rigid,branch(Pat,N,rigid,[rule(Pat,_)])). % last part of tree
makeExternalEvalTree2(Pat,N:rigid(_X=>Tree),branch(Pat,N,rigid,[ErgTree])):-
	makeExternalEvalTree2(Pat,Tree,ErgTree).



%-------------------------

%%%getEvalTree(Func,Tree,Pat,Rules,Flag,RestRules,ErgTree)
%%%Tree = specified eval annotation
%%%Rules = list of rules for Func
%%% RestRules = list of rules that are left after getEvalTree has finished
%%% this list may be not empty if e.g. it is called with an "or" eval annot.
%%% The first branch of the "or" tree shell use some rules but not all.

getEvalTree(Func,Tree,_Pat,[],_Rules,_ErgTree):-
        nl,write('****Error: No rules left for creating evaluation tree.'),nl,
	write('Function: '),write(Func),nl,
	write('Annotation: '),write(Tree),nl,!,fail.

%%%create tree for "rule" annotation. 
%%%The DP-set must be empty, i.e. no rule with a construcor at a variable
%%%position of the pattern shell exist. 
getEvalTree(Func,rule,Pat,[Rule|Rules],Rules,Rule):-!,
        testForCompletedEvaluation(Func,Pat,rule,[Rule]).  %check if there are arguments that must
		                                      %be evaluated (see definition of function)


%-------------------------

% 4.9.97: and-trees disabled, also in curry.pl

% and-Tree
% both parts of the and-tree work with the complete set of rules!
%getEvalTree(Func,(Tree1 and Tree2),Pat,Rules,RestRules,and(Erg1,Erg2)):-!,
%	getEvalTree(Func,Tree1,Pat,Rules,RestRules1,Erg1),!,
%	getEvalTree(Func,Tree2,Pat,Rules,RestRules2,Erg2),!,
%        ifc(RestRules1==RestRules2,RestRules=RestRules1,
%           (nl,write('****Error: Wrong definition of eval annotation or rules.'),
%	    nl,
%	    write('Branches of "and" eval annotation do not use the same set of rules!'),nl,
%	    write('Function: '),write(Func),nl,
%	    write('Annotation: '),write(Tree1 and Tree2),nl,!,fail)).


%-----------------------

% or-Tree
% the second part of the tree works with the rules that have not been
% used by the first part
getEvalTree(Func,(Tree1 or Tree2),Pat,Rules,RestRules,or(Erg1,Erg2)):-!,
	getEvalTree(Func,Tree1,Pat,Rules,RestRules1,Erg1),!,
	getEvalTree(Func,Tree2,Pat,RestRules1,RestRules,Erg2).

%-----------------------

%%% Tree with position, flag and possibly further cons-annotations

%%% first retract the position from the pattern, then divide the rest
%%% into the flag and (possibly) further annotations
getEvalTree(Func,(Position:Flexrigid),Pat,Rules,RestRules,ErgTree):-!,
	ifc(correctPosition(Position),   %check if position has correct syntax
	   (divideFlagAndAnnos(Func,Flexrigid,Flag,Annos),!,  %divide into the flag and the rest
            getEvalBranch(Func,Position,Flag,Annos,Pat,Rules,RestRules,ErgTree)),
	   (nl,write('****Error: Position in annotation has wrong format.'),nl,
	    write('Function: '),write(Func),nl,
	    evaltree(Func,ETree),
	    write('Annotation: '),write(ETree),nl,
	    write('Error at: '),
	    write((Position:Flexrigid)),nl,!,fail)).


%----------------------

%%%error case. 
getEvalTree(Func,Position,_,_,_,_):-correctPosition(Position),
	nl,write('****Error: Position in eval annotation must be followed by rigid/flex.'),nl,
	write('Function: '),write(Func),nl,
	evaltree(Func,ETree),
	write('Annotation: '),write(ETree),nl,
	write('Postion: '),write(Position),nl,!,fail.


getEvalTree(Func,Tree,_,_,_,_):-
	nl,write('****Error: Unknown construction in eval annotation:'),nl,
        write('Function: '),write(Func),nl,
        evaltree(Func,ETree),
	write('Annotation: '),write(ETree),nl,
	write('Error at: '),write(Tree),nl,!,fail.
	

%-----------------------

%%% getEvalBranch
%%% If annotationList is empty then make simple branch tree without rekursion

%%% First check type of parameter at position. 
%%% If the position is a single number, then the parameter can be of three types:
%%% polymorph, function type, simple type.
%%% If it is polymorph, take one rule (all rules have variables 
%%% at this position). If it is a function type, report an error (no branch on
%%% function types). If it is a simple type then take one rule for each constructor
%%% of this type or, if no exist, try to find a rule with a variable (this is valid only
%%% if the position is a simple number).
%%% If the position is of type A#B... then only rules with a constructor at this position
%%% are valid. Out of these try again to take one rule for each constructor.



getEvalBranch(Func,Position,Flag,[],Pat,Rules,RestRules,ErgTree):-
        checkMainTypeOfParameter(Func,Position,Type),!,
        ifc(isType(Type),   
%simple type, then check for constructor based rules
	    ifc(getAnyConsRule(Position,Rules,Cons,Rule),   %Rule with constructor exists?
		 (consType(Cons,ConsType),         %get type of constructor
		  getTypen(ConsType,Position,Rules,ConsList),        %get all constructors of this type
		  getRuleForEachCons(Position,Rules,ConsList,ErgRules,RestRules),!,
		  tryToChangePattern(Func,Pat,Position,Cons,NewPat),!,  
%change pattern because the actual position must not be tested by the following test
		  testForCompletedEvaluation(Func,NewPat,(Position:Flag),ErgRules),!,
%test if there is any rule with parameters that must be evaluated but that are NOT evaluated
%by the eval annoation
		  ifc(ErgRules=[],
		      (nl,write('****Error: No rules left for creating evaluation tree.'),
		       nl,write('Function: '),write(Func),nl,
		       evaltree(Func,ETree),
		       write('Annotation: '),write(ETree),nl,
		       write('Error at: '),
		       write((Position:Flag)),nl,!,fail),
		      true)),
%else: no consrule exists, search for one variable rule
		(ifc(number(Position),
%if position is a number, then a rule with a variable is allowed at this position
%as only rules with a variable or a constructor at a number position can exist, there
%must be a variable rule in this case. The case that no rules exists is matched above
		      (getVarRule(Position,Rules,Rule),
		       getEvalTree(Func,rule,Pat,Rules,RestRules,RuleTree),   %take one rule
			   ErgRules=[RuleTree]),
		      (nl,write('***Error: No rule with a constructor at the specified position does exist.'),
		       nl,write('Function: '),write(Func),nl,
		       evaltree(Func,ETree),
		       write('Annotation: '),write(ETree),nl,
		       write('Error at: '),write((Position:Flag)),
		       nl,!,fail)))),
	    polymorphOrFuncType(Func,Position,Type,Pat,Rules,RestRules,ErgRules)),
%Type is polymorph or variable

	makeBranch(Pat,Position,Flag,ErgRules,ErgTree).		      




%%% annotation-list is NOT empty. Again check type of parameter at Position.
%%% If type is
%%%	- polymorph: error, because only rules with variables shell exist and
%%%	  so a cons-annotation is not allowed
%%%	- simpleType: rekursively create trees for each cons-annotation. At the
%%%	  end take one rule for each constructor of this type, for that no annotation
%%%	  was specified, e.g.: 1:rigid(s=>...)   Create tree with all s-Rules, then
%%%	  take ONE rule with 0 at this position (for type is Num with constructors s and 0
%%%	- functionType: error


getEvalBranch(Func,Position,Flag,Annotations,Pat,Rules,RestRules,ErgTree):-
	checkMainTypeOfParameter(Func,Position,Type),!,
	ifc(var(Type),
	   (nl,write('****Error: Branch with constructors on polymorph type!'),
	    nl,write('Function: '),
	    write(Func),nl,
	    evaltree(Func,ETree),
	    write('Annotation: '),write(ETree),nl,
	    write('Error at: '),
	    write(Position),write(':'),write(Position),write(Annotations),nl,!,fail),
	   (ifc(isType(Type),
  	    %(annosIntoList(Annotations,AnnoList),   % put annotations in list form
	    (getTypen(Type,Position,Rules,ConsList),!,       	    %get all constructors for type
	     ifc(makeTreesForAnnos(Func,Position,Annotations,Pat,ConsList,Rules,RestRules,ErgTrees),
		  makeBranch(Pat,Position,Flag,ErgTrees,ErgTree),  %create the tree
		 fail)),
            branchError(Func,Position)))
	).





%----------------------

polymorphOrFuncType(Func,Position,Type,Pat,Rules,RestRules,ErgRules):-
        ifc(var(Type),   %variable type
	    ifc(number(Position),  %then only a simple number is ok as position
		(getEvalTree(Func,rule,Pat,Rules,RestRules,RuleTree),   %take one rule
		 ErgRules=[RuleTree]),
		(nl,write('****Error: Invalid eval annoation. Position not valid for polymorph parameter.'),
		 nl,write('Function: '),write(Func),
		 evaltree(Func,ETree),
		 write('Annotation: '),write(ETree),nl,
		 nl,write('Position: '),write(Position),nl,!,fail)),
	    branchError(Func,Position)).   


%---------------------

%tryToChangePattern
%called by getEvalBranch. Try to change the pattern to enable the test for completedEvaluation
%This might not be possible, e.g. the pattern is f(X) and the position is 1#1. In this case
%the annotation was wrong, i.e. no 1:flag was specified before 1#1 was specified.

tryToChangePattern(Func,Pat,Position,Cons,NewPat):-
        ifc(changePattern(Pat,Position,Cons,NewPat),
	    true,
	    (nl,write('****Error: Wrong annotation format. A position n#m must be preceded '),
	     nl,write('by a n:flag(cons=>...) notation.'),
	     nl,write('Function: '),write(Func),
	     evaltree(Func,ETree),
	     write('Annotation: '),write(ETree),nl,
	     nl,write('Invalid position: '),write(Position),nl,!,fail)).

%---------------------

branchError(Func,Position):-
        ifc(Position=A#_B,         % parameter is of function type. Only first number of position
	    Pos=A,                % is interesting for error report
	    Pos=Position),
	nl,write('*** Error in tree: Branch defined for function-type.'),nl,
        write('Function: '),write(Func),nl,
        evaltree(Func,ETree),
	write('Annotation: '),write(ETree),nl, 
        write('No. of parameter: '),write(Pos),nl,!,fail.

%----------------------

%%% AnnosIntoList
%%% puts the annotations into a list, e.g. (s=>...,0=>...) into [s=>...,0=>...]

annosIntoList(C=>Anno,[C=>Anno]).   %only one cons annotation
annosIntoList((ConsAnno1,ConsAnno2),[ConsAnno1|RestConsAnnos]):-
	annosIntoList(ConsAnno2,RestConsAnnos).  %more than one cons annotations

%----------------------

%%% makeTreesForAnnos(Func,Position,Annos,Pat,Constructors,Rules,RestRules,ErgRules)
%%% creates trees for each cons annotation. ErgRules will be a list of these trees
%%% This list will be formatted into a branch tree by getEvalTree
%%% For one annotation get all rules with the specified constructor at this position
%%% and create tree rekursively. At the end get one rule for each constructor for
%%% that no annotation was specified (According to the branch principle)

makeTreesForAnnos(_Func,_Position,[],_Pat,[],Rules,Rules,[]).      %no constructors left
makeTreesForAnnos(_Func,_Position,[],_Pat,_Constructors,[],[],[]).  %no rules left
makeTreesForAnnos(Func,Position,[],Pat,Constructors,Rules,RestRules,ConsRules):-
    %no more annotations
	getRuleForEachCons(Position,Rules,Constructors,ConsRules,RestRules),!,
	Constructors=[C|_],
	tryToChangePattern(Func,Pat,Position,C,NewPat),
    %actual position shell not be tested, therefore change pattern
	testForCompletedEvaluation(Func,NewPat,Position,ConsRules).
    %then get one rule for each constructor that is left, i.e. for that no annotation was given



%-----
%one or more normal cons annotations	
makeTreesForAnnos(Func,Position,[C=>Anno|Annos],Pat,Constructors,Rules,RestRules,ErgTrees):-nonvar(C),!,
	ifc(checkTypeValidity(Func,Position,C),    % specified constructor allowed for function?
	   (remove(Constructors,C,RestConstructors),   
	    getAllConsRules(Position,C,Rules,ConsRules,OtherRules),  
%get all rules that have the specified constructor
	    tryToChangePattern(Func,Pat,Position,C,NewPat),   
%create new pattern for rekursive call
            !,getEvalTree(Func,Anno,NewPat,ConsRules,RestRules1,ErgTree),
%create tree for annotation
	    !,makeTreesForAnnos(Func,Position,Annos,Pat,RestConstructors,OtherRules,RestRules2,ErgTrees1),
%create trees for remaining annos. 
	    append(RestRules1,RestRules2,RestRules),
%give back all rules that have not been used
	    append(ErgTrees1,[ErgTree],ErgTrees)),
%give back all created Trees as list

%else
	   (nl,write('****Error: Function and constructor in annotation have different types. '),nl,
	    write('Function: '),
	    write(Func),nl,
	    evaltree(Func,ETree),
	    write('Annotation: '),write(ETree),nl,
	    write('Argument Type: '),
	    checkMainTypeOfParameter(Func,Position,ArgType),
	    write(ArgType),nl,
	    write('Constructor '),
	    write(C),write(' in annotation is of type '),
	    consType(C,CType),write(CType),nl,!,fail)).
			
%-----
%special case: Variable=>Annotation
%then only one annotation shall exist,i.e. this one. Then easily create a tree for Annotation
%and place a branch-tree around it (will be done by the calling getEvalTree).
%e.g. (1:flex(_Var=>2:...) will finally result in branch(f(X,Y),1,flex,[branch(f(X,Y),2,...]
%
%The only rules allowed are those with a variable at the position that is followed bei _Var
%for it might not be possible (for external declared functions) to extend the _Var to all
%possible constructors and find rules for all these. 

%this special must be done to catch the wrong format n:flag(X) instead of n:flag(X=>...).
%it must be checked first, otherwise the Variable will be bound to Var=>Annos, and a
%new variable Var will be created, resulting in an error.
makeTreesForAnnos(Func,_Position,[Var],_Pat,_Constructors,_Rules,_RestRules,_ErgTrees):-var(Var),!,
        nl,write('****Error: Wrong annotation format:'),nl,
	write('Function: '),write(Func),nl,
	evaltree(Func,ETree),
	write('Annotation: '),write(ETree),nl,
	write('Error at: '),write(Var),nl,!,fail.

makeTreesForAnnos(Func,Position,[(Var=>Anno)],Pat,_Constructors,Rules,RestRules,ErgTrees):-var(Var),!,
	separateRules(Position,Rules,ConsRules,VarRules),!,
	ifc(VarRules=[],
	   (nl,write('****Error: Variable specified in eval annotation,'),nl, 
	    write('but no rules with a variable at specified position found.'),nl,
	    write('Function: '),write(Func),nl,
	    evaltree(Func,ETree),
	    write('Annotation: '),write(ETree),nl,
	    write('Position: '),write(Position),!,fail),
	    (getEvalTree(Func,Anno,Pat,VarRules,RestRules1,ErgTree),
	     append(ConsRules,RestRules1,RestRules),								
	     ErgTrees=[ErgTree])).	

makeTreesForAnnos(Func,_Position,[Anno|_Annos],_Pat,_Constructors,_Rules,_RestRules,_ErgRules):-!,
	nl,write('****Error: Wrong annotation format:'),nl,
	write('Function: '),write(Func),nl,
	evaltree(Func,ETree),
	write('Annotation: '),write(ETree),nl,
	write('Error at: '),write(Anno),nl,!,fail.
	
%-----------------


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Tools 

%getTypen
%for type int it is not possible to get all constructors for this
%are all integer numbers. So collect from the rules all constructors
%that do really appear.
getTypen(Type,Pos,Rules,ConsList):-
	if(Type==int,
           collectConsFromRules(Pos,Rules,ConsList),
           typen(Type,ConsList)).

collectConsFromRules(_Pos,[],[]).
collectConsFromRules(Pos,[rule(L,_R)|Rules],ConsList):-
	teilterm(L,Pos,Teilterm),
        collectConsFromRules(Pos,Rules,ConsList1),
        if(nonvar(Teilterm),
	   addToSet(Teilterm,ConsList1,ConsList),
	   ConsList=ConsList1).

addToSet(X,[],[X]).
addToSet(X,List,List):-member(X,List).
addToSet(X,List,[X|List]).


%---------------

%writeRuleList write a list of rules to the screen, in L=R format.
writeRuleList([]):-nl.
writeRuleList([rule(L,R)|Xs]):-nl,writeNames(L=R),writeRuleList(Xs).

%---------------

%checkTypeValidity
%check if the constructor C is valid with the type of the function at position.
%for positions of type A#B this is not implemented yet
checkTypeValidity(Func,Position,C):-
        number(Position),!,
	ithType(Func,Position,Arg),
	consType(C,CType),
	Arg=CType.
checkTypeValidity(_,_,_).
  


%checkMainTypeOfParameter gets the type of a parameter of the function. The first number (A)
%of a position of type A#B is the one we need.

checkMainTypeOfParameter(Func,A#_B,Type):-!,
        ithType(Func,A,Type).
checkMainTypeOfParameter(Func,A,Type):-
        ithType(Func,A,Type).

%---------------

% makeBranch erstellt aus einer Liste von Regeln einen Branch-Baum.

makeBranch(Pattern,Number,Flag,Rules,branch(Pattern,Number,Flag,Rules)).

%---------------

testForCompletedEvaluation(_Func,_Pat,_Anno,[]).
testForCompletedEvaluation(Func,Pat,Anno,[Rule|Rules]):-!,
        testForCompletedEvaluation1(Func,Pat,Anno,Rule),!,
	testForCompletedEvaluation(Func,Pat,Anno,Rules).							     
testForCompletedEvaluation1(Func,Pat,_Anno,rule(L,R)):-
        makeDP(Pat,DP),!,
	checkDP(DP,[rule(L,R)],Positions),   %test if there are arguments that must be evaluated
	ifc(Positions==[],
           true,
           (nl,write('****Error: Rule for function has arguments that must be evaluated,'),nl,
	    write('i.e. the arguments are not variables but constructors,'),nl,
	    write('but the eval annotation does not specify these positions.'),nl,
	    write('Function: '),write(Func),nl,
	    write('Rule: '),writeNames((L=R)),nl,
	    evaltree(Func,ETree),
	    write('Annotation: '),write(ETree),nl,
	    write('Unspecified Positions: '),write(Positions),!,fail)).


%---------------

%Flag zu einer Funktion bestimmen.
%Wenn standard als Strategie gespeichert ist, dann per
%Konvention: Praedikate als flex, Funktionen als rigid
getFlag(Func,Flag):-
	functionTreeFlag(Func,FuncFlag), %TreeFlag abfragen. Existiert immer
	makeFlag(Func,FuncFlag,Flag).

makeFlag(_Func,narrow,flex).
makeFlag(_Func,residuate,rigid).
makeFlag(Func,standard,Flag):-
	getStandardFlag(Func,Flag).

getStandardFlag(Func,Flag):-
	ergType(Func,Type),
	%IO actions are rigid, other functions flex:
	((nonvar(Type), Type=io(_)) -> Flag=rigid ; Flag=flex).

% needed for eval annotations "f eval flex" and "eval f rigid".
changeFlagInto(Func,Flag):-
	retract(functionTreeFlag(Func,_)),
	assertz(functionTreeFlag(Func,Flag)).

%---------

%correctPosition
%checks if the position has correct format. Position=Number |Number#Position
correctPosition(A#B):-!,number(A),correctPosition(B).
correctPosition(A):-number(A).


%divideFlagAndAnnos
%divides an annotation into the flag and the following annotation

divideFlagAndAnnos(Func,FlagAndAnnos,Flag,Annos):-
	FlagAndAnnos=..[Flag|Annos],!,
	testFlag(Func,Flag,Annos).

testFlag(_Func,rigid,_Annos).
testFlag(_Func,flex,_Annos).
testFlag(Func,Flag,Annos):-
        nl,write('****Error: Wrong or missing flag in annotation. '),nl,
	write('Function: '),write(Func),nl,write('Annotation: '),write(Flag),write(Annos),nl,!,fail.

%divideFlagAndAnnos(_Func,(flex:Annos),flex,Annos).
%divideFlagAndAnnos(_Func,(rigid:Annos),rigid,Annos).
%divideFlagAndAnnos(_Func,flex,flex,[]).
%divideFlagAndAnnos(_Func,rigid,rigid,[]).
%divideFlagAndAnnos(Func,FlagAndAnnos,_,_):-
%	nl,write('****Error: Wrong or missing flag in annotation. '),nl,
%	write('Function: '),write(Func),nl,write('Annotation: '),write(FlagAndAnnos),nl,!,fail.

%makePattern
%for an n-ary function f create pattern f(X1,..,Xn) with Xi are new variables
makePattern(Func,N,Pattern):-
	makeNewVarList(N,List),
	Pattern=..[Func|List].

makePatternList(Func,N,Pattern,List):-
	makeNewVarList(N,List),
	Pattern=..[Func|List].
				
				

%replace part of a pattern with a new pattern, e.g. f(X,Y) -> f(s(Z),Y)
changePattern(Pat,Position,C,NewPat):-
	cArity(C,Arity),
	makePattern(C,Arity,CPat),
	substTerm(Pat,Position,CPat,NewPat).


%remove(A,B,C)
%remove all appearances of A from B
remove([],_,[]).
remove([Y|Xs],Y,E):-remove(Xs,Y,E).
remove([X|Xs],Y,[X|E]):-remove(Xs,Y,E).
