------------------------------------------------------------------------------
-- Translator from (Flat)Curry into old TasteCurry syntax
--
-- Michael Hanus, April 2005
------------------------------------------------------------------------------

module Curry2Fl where

import FlatCurry
import FlexRigid
import List
import Char
import FileGoodies
import System
import Distribution

---------------------------------------------------------------------
-- Check arguments and call main function:
main = do
  args <- getArgs
  case args of
   [a] -> store (stripSuffix a)
   _ -> putStrLn $ "ERROR: Illegal arguments for currytest: " ++
                   concat (intersperse " " args) ++ "\n" ++
                   "Usage: fcy2fl <module_name>"

printfl mod = do
  prog <- readFlatCurry mod
  putStrLn $ showFlMod prog

store mod =
  getLoadPathForFile mod >>= \loadpath ->
  readFlatCurry mod >>= \prog ->
  lookupFileInPath (baseName mod) [".fcy"] loadpath >>=
  maybe (error ("File "++mod++".fcy not found in load path"))
        (\f -> let flfile = dirName f++"/"++baseName mod++".fl" in
               writeFile flfile (showFlMod prog) >>
               putStrLn ("File "++flfile++" written."))

---------------------------------------------------------------------------
-- generate a human-readable representation of a Curry module:

showFlMod :: Prog -> String
showFlMod (Prog _ _ types funcs ops) =
  concatMap showOpDecl ops ++
  (if null ops then "" else "\n") ++
  concatMap showFlDataDecl (filter declaredTypes types)
  ++ "\n" ++
  concatMap showFlFuncDecl (filter declaredFuncs funcs)
  ++ "\n"

declaredTypes (TypeSyn _ _ _ _) = False
declaredTypes (Type (m,tc) _ _ _) =
 not (m=="Prelude" &&
      tc `elem` ["Int","Float","Char","String","[]","IO","Success","(,)"])

declaredFuncs (Func (m,f) _ _ _ _) =
  not (m=="Prelude" && f `elem` ["if_then_else"])

-- show operator declaration
showOpDecl (Op op InfixOp  prec) = "%infix("++show prec++",'"++snd op++"').\n"
showOpDecl (Op op InfixlOp prec) = "%infixl("++show prec++",'"++snd op++"').\n"
showOpDecl (Op op InfixrOp prec) = "%infixr("++show prec++",'"++snd op++"').\n"

-- show identifier in Prolog conform format:
showId (_,id) =
  if isLower (head id) && all (\c->isAlphaNum c || c=='_') (tail id)
  then id
  else "'" ++ concatMap encodeApo id ++ "'"
 where
  encodeApo c = if c=='\'' then "\\\'" else [c]

showArgs args | null args = ""
              | otherwise = "(" ++ concat (intersperse "," args) ++ ")"

showFlDataDecl (Type tcons _ tvars constrs) =
  "data " ++ showTC tcons ++ showArgs (map (\i->[chr (97+i)]) tvars) ++
  " = " ++
  concat (intersperse " ; " (map showFlConsDecl constrs))
  ++ ".\n"
showFlDataDecl (TypeSyn tcons _ tvars texp) =
  "%type " ++ snd tcons ++ concatMap (\i->[' ',chr (97+i)]) tvars ++
  " = " ++ showFlType texp ++ "\n"

showFlConsDecl (Cons cname _ _ argtypes) =
  showCF cname ++ showArgs (map (\t->showFlType t) argtypes)


--showFlType (TVar i) = [chr (65+i)]
showFlType (TVar i) = 'T':show i
showFlType (FuncType t1 t2) =
  "(" ++showFlType t1 ++ " -> " ++ showFlType t2 ++ ")"
showFlType (TCons tc ts)
 | ts==[]  = showTC tc
 | otherwise
  = showTC tc ++ showArgs (map (\t->showFlType t) ts)


showTC (m,tc) = showId (m, transTypeId tc)

transTypeId n | n=="Int"  = "int"
              | n=="Char" = "int"
              | n=="Bool" = "bool"
              | n=="IO"   = "io"
              | n=="[]"   = "list"
              | n=="Success"   = "constraint"
              | n=="Port" = "port"
              | n=="()"   = "unit"
              | n=="(,)"  = "pair"
              | take 2 n == "(," = "tuple" ++ show (length n - 1)
              | otherwise = n

showCF (m,cf) | m=="Prelude" && cf=="&"   = "/\\"
              | m=="Prelude" && cf=="=:=" = "="
              | m=="Prelude" && cf=="[]"  = "[]"
              | m=="Prelude" && cf==":"   = "."
              | otherwise                 = showId (m, transId cf)

transId n | n=="True"  = "true"
          | n=="False" = "false"
          | n=="if_then_else"   = "$ite"
          | n=="success"   = "{}"
          | n=="apply" = "@"
          | n==":"     = "."
          | n=="."     = "$comp"
          | n=="()"    = "unit"
          | take 2 n == "(," = "tuple" ++ show (length n - 1)
          | otherwise  = n


-- generate function definitions:
showFlFuncDecl (Func fname _ _ ftype (External _)) =
  "% external " ++ showCF fname ++" :: "++ showFlType ftype ++ ".\n"
showFlFuncDecl (Func fname _ _ ftype (Rule lhs rhs)) =
  showCF fname ++" :: "++ showFlType ftype ++ ".\n" ++
  showFlRule fname lhs rhs

-- format rule as set of pattern matching rules:
showFlRule fname lhs rhs
 | isCommChoice rhs
  = let Comb _ _ [rhs1] = rhs
        patternrules = rule2equations (shallowPattern2Expr fname lhs) rhs1
     in showCF fname ++ " eval choice.\n" ++
        concatMap (\(l,r)->showFlPatternRule l r) patternrules ++ "\n"
 | otherwise
  = let patternrules = rule2equations (shallowPattern2Expr fname lhs) rhs
     in showEvalAnnot (getFlexRigid rhs) ++
        concatMap (\(l,r)->showFlPatternRule l r) patternrules ++ "\n"
 where
   showEvalAnnot ConflictFR = "%"++showCF fname++" eval rigidflex.%CONFLICT!!\n"
   showEvalAnnot UnknownFR = ""
   showEvalAnnot KnownRigid = showCF fname ++ " eval rigid.\n"
   showEvalAnnot KnownFlex  = ""

-- Is the expression a representation of committed choice?
isCommChoice :: Expr -> Bool
isCommChoice e = case e of
  Comb _ f _ -> f == ("Prelude","commit")
  _ -> False

splitFreeVars exp = case exp of
  Free vars e -> (vars,e)
  _ -> ([],exp)

showFlPatternRule l r = let (vars,e) = splitFreeVars r in
   showFlExpr l ++ showFlCRHS vars e ++ ".\n"

showFlCRHS [] r =
  if isGuardedExpr r
  then " if " ++ showFlCondRule r
  else " = " ++ showFlExpr r
showFlCRHS (v:vs) r =
  if isGuardedExpr r
  then " if [" ++ concat (intersperse "," (map showFlVar (v:vs))) ++
           "] localIn " ++ showFlCondRule r
  else " = {local [" ++ concat (intersperse "," (map showFlVar (v:vs))) ++
           "] in " ++ showFlExpr r ++ "}"

showFlCondRule (Comb _ _ [e1,e2]) =
     showFlExpr e1 ++ " = " ++ showFlExpr e2


-- transform a rule consisting of a left- and a right-hand side
-- (represented as expressions) into a set of pattern matching rules:
rule2equations :: Expr -> Expr -> [(Expr,Expr)]
rule2equations lhs (Or e1 e2) =
   rule2equations lhs e1 ++ rule2equations lhs e2
rule2equations lhs (Case ctype e bs) =
   if isVarExpr e then let Var i = e  in  caseIntoLhs lhs i bs
                  else [(lhs,Case ctype e bs)]
rule2equations lhs (Var i) = [(lhs,Var i)]
rule2equations lhs (Lit l) = [(lhs,Lit l)]
rule2equations lhs (Comb ct name args) = [(lhs,Comb ct name args)]
rule2equations lhs (Free vs e) = [(lhs,Free vs e)]
rule2equations lhs (Let bs e) = [(lhs,Let bs e)]

caseIntoLhs _ _ [] = []
caseIntoLhs lhs vi (Branch (Pattern c vs) e : bs) =
  rule2equations (substitute [vi] [shallowPattern2Expr c vs] lhs) e
  ++ caseIntoLhs lhs vi bs
caseIntoLhs lhs vi (Branch (LPattern lit) e : bs) =
  rule2equations (substitute [vi] [Lit lit] lhs) e
  ++ caseIntoLhs lhs vi bs

shallowPattern2Expr name vars =
               Comb ConsCall name (map (\i->Var i) vars)


-- (substitute vars exps expr) = expr[vars/exps]
-- i.e., replace all occurrences of vars by corresponding exps in the
-- expression expr
substitute vars exps expr = substituteAll vars exps 0 expr

-- (substituteAll vars exps base expr):
-- substitute all occurrences of variables by corresonding expressions:
-- * substitute all occurrences of var_i by exp_i in expr
--   (if vars=[var_1,...,var_n] and exps=[exp_1,...,exp_n])
-- * substitute all other variables (Var j) by (Var (base+j))
--
-- here we assume that the new variables in guards and case patterns
-- do not occur in the list "vars" of replaced variables!

substituteAll :: [Int] -> [Expr] -> Int -> Expr -> Expr
substituteAll vars exps b (Var i) = replaceVar vars exps i
  where replaceVar [] [] var = Var (b+var)
        replaceVar (v:vs) (e:es) var = if v==var then e
                                                 else replaceVar vs es var
substituteAll _  _  _ (Lit l) = Lit l
substituteAll vs es b (Comb combtype c exps) =
                 Comb combtype c (map (substituteAll vs es b) exps)
substituteAll vs es b (Let bindings exp) =
                 Let (map (\(x,e)->(x+b,substituteAll vs es b e)) bindings)
                     (substituteAll vs es b exp)
substituteAll vs es b (Free vars e) =
                 Free (map (+b) vars) (substituteAll vs es b e)
substituteAll vs es b (Or e1 e2) =
                 Or (substituteAll vs es b e1) (substituteAll vs es b e2)
substituteAll vs es b (Case ctype e cases) =
   Case ctype (substituteAll vs es b e) (map (substituteAllCase vs es b) cases)

substituteAllCase vs es b (Branch (Pattern l pvs) e) =
                 Branch (Pattern l (map (+b) pvs)) (substituteAll vs es b e)
substituteAllCase vs es b (Branch (LPattern l) e) =
                 Branch (LPattern l) (substituteAll vs es b e)


-- Is the expression a guarded expressions?
isGuardedExpr :: Expr -> Bool
isGuardedExpr e = case e of
  Comb _ f _ -> f == ("Prelude","cond")
  _ -> False

-- Is the expression a variable?
isVarExpr :: Expr -> Bool
isVarExpr e = case e of
  Var _ -> True
  _ -> False

------------------------------------------------------------------------------
--- Shows a FlatCurry expressions in FL syntax.
---
--- @param expr - the FlatCurry expression to be formatted
--- @return the String representation of the formatted expression

showFlExpr :: Expr -> String

showFlExpr (Var n) = showFlVar n

showFlExpr (Lit l) = showFlLit l

showFlExpr (Comb _ cf []) = showCF cf
showFlExpr (Comb _ cf [e]) = showCF cf ++ showArgs [showFlExpr e]
showFlExpr (Comb ct (m,cf) [e1,e2])
 | isFiniteList (Comb ct (m,cf) [e1,e2])
  = "[" ++ concat (intersperse "," (showFlFiniteList (Comb ct (m,cf) [e1,e2]))) ++"]"
 | cf == "(,)" -- pair constructor?
  = "(" ++ showFlExpr e1 ++ "," ++ showFlExpr e2 ++ ")"
 | otherwise
  = showCF (m,cf) ++ showArgs [showFlExpr e1,showFlExpr e2]
showFlExpr (Comb _ (m,cf) (e1:e2:e3:es)) =
  showCF (m,cf) ++ showArgs (map showFlExpr (e1:e2:e3:es))

showFlExpr (Let _ _) =
  "error(\"LET occurred (not yet implemented in TasteCurry)\")"

showFlExpr (Free [] e) = showFlExpr e

showFlExpr (Free (x:xs) e) =
  "{local [" ++ concat (intersperse "," (map showFlVar (x:xs))) ++
     "] in " ++ showFlExpr e ++ "}"

showFlExpr (Or e1 e2) =
  "'?'" ++ showArgs [showFlExpr e1,showFlExpr e2]

showFlExpr (Case _ _ _) =
  "error(\"Complex CASE occurred (not yet implemented in TasteCurry)\")"

showFlVar i = "V" ++ show i

showFlLit (Intc   i) = show i
showFlLit (Floatc f) = show f
showFlLit (Charc  c) = show (ord c)

showFlFiniteList (Comb _ ("Prelude","[]") []) = []
showFlFiniteList (Comb _ ("Prelude",":") [e1,e2]) =
  showFlExpr e1 : showFlFiniteList e2


-- Is the expression a finite list (with an empty list at the end)?
isFiniteList :: Expr -> Bool
isFiniteList (Var _) = False
isFiniteList (Lit _) = False
isFiniteList (Comb _ name args)
  | name==("Prelude","[]") && args==[] = True
  | name==("Prelude",":") && length args == 2 = isFiniteList (args!!1)
  | otherwise = False
isFiniteList (Let _ _) = False
isFiniteList (Free _ _) = False
isFiniteList (Or _ _) = False
isFiniteList (Case _ _ _) = False


------------------------------------------------------------------------------
